<?php
/**
 * Visual Composer validator.
 *
 * Validates Visual Composer shortcode structure before injection.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/visual-composer-intelligence
 * @since      1.4.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-validator-base.php';

/**
 * Visual Composer validator class.
 *
 * @since 1.4.0
 */
class Respira_Visual_Composer_Validator extends Respira_Builder_Validator_Base {

	/**
	 * Validate a layout structure.
	 *
	 * @since  1.4.0
	 * @param  string $content Shortcode content to validate.
	 * @return array Validation result with 'valid' boolean and 'errors' array.
	 */
	public function validate_layout( $content ) {
		$this->clear_errors();

		if ( ! is_string( $content ) ) {
			$this->add_error( __( 'Content must be a string.', 'respira-for-wordpress' ) );
			return array(
				'valid'  => false,
				'errors' => $this->get_errors(),
			);
		}

		// Parse shortcodes and validate structure.
		$this->validate_shortcode_structure( $content );

		return array(
			'valid'  => empty( $this->errors ),
			'errors' => $this->get_errors(),
		);
	}

	/**
	 * Validate shortcode structure.
	 *
	 * @since  1.4.0
	 * @param  string $content Shortcode content.
	 */
	private function validate_shortcode_structure( $content ) {
		// Check for balanced shortcode tags.
		if ( ! $this->validate_balanced_shortcodes( $content ) ) {
			$this->add_error( __( 'Unbalanced shortcode tags detected.', 'respira-for-wordpress' ) );
			return;
		}

		// Extract all shortcodes.
		$pattern = get_shortcode_regex( array( 'vc_row', 'vc_column', 'vc_row_inner', 'vc_column_inner' ) );
		if ( preg_match_all( '/' . $pattern . '/s', $content, $matches ) ) {
			foreach ( $matches[2] as $index => $tag ) {
				// Validate known VC tags.
				if ( strpos( $tag, 'vc_' ) === 0 ) {
					$this->validate_vc_shortcode( $tag, $matches[3][ $index ] );
				}
			}
		}

		// Validate nesting rules.
		$this->validate_vc_nesting( $content );
	}

	/**
	 * Validate a single VC shortcode.
	 *
	 * @since  1.4.0
	 * @param  string $tag        Shortcode tag.
	 * @param  string $attributes Shortcode attributes.
	 */
	private function validate_vc_shortcode( $tag, $attributes ) {
		// Get element info from registry.
		$element = Respira_Visual_Composer_Element_Registry::get_element( $tag );
		if ( ! $element ) {
			// Unknown element - allow it (may be from extension).
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( "Respira: Unknown Visual Composer element: {$tag}" );
			}
			return;
		}

		// Parse attributes.
		$atts = shortcode_parse_atts( $attributes );
		if ( ! is_array( $atts ) ) {
			$atts = array();
		}

		// Validate specific attributes.
		foreach ( $atts as $attr_name => $attr_value ) {
			$this->validate_attribute( $tag, $attr_name, $attr_value );
		}
	}

	/**
	 * Validate an attribute value.
	 *
	 * @since  1.4.0
	 * @param  string $tag        Shortcode tag.
	 * @param  string $attr_name  Attribute name.
	 * @param  mixed  $attr_value Attribute value.
	 */
	private function validate_attribute( $tag, $attr_name, $attr_value ) {
		// Color validation.
		if ( strpos( $attr_name, 'color' ) !== false ) {
			// VC colors can be color names or hex values.
			if ( ! empty( $attr_value ) && strpos( $attr_value, '#' ) === 0 ) {
				if ( ! $this->validate_color( $attr_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: attribute name, 2: shortcode tag */
							__( 'Invalid color format for %1$s in %2$s. Use hex format (#RRGGBB) or color name.', 'respira-for-wordpress' ),
							$attr_name,
							$tag
						)
					);
				}
			}
		}

		// URL validation.
		if ( $attr_name === 'link' || $attr_name === 'url' ) {
			// VC link format: url:http://example.com|title:Title|target:_blank
			// Simple validation - check if it's a URL or VC link format.
			if ( ! empty( $attr_value ) ) {
				if ( strpos( $attr_value, 'url:' ) === 0 ) {
					// VC link format - basic validation.
					if ( ! preg_match( '/url:([^|]+)/', $attr_value ) ) {
						$this->add_error(
							sprintf(
								/* translators: 1: attribute name, 2: shortcode tag */
								__( 'Invalid VC link format for %1$s in %2$s.', 'respira-for-wordpress' ),
								$attr_name,
								$tag
							)
						);
					}
				} elseif ( ! $this->validate_url( $attr_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: attribute name, 2: shortcode tag */
							__( 'Invalid URL format for %1$s in %2$s.', 'respira-for-wordpress' ),
							$attr_name,
							$tag
						)
					);
				}
			}
		}

		// Numeric validation.
		if ( in_array( $attr_name, array( 'count', 'interval', 'value', 'gap', 'border_width' ), true ) ) {
			if ( ! empty( $attr_value ) && ! is_numeric( $attr_value ) ) {
				$this->add_error(
					sprintf(
						/* translators: 1: attribute name, 2: shortcode tag */
						__( 'Invalid numeric value for %1$s in %2$s.', 'respira-for-wordpress' ),
						$attr_name,
						$tag
					)
				);
			}
		}
	}

	/**
	 * Validate Visual Composer nesting rules.
	 *
	 * @since  1.4.0
	 * @param  string $content Shortcode content.
	 */
	private function validate_vc_nesting( $content ) {
		// Basic nesting validation: vc_row should contain vc_column.
		// This is a simplified check - proper validation would parse the full tree.

		// Check for rows without columns.
		if ( preg_match( '/\[vc_row[^\]]*\]((?!\[vc_column).)*\[\/vc_row\]/s', $content ) ) {
			// This pattern might have nested content, so we need to be careful.
			// For now, we'll allow it and just warn in debug mode.
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'Respira: vc_row without vc_column detected (may be intentional)' );
			}
		}

		// Check for columns outside rows.
		$rows_removed = preg_replace( '/\[vc_row[^\]]*\].*?\[\/vc_row\]/s', '', $content );
		if ( preg_match( '/\[vc_column/', $rows_removed ) ) {
			$this->add_error( __( 'vc_column elements must be inside vc_row.', 'respira-for-wordpress' ) );
		}
	}

	/**
	 * Validate balanced shortcodes.
	 *
	 * @since  1.4.0
	 * @param  string $content Shortcode content.
	 * @return bool True if balanced.
	 */
	private function validate_balanced_shortcodes( $content ) {
		// Stack-based validation for nested shortcodes.
		$stack   = array();
		$pattern = get_shortcode_regex();

		if ( preg_match_all( '/' . $pattern . '/s', $content, $matches, PREG_SET_ORDER ) ) {
			foreach ( $matches as $match ) {
				$tag     = $match[2];
				$closing = ! empty( $match[1] ); // Check if it's a closing tag.

				if ( $closing ) {
					// Closing tag - pop from stack.
					if ( empty( $stack ) || array_pop( $stack ) !== $tag ) {
						return false; // Unbalanced.
					}
				} else {
					// Opening tag - check if it has content (nested shortcodes).
					if ( ! empty( $match[5] ) ) {
						// Self-closing or has content.
						$stack[] = $tag;
					}
				}
			}
		}

		// Stack should be empty if all tags are balanced.
		return empty( $stack );
	}

	/**
	 * Validate shortcode syntax.
	 *
	 * @since  1.4.0
	 * @param  string $shortcode Shortcode string.
	 * @return bool True if valid syntax.
	 */
	public function validate_shortcode_syntax( $shortcode ) {
		if ( ! is_string( $shortcode ) ) {
			$this->add_error( __( 'Shortcode must be a string.', 'respira-for-wordpress' ) );
			return false;
		}

		// Basic syntax check.
		if ( ! preg_match( '/^\[vc_\w+/', $shortcode ) ) {
			$this->add_error( __( 'Invalid Visual Composer shortcode syntax.', 'respira-for-wordpress' ) );
			return false;
		}

		return true;
	}
}
