<?php
/**
 * Visual Composer element registry.
 *
 * Dynamic element detection from Visual Composer's registry.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/visual-composer-intelligence
 * @since      1.4.0
 */

/**
 * Visual Composer element registry class.
 *
 * @since 1.4.0
 */
class Respira_Visual_Composer_Element_Registry {

	/**
	 * Get all registered elements with metadata.
	 *
	 * @since  1.4.0
	 * @return array Array of element information.
	 */
	public static function get_all_elements() {
		// Check cache first.
		$cached = get_transient( 'respira_visual_composer_elements' );
		if ( false !== $cached ) {
			return $cached;
		}

		$element_list = array();

		// Try to detect elements from Visual Composer's internal registry.
		if ( function_exists( 'vc_map_get_all' ) ) {
			$element_list = self::detect_from_api();
		}

		// Fallback: use known element types from Visual Composer documentation.
		if ( empty( $element_list ) ) {
			$element_list = self::get_known_elements();
		}

		// Cache for 24 hours.
		set_transient( 'respira_visual_composer_elements', $element_list, DAY_IN_SECONDS );

		return $element_list;
	}

	/**
	 * Detect elements from Visual Composer API (if available).
	 *
	 * @since  1.4.0
	 * @return array Array of detected elements.
	 */
	private static function detect_from_api() {
		$elements = array();

		try {
			// Visual Composer provides vc_map_get_all() to get all shortcodes.
			if ( function_exists( 'vc_map_get_all' ) ) {
				$vc_elements = vc_map_get_all();

				foreach ( $vc_elements as $tag => $element_data ) {
					$elements[] = array(
						'name'        => isset( $element_data['name'] ) ? $element_data['name'] : $tag,
						'title'       => isset( $element_data['name'] ) ? $element_data['name'] : $tag,
						'type'        => $tag,
						'description' => isset( $element_data['description'] ) ? $element_data['description'] : '',
						'properties'  => isset( $element_data['params'] ) ? array_column( $element_data['params'], 'param_name' ) : array(),
					);
				}
			}
		} catch ( Exception $e ) {
			// Silently fall back to known elements.
		}

		return $elements;
	}

	/**
	 * Get known Visual Composer elements from documentation.
	 *
	 * @since  1.4.0
	 * @return array Array of known elements.
	 */
	private static function get_known_elements() {
		return array(
			// Layout elements.
			array(
				'name'        => 'Row',
				'title'       => 'Row',
				'type'        => 'vc_row',
				'description' => 'Container row for columns',
				'properties'  => array( 'full_width', 'gap', 'columns_placement', 'css' ),
			),
			array(
				'name'        => 'Column',
				'title'       => 'Column',
				'type'        => 'vc_column',
				'description' => 'Column container for elements',
				'properties'  => array( 'width', 'offset', 'css' ),
			),
			array(
				'name'        => 'Inner Row',
				'title'       => 'Inner Row',
				'type'        => 'vc_row_inner',
				'description' => 'Nested row inside a column',
				'properties'  => array( 'gap', 'css' ),
			),
			array(
				'name'        => 'Inner Column',
				'title'       => 'Inner Column',
				'type'        => 'vc_column_inner',
				'description' => 'Column inside an inner row',
				'properties'  => array( 'width', 'css' ),
			),

			// Content elements.
			array(
				'name'        => 'Text Block',
				'title'       => 'Text Block',
				'type'        => 'vc_column_text',
				'description' => 'Text content with WYSIWYG editor',
				'properties'  => array( 'content', 'css' ),
			),
			array(
				'name'        => 'Separator',
				'title'       => 'Separator',
				'type'        => 'vc_separator',
				'description' => 'Horizontal separator line',
				'properties'  => array( 'color', 'border_width', 'style', 'css' ),
			),
			array(
				'name'        => 'Button',
				'title'       => 'Button',
				'type'        => 'vc_btn',
				'description' => 'Call-to-action button',
				'properties'  => array( 'title', 'link', 'style', 'color', 'size', 'align', 'i_icon_fontawesome', 'css' ),
			),
			array(
				'name'        => 'Single Image',
				'title'       => 'Single Image',
				'type'        => 'vc_single_image',
				'description' => 'Single image with alignment and styling',
				'properties'  => array( 'image', 'img_size', 'alignment', 'onclick', 'link', 'title', 'css' ),
			),
			array(
				'name'        => 'Image Gallery',
				'title'       => 'Image Gallery',
				'type'        => 'vc_gallery',
				'description' => 'Image gallery with various layouts',
				'properties'  => array( 'images', 'type', 'interval', 'img_size', 'onclick', 'css' ),
			),
			array(
				'name'        => 'Custom Heading',
				'title'       => 'Custom Heading',
				'type'        => 'vc_custom_heading',
				'description' => 'Heading with custom styling',
				'properties'  => array( 'text', 'font_container', 'use_theme_fonts', 'css' ),
			),

			// Social elements.
			array(
				'name'        => 'Facebook',
				'title'       => 'Facebook',
				'type'        => 'vc_facebook',
				'description' => 'Facebook embed',
				'properties'  => array( 'type', 'url', 'css' ),
			),
			array(
				'name'        => 'Twitter',
				'title'       => 'Twitter',
				'type'        => 'vc_tweetmeme',
				'description' => 'Twitter button',
				'properties'  => array( 'type', 'url', 'css' ),
			),
			array(
				'name'        => 'Pinterest',
				'title'       => 'Pinterest',
				'type'        => 'vc_pinterest',
				'description' => 'Pinterest button',
				'properties'  => array( 'type', 'url', 'css' ),
			),

			// Media elements.
			array(
				'name'        => 'Video Player',
				'title'       => 'Video Player',
				'type'        => 'vc_video',
				'description' => 'Video embed (YouTube, Vimeo, etc.)',
				'properties'  => array( 'link', 'title', 'css' ),
			),
			array(
				'name'        => 'Google Maps',
				'title'       => 'Google Maps',
				'type'        => 'vc_gmaps',
				'description' => 'Google Maps embed',
				'properties'  => array( 'link', 'size', 'title', 'css' ),
			),
			array(
				'name'        => 'Raw HTML',
				'title'       => 'Raw HTML',
				'type'        => 'vc_raw_html',
				'description' => 'Raw HTML code',
				'properties'  => array( 'content' ),
			),
			array(
				'name'        => 'Raw JavaScript',
				'title'       => 'Raw JavaScript',
				'type'        => 'vc_raw_js',
				'description' => 'Raw JavaScript code',
				'properties'  => array( 'content' ),
			),

			// Widget elements.
			array(
				'name'        => 'Flickr Widget',
				'title'       => 'Flickr Widget',
				'type'        => 'vc_flickr',
				'description' => 'Flickr photo stream',
				'properties'  => array( 'flickr_id', 'count', 'type', 'display', 'css' ),
			),
			array(
				'name'        => 'Posts Slider',
				'title'       => 'Posts Slider',
				'type'        => 'vc_posts_slider',
				'description' => 'Slider with posts',
				'properties'  => array( 'title', 'type', 'count', 'interval', 'css' ),
			),
			array(
				'name'        => 'Post Grid',
				'title'       => 'Post Grid',
				'type'        => 'vc_posts_grid',
				'description' => 'Grid of posts',
				'properties'  => array( 'title', 'type', 'item', 'grid_columns_count', 'grid_layout', 'css' ),
			),
			array(
				'name'        => 'Carousel',
				'title'       => 'Carousel',
				'type'        => 'vc_carousel',
				'description' => 'Content carousel',
				'properties'  => array( 'title', 'interval', 'loop', 'css' ),
			),

			// Interactive elements.
			array(
				'name'        => 'Tabs',
				'title'       => 'Tabs',
				'type'        => 'vc_tabs',
				'description' => 'Tabbed content container',
				'properties'  => array( 'title', 'interval', 'css' ),
			),
			array(
				'name'        => 'Tab',
				'title'       => 'Tab',
				'type'        => 'vc_tab',
				'description' => 'Individual tab',
				'properties'  => array( 'title', 'content' ),
			),
			array(
				'name'        => 'Tour',
				'title'       => 'Tour',
				'type'        => 'vc_tour',
				'description' => 'Vertical tabs (tour)',
				'properties'  => array( 'title', 'interval', 'css' ),
			),
			array(
				'name'        => 'Accordion',
				'title'       => 'Accordion',
				'type'        => 'vc_accordion',
				'description' => 'Accordion container',
				'properties'  => array( 'title', 'interval', 'css' ),
			),
			array(
				'name'        => 'Accordion Section',
				'title'       => 'Accordion Section',
				'type'        => 'vc_accordion_tab',
				'description' => 'Individual accordion section',
				'properties'  => array( 'title', 'content' ),
			),

			// Structure elements.
			array(
				'name'        => 'Message Box',
				'title'       => 'Message Box',
				'type'        => 'vc_message',
				'description' => 'Styled message box',
				'properties'  => array( 'message_box_style', 'message_box_color', 'content', 'css' ),
			),
			array(
				'name'        => 'Toggle',
				'title'       => 'Toggle',
				'type'        => 'vc_toggle',
				'description' => 'Toggle/collapsible content',
				'properties'  => array( 'title', 'content', 'style', 'color', 'size', 'css' ),
			),
			array(
				'name'        => 'Progress Bar',
				'title'       => 'Progress Bar',
				'type'        => 'vc_progress_bar',
				'description' => 'Animated progress bar',
				'properties'  => array( 'values', 'units', 'bgcolor', 'custombgcolor', 'css' ),
			),
			array(
				'name'        => 'Pie Chart',
				'title'       => 'Pie Chart',
				'type'        => 'vc_pie',
				'description' => 'Animated pie chart',
				'properties'  => array( 'value', 'label', 'units', 'color', 'css' ),
			),
			array(
				'name'        => 'Icon',
				'title'       => 'Icon',
				'type'        => 'vc_icon',
				'description' => 'Icon from icon libraries',
				'properties'  => array( 'type', 'icon_fontawesome', 'color', 'size', 'align', 'link', 'css' ),
			),
			array(
				'name'        => 'Empty Space',
				'title'       => 'Empty Space',
				'type'        => 'vc_empty_space',
				'description' => 'Vertical spacing',
				'properties'  => array( 'height', 'css' ),
			),
		);
	}

	/**
	 * Get element by name or type.
	 *
	 * @since  1.4.0
	 * @param  string $element_name Element name or type.
	 * @return array|null Element information or null if not found.
	 */
	public static function get_element( $element_name ) {
		$elements = self::get_all_elements();

		foreach ( $elements as $element ) {
			if ( $element['name'] === $element_name || $element['type'] === $element_name ) {
				return $element;
			}
		}

		return null;
	}

	/**
	 * Clear element cache.
	 *
	 * @since 1.4.0
	 */
	public static function clear_cache() {
		delete_transient( 'respira_visual_composer_elements' );
	}
}
