<?php
/**
 * Thrive Architect validator.
 *
 * Validates Thrive Architect element structure before injection.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/thrive-intelligence
 * @since      1.4.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-validator-base.php';

/**
 * Thrive Architect validator class.
 *
 * @since 1.4.0
 */
class Respira_Thrive_Validator extends Respira_Builder_Validator_Base {

	/**
	 * Validate a layout structure.
	 *
	 * @since  1.4.0
	 * @param  mixed $content Content structure to validate (HTML string or array).
	 * @return array Validation result with 'valid' boolean and 'errors' array.
	 */
	public function validate_layout( $content ) {
		$this->clear_errors();

		// Thrive can be HTML string or array structure.
		if ( is_string( $content ) ) {
			return $this->validate_html_structure( $content );
		} elseif ( is_array( $content ) ) {
			return $this->validate_array_structure( $content );
		}

		$this->add_error( __( 'Content must be a string or array.', 'respira-for-wordpress' ) );
		return array(
			'valid'  => false,
			'errors' => $this->get_errors(),
		);
	}

	/**
	 * Validate HTML structure.
	 *
	 * @since  1.4.0
	 * @param  string $html HTML content to validate.
	 * @return array Validation result.
	 */
	private function validate_html_structure( $html ) {
		// Check for valid HTML.
		if ( empty( $html ) ) {
			$this->add_error( __( 'HTML content cannot be empty.', 'respira-for-wordpress' ) );
		}

		// Check for Thrive-specific data attributes.
		if ( ! preg_match( '/data-tcb-elem-type/', $html ) ) {
			// This might be plain HTML, which is acceptable but should be noted.
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'Respira: HTML content does not contain Thrive-specific attributes' );
			}
		}

		// Basic HTML validation (check for balanced tags).
		libxml_use_internal_errors( true );
		$dom = new DOMDocument();
		$dom->loadHTML( $html, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD );
		$errors = libxml_get_errors();
		libxml_clear_errors();

		if ( ! empty( $errors ) ) {
			foreach ( $errors as $error ) {
				// Only report serious errors, ignore warnings.
				if ( $error->level === LIBXML_ERR_ERROR || $error->level === LIBXML_ERR_FATAL ) {
					$this->add_error(
						sprintf(
							/* translators: 1: error message, 2: line number */
							__( 'HTML validation error: %1$s (Line %2$d)', 'respira-for-wordpress' ),
							trim( $error->message ),
							$error->line
						)
					);
				}
			}
		}

		return array(
			'valid'  => empty( $this->errors ),
			'errors' => $this->get_errors(),
		);
	}

	/**
	 * Validate array structure.
	 *
	 * @since  1.4.0
	 * @param  array $content Content structure to validate.
	 * @return array Validation result.
	 */
	private function validate_array_structure( $content ) {
		// Define nesting rules: Section > Column > Elements.
		$nesting_rules = array(
			'section'     => array(
				'column',
				'contentbox',
				'text',
				'paragraph',
				'heading',
				'button',
				'image',
				'video',
			),
			'column'      => array(
				'text',
				'paragraph',
				'heading',
				'button',
				'lead_generation',
				'optin',
				'image',
				'video',
				'audio',
				'social_share',
				'social_follow',
				'contentbox',
				'tabs',
				'toggle',
				'countdown',
				'testimonial',
				'icon',
				'pricing_table',
				'progressbar',
				'fill_counter',
				'table',
				'menu',
				'custom_html',
				'wordpress',
				'divider',
				'star_rating',
			),
			'contentbox'  => array(
				'text',
				'paragraph',
				'heading',
				'button',
				'image',
				'video',
				'icon',
				'divider',
			),
			'tabs'        => array(
				'text',
				'paragraph',
				'heading',
				'button',
				'image',
				'video',
			),
			'toggle'      => array(
				'text',
				'paragraph',
				'heading',
				'button',
				'image',
			),
		);

		// Validate structure nesting.
		$this->validate_nesting( $content, $nesting_rules );

		// Validate each element.
		foreach ( $content as $element ) {
			$this->validate_element( $element );
		}

		return array(
			'valid'  => empty( $this->errors ),
			'errors' => $this->get_errors(),
		);
	}

	/**
	 * Validate a single element.
	 *
	 * @since  1.4.0
	 * @param  array $element Element structure.
	 */
	private function validate_element( $element ) {
		// Check required fields.
		if ( ! isset( $element['type'] ) ) {
			$this->add_error( __( 'Element must have a type.', 'respira-for-wordpress' ) );
			return;
		}

		$element_type = $element['type'];

		// Validate element type is known.
		$known_element = Respira_Thrive_Element_Registry::get_element( $element_type );
		if ( ! $known_element ) {
			// Unknown element - allow it (may be from a plugin or newer version).
			// But add a warning in debug mode.
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( "Respira: Unknown Thrive element type: {$element_type}" );
			}
		}

		// Validate properties if present.
		if ( isset( $element['properties'] ) ) {
			if ( ! is_array( $element['properties'] ) ) {
				$this->add_error(
					sprintf(
						/* translators: %s: element type */
						__( 'Element properties must be an array for type %s.', 'respira-for-wordpress' ),
						$element_type
					)
				);
			} else {
				$this->validate_element_properties( $element_type, $element['properties'] );
			}
		}

		// Validate attributes (alternative to properties).
		if ( isset( $element['attributes'] ) ) {
			if ( ! is_array( $element['attributes'] ) ) {
				$this->add_error(
					sprintf(
						/* translators: %s: element type */
						__( 'Element attributes must be an array for type %s.', 'respira-for-wordpress' ),
						$element_type
					)
				);
			}
		}

		// Validate nested elements recursively.
		if ( ! empty( $element['children'] ) && is_array( $element['children'] ) ) {
			foreach ( $element['children'] as $nested_element ) {
				$this->validate_element( $nested_element );
			}
		}

		// Validate items array (used by container elements).
		if ( ! empty( $element['items'] ) && is_array( $element['items'] ) ) {
			foreach ( $element['items'] as $item ) {
				$this->validate_element( $item );
			}
		}

		// Validate columns array (used by sections).
		if ( ! empty( $element['columns'] ) && is_array( $element['columns'] ) ) {
			foreach ( $element['columns'] as $column ) {
				$this->validate_element( $column );
			}
		}

		// Validate tabs array (used by tabs element).
		if ( ! empty( $element['tabs'] ) && is_array( $element['tabs'] ) ) {
			foreach ( $element['tabs'] as $tab ) {
				if ( ! isset( $tab['title'] ) || ! isset( $tab['content'] ) ) {
					$this->add_error( __( 'Each tab must have title and content properties.', 'respira-for-wordpress' ) );
				}
			}
		}
	}

	/**
	 * Validate element properties.
	 *
	 * @since  1.4.0
	 * @param  string $element_type Element type.
	 * @param  array  $props        Element properties.
	 */
	private function validate_element_properties( $element_type, $props ) {
		$element = Respira_Thrive_Element_Registry::get_element( $element_type );
		if ( ! $element ) {
			return; // Unknown element, skip validation.
		}

		// Get expected properties from registry.
		$expected_properties = isset( $element['properties'] ) ? $element['properties'] : array();

		// Validate each property.
		foreach ( $props as $prop_name => $prop_value ) {
			// If property is expected, validate its format.
			if ( in_array( $prop_name, $expected_properties, true ) ) {
				$this->validate_property_format( $prop_name, $prop_value, $element_type );
			}
			// Allow unknown properties (may be from newer versions or custom extensions).
		}
	}

	/**
	 * Validate property format.
	 *
	 * @since  1.4.0
	 * @param  string $prop_name   Property name.
	 * @param  mixed  $prop_value  Property value.
	 * @param  string $element_type Element type.
	 */
	private function validate_property_format( $prop_name, $prop_value, $element_type ) {
		// Validate based on property name patterns.
		if ( strpos( $prop_name, 'color' ) !== false || strpos( $prop_name, 'Color' ) !== false ) {
			if ( is_string( $prop_value ) && ! $this->validate_color( $prop_value ) ) {
				$this->add_error(
					sprintf(
						/* translators: 1: property name, 2: element type */
						__( 'Invalid color format for %1$s in element %2$s. Use hex format (#RRGGBB).', 'respira-for-wordpress' ),
						$prop_name,
						$element_type
					)
				);
			}
		} elseif ( strpos( $prop_name, 'url' ) !== false || strpos( $prop_name, 'Url' ) !== false ||
				   strpos( $prop_name, 'src' ) !== false || $prop_name === 'link' ) {
			if ( ! empty( $prop_value ) && is_string( $prop_value ) && ! $this->validate_url( $prop_value ) ) {
				$this->add_error(
					sprintf(
						/* translators: 1: property name, 2: element type */
						__( 'Invalid URL format for %1$s in element %2$s.', 'respira-for-wordpress' ),
						$prop_name,
						$element_type
					)
				);
			}
		} elseif ( ( strpos( $prop_name, 'width' ) !== false || strpos( $prop_name, 'Width' ) !== false ||
				     strpos( $prop_name, 'height' ) !== false || strpos( $prop_name, 'Height' ) !== false ) &&
				   ! preg_match( '/^(content|background)/', $prop_name ) ) {
			// Width/height can be numeric or string with units.
			if ( ! is_numeric( $prop_value ) && ! preg_match( '/^\d+(px|%|em|rem|vh|vw)?$/', $prop_value ) ) {
				$this->add_error(
					sprintf(
						/* translators: 1: property name, 2: element type */
						__( 'Invalid dimension value for %1$s in element %2$s. Use number or string with units (e.g., "100px").', 'respira-for-wordpress' ),
						$prop_name,
						$element_type
					)
				);
			}
		} elseif ( ( $prop_name === 'percentage' || $prop_name === 'rating' ) && ! is_numeric( $prop_value ) ) {
			$this->add_error(
				sprintf(
					/* translators: 1: property name, 2: element type */
					__( 'Invalid numeric value for %1$s in element %2$s.', 'respira-for-wordpress' ),
					$prop_name,
					$element_type
				)
			);
		}

		// Validate specific properties.
		if ( ( $prop_name === 'html' || $prop_name === 'content' ) && ! is_string( $prop_value ) ) {
			$this->add_error(
				sprintf(
					/* translators: 1: property name, 2: element type */
					__( 'Property %1$s must be a string in element %2$s.', 'respira-for-wordpress' ),
					$prop_name,
					$element_type
				)
			);
		}

		if ( ( $prop_name === 'fields' || $prop_name === 'tabs' || $prop_name === 'columns' ||
			   $prop_name === 'features' || $prop_name === 'prices' || $prop_name === 'rows' ||
			   $prop_name === 'networks' ) && ! is_array( $prop_value ) ) {
			$this->add_error(
				sprintf(
					/* translators: 1: property name, 2: element type */
					__( 'Property %1$s must be an array in element %2$s.', 'respira-for-wordpress' ),
					$prop_name,
					$element_type
				)
			);
		}

		if ( ( $prop_name === 'autoplay' || $prop_name === 'controls' ) && ! is_bool( $prop_value ) ) {
			$this->add_error(
				sprintf(
					/* translators: 1: property name, 2: element type */
					__( 'Property %1$s must be a boolean in element %2$s.', 'respira-for-wordpress' ),
					$prop_name,
					$element_type
				)
			);
		}
	}

	/**
	 * Validate JSON structure.
	 *
	 * @since  1.4.0
	 * @param  mixed $data Data to validate.
	 * @return bool  True if valid JSON structure.
	 */
	public function validate_json_structure( $data ) {
		if ( is_string( $data ) ) {
			// Try to decode JSON string.
			$decoded = json_decode( $data, true );
			if ( json_last_error() !== JSON_ERROR_NONE ) {
				$this->add_error(
					sprintf(
						/* translators: %s: JSON error message */
						__( 'Invalid JSON: %s', 'respira-for-wordpress' ),
						json_last_error_msg()
					)
				);
				return false;
			}
			return true;
		} elseif ( is_array( $data ) ) {
			// Already an array, that's fine.
			return true;
		}

		$this->add_error( __( 'Data must be a JSON string or array.', 'respira-for-wordpress' ) );
		return false;
	}
}
