<?php
/**
 * Base schema class for page builders.
 *
 * Provides common schema generation methods for AI context.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/intelligence
 * @since      1.3.0
 */

/**
 * Base schema class.
 *
 * @since 1.3.0
 */
abstract class Respira_Builder_Schema_Base {

	/**
	 * Builder name.
	 *
	 * @since 1.3.0
	 * @var string
	 */
	protected $builder_name;

	/**
	 * Builder version.
	 *
	 * @since 1.3.0
	 * @var string
	 */
	protected $builder_version;

	/**
	 * Constructor.
	 *
	 * @since 1.3.0
	 * @param string $builder_name    Builder name.
	 * @param string $builder_version Builder version.
	 */
	public function __construct( $builder_name, $builder_version ) {
		$this->builder_name    = $builder_name;
		$this->builder_version = $builder_version;
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since  1.3.0
	 * @param  array $modules_used Optional. Array of module names used on the page.
	 * @return array Builder schema with module information.
	 */
	abstract public function get_builder_schema( $modules_used = array() );

	/**
	 * Get quick reference guide.
	 *
	 * @since  1.3.0
	 * @return array Quick reference for common formats.
	 */
	protected function get_quick_reference() {
		return array(
			'colors'  => __( 'Use hex format: #0000FF', 'respira-for-wordpress' ),
			'spacing' => __( 'Format: 10px|20px|10px|20px (top|right|bottom|left)', 'respira-for-wordpress' ),
			'sizing'  => __( 'Use px, em, %, or vw', 'respira-for-wordpress' ),
			'fonts'   => __( 'Font names from Google Fonts or system fonts', 'respira-for-wordpress' ),
		);
	}

	/**
	 * Normalize attribute type.
	 *
	 * @since  1.3.0
	 * @param  string $type Raw type from builder.
	 * @return string Normalized type.
	 */
	protected function normalize_type( $type ) {
		$type = strtolower( $type );

		// Map common variations.
		$type_map = array(
			'text'     => 'string',
			'textarea' => 'string',
			'int'      => 'number',
			'integer'  => 'number',
			'float'    => 'number',
			'bool'     => 'boolean',
			'checkbox' => 'boolean',
			'select'   => 'string',
			'dropdown' => 'string',
		);

		return isset( $type_map[ $type ] ) ? $type_map[ $type ] : $type;
	}

	/**
	 * Get format hint for attribute.
	 *
	 * @since  1.3.0
	 * @param  string $type  Attribute type.
	 * @param  string $name  Attribute name.
	 * @return string Format hint.
	 */
	protected function get_format_hint( $type, $name = '' ) {
		$type = strtolower( $type );
		$name = strtolower( $name );

		// Color-related.
		if ( 'color' === $type || false !== strpos( $name, 'color' ) ) {
			return 'hex';
		}

		// URL-related.
		if ( 'url' === $type || false !== strpos( $name, 'url' ) || false !== strpos( $name, 'link' ) ) {
			return 'url';
		}

		// Email.
		if ( 'email' === $type || false !== strpos( $name, 'email' ) ) {
			return 'email';
		}

		// Number.
		if ( 'number' === $type ) {
			return 'number';
		}

		// Boolean.
		if ( 'boolean' === $type ) {
			return 'boolean';
		}

		return 'text';
	}

	/**
	 * Get example value for attribute.
	 *
	 * @since  1.3.0
	 * @param  string $type Attribute type.
	 * @param  string $name Attribute name.
	 * @return string Example value.
	 */
	protected function get_example_value( $type, $name = '' ) {
		$type = strtolower( $type );
		$name = strtolower( $name );

		// Color.
		if ( 'color' === $type || false !== strpos( $name, 'color' ) ) {
			return '#0000FF';
		}

		// URL.
		if ( 'url' === $type || false !== strpos( $name, 'url' ) || false !== strpos( $name, 'link' ) ) {
			return 'https://example.com';
		}

		// Email.
		if ( 'email' === $type || false !== strpos( $name, 'email' ) ) {
			return 'user@example.com';
		}

		// Number.
		if ( 'number' === $type ) {
			return '10';
		}

		// Boolean.
		if ( 'boolean' === $type ) {
			return 'true';
		}

		// Text.
		return __( 'Example text', 'respira-for-wordpress' );
	}

	/**
	 * Cache schema data.
	 *
	 * @since  1.3.0
	 * @param  string $key  Cache key.
	 * @param  mixed  $data Data to cache.
	 * @param  int    $expiration Expiration in seconds. Default 24 hours.
	 * @return bool   True on success.
	 */
	protected function cache_schema( $key, $data, $expiration = DAY_IN_SECONDS ) {
		return set_transient( 'respira_schema_' . $this->builder_name . '_' . $key, $data, $expiration );
	}

	/**
	 * Get cached schema data.
	 *
	 * @since  1.3.0
	 * @param  string $key Cache key.
	 * @return mixed  Cached data or false if not found.
	 */
	protected function get_cached_schema( $key ) {
		return get_transient( 'respira_schema_' . $this->builder_name . '_' . $key );
	}

	/**
	 * Clear schema cache.
	 *
	 * @since 1.3.0
	 * @param string $key Optional. Specific cache key to clear.
	 */
	public function clear_cache( $key = '' ) {
		if ( ! empty( $key ) ) {
			delete_transient( 'respira_schema_' . $this->builder_name . '_' . $key );
		} else {
			// Clear all schema caches for this builder.
			global $wpdb;
			$wpdb->query(
				$wpdb->prepare(
					"DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
					$wpdb->esc_like( '_transient_respira_schema_' . $this->builder_name . '_' ) . '%'
				)
			);
		}
	}
}

