<?php
/**
 * Gutenberg (Block Editor) page builder implementation.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders
 */

/**
 * Gutenberg builder class.
 *
 * @since 1.0.0
 */
class Respira_Builder_Gutenberg extends Respira_Builder_Interface {

	/**
	 * Detect if Gutenberg is active.
	 *
	 * @since 1.0.0
	 * @return bool True if Gutenberg is active.
	 */
	public function detect() {
		// Gutenberg is always available in WordPress 5.0+.
		return function_exists( 'has_blocks' );
	}

	/**
	 * Get builder name.
	 *
	 * @since 1.0.0
	 * @return string Builder name.
	 */
	public function get_name() {
		return 'Gutenberg';
	}

	/**
	 * Get builder version.
	 *
	 * @since 1.0.0
	 * @return string Builder version.
	 */
	public function get_version() {
		return get_bloginfo( 'version' );
	}

	/**
	 * Check if post uses Gutenberg blocks.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if uses blocks.
	 */
	public function is_builder_used( $post_id ) {
		return has_blocks( $post_id );
	}

	/**
	 * Extract content from post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Extracted blocks.
	 */
	public function extract_content( $post_id ) {
		$start = microtime( true );

		$post = get_post( $post_id );

		if ( ! $post ) {
			return array();
		}

		// Parse blocks.
		$blocks = parse_blocks( $post->post_content );

		// Simplify block structure for AI.
		$result = $this->simplify_blocks( $blocks );

		$this->log_performance( 'extract', $post_id, microtime( true ) - $start );

		return $result;
	}

	/**
	 * Inject content into post.
	 *
	 * @since 1.0.0
	 * @param int   $post_id Post ID.
	 * @param array $content Simplified block content.
	 * @return bool|WP_Error True on success.
	 */
	public function inject_content( $post_id, $content ) {
		$start = microtime( true );

		// Validate layout before injection.
		if ( class_exists( 'Respira_Gutenberg_Validator' ) ) {
			$validator = new Respira_Gutenberg_Validator();
			$validation = $validator->validate_layout( $content );
			if ( ! $validation['valid'] ) {
				return new WP_Error(
					'respira_gutenberg_validation_failed',
					__( 'Gutenberg layout validation failed:', 'respira-for-wordpress' ) . ' ' . implode( ' ', $validation['errors'] ),
					$validation['errors']
				);
			}
		}

		// Convert simplified structure back to block format.
		$blocks = $this->complexify_blocks( $content );

		// Serialize blocks to HTML.
		$block_content = serialize_blocks( $blocks );

		// Update post content.
		$result = wp_update_post(
			array(
				'ID'           => $post_id,
				'post_content' => $block_content,
			),
			true
		);

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		$this->log_performance( 'inject', $post_id, microtime( true ) - $start );

		return true;
	}

	/**
	 * Create a code block.
	 *
	 * @since 1.0.0
	 * @param int    $post_id Post ID.
	 * @param string $html    HTML content.
	 * @param string $css     CSS content.
	 * @param string $js      JavaScript content.
	 * @return bool|WP_Error True on success.
	 */
	public function create_code_block( $post_id, $html, $css = '', $js = '' ) {
		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'respira_post_not_found',
				__( 'Post not found.', 'respira-for-wordpress' )
			);
		}

		// Combine HTML, CSS, and JS.
		$combined_code = $html;

		if ( ! empty( $css ) ) {
			$combined_code .= "\n\n<style>\n" . $css . "\n</style>";
		}

		if ( ! empty( $js ) ) {
			$combined_code .= "\n\n<script>\n" . $js . "\n</script>";
		}

		// Create HTML block.
		$html_block = array(
			'blockName'    => 'core/html',
			'attrs'        => array(),
			'innerContent' => array( $combined_code ),
			'innerHTML'    => $combined_code,
		);

		// Get existing blocks.
		$existing_blocks = parse_blocks( $post->post_content );

		// Add new block.
		$existing_blocks[] = $html_block;

		// Serialize and update.
		$block_content = serialize_blocks( $existing_blocks );

		$result = wp_update_post(
			array(
				'ID'           => $post_id,
				'post_content' => $block_content,
			),
			true
		);

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return true;
	}

	/**
	 * Get documentation.
	 *
	 * @since 1.0.0
	 * @return array Documentation.
	 */
	public function get_documentation() {
		$documentation = array(
			'name'        => 'Gutenberg (Block Editor)',
			'description' => 'Default WordPress block editor introduced in WordPress 5.0',
			'overview'    => 'Gutenberg uses blocks as the fundamental unit of content. Each block can contain text, media, embeds, or custom content.',
			'resources'   => array(
				'https://developer.wordpress.org/block-editor/',
				'https://wordpress.org/gutenberg/handbook/',
			),
		);

		// Add block catalog if available.
		if ( class_exists( 'Respira_Gutenberg_Block_Registry' ) ) {
			$all_blocks = Respira_Gutenberg_Block_Registry::get_all_blocks();
			$documentation['blocks'] = array(
				'total'     => count( $all_blocks ),
				'available' => array_column( $all_blocks, 'name' ),
			);
		}

		// Add patterns if available.
		if ( function_exists( 'respira_get_gutenberg_patterns' ) ) {
			$patterns = respira_get_gutenberg_patterns();
			$documentation['patterns'] = array(
				'count'    => count( $patterns ),
				'available' => array_keys( $patterns ),
			);
		}

		return $documentation;
	}

	/**
	 * Get available modules.
	 *
	 * @since 1.0.0
	 * @return array Available blocks.
	 */
	public function get_available_modules() {
		// Use dynamic block registry if available.
		if ( class_exists( 'Respira_Gutenberg_Block_Registry' ) ) {
			return Respira_Gutenberg_Block_Registry::get_all_blocks();
		}

		// Fallback to basic enumeration.
		$blocks = \WP_Block_Type_Registry::get_instance()->get_all_registered();

		$available = array();

		foreach ( $blocks as $block_name => $block_type ) {
			$available[] = array(
				'name'        => $block_name,
				'title'       => isset( $block_type->title ) ? $block_type->title : $block_name,
				'category'    => isset( $block_type->category ) ? $block_type->category : 'common',
				'description' => isset( $block_type->description ) ? $block_type->description : '',
				'supports'    => isset( $block_type->supports ) ? $block_type->supports : array(),
			);
		}

		return $available;
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since  1.3.0
	 * @param  array $blocks_used Optional. Array of block names used on the page.
	 * @return array Builder schema with block information.
	 */
	public function get_builder_schema( $blocks_used = array() ) {
		// Use Gutenberg Intelligence if available.
		if ( class_exists( 'Respira_Gutenberg_Block_Schema' ) ) {
			$schema_generator = new Respira_Gutenberg_Block_Schema();
			return $schema_generator->get_builder_schema( $blocks_used );
		}

		// Fallback to base implementation.
		return parent::get_builder_schema( $blocks_used );
	}

	/**
	 * Simplify blocks for AI.
	 *
	 * @since  1.0.0
	 * @param  array $blocks Parsed blocks.
	 * @return array Simplified structure.
	 */
	private function simplify_blocks( $blocks ) {
		$simplified = array();

		foreach ( $blocks as $block ) {
			$simple_block = array(
				'type'    => $block['blockName'],
				'attrs'   => $block['attrs'] ?? array(),
				'content' => $block['innerHTML'] ?? '',
			);

			// Handle inner blocks recursively.
			if ( ! empty( $block['innerBlocks'] ) ) {
				$simple_block['inner_blocks'] = $this->simplify_blocks( $block['innerBlocks'] );
			}

			$simplified[] = $simple_block;
		}

		return $simplified;
	}

	/**
	 * Convert simplified structure back to blocks.
	 *
	 * @since  1.0.0
	 * @param  array $simplified Simplified blocks.
	 * @return array Block structure.
	 */
	private function complexify_blocks( $simplified ) {
		$blocks = array();

		foreach ( $simplified as $simple_block ) {
			$block = array(
				'blockName'    => $simple_block['type'] ?? 'core/paragraph',
				'attrs'        => $simple_block['attrs'] ?? array(),
				'innerContent' => array( $simple_block['content'] ?? '' ),
				'innerHTML'    => $simple_block['content'] ?? '',
				'innerBlocks'  => array(),
			);

			// Handle inner blocks recursively.
			if ( ! empty( $simple_block['inner_blocks'] ) ) {
				$block['innerBlocks'] = $this->complexify_blocks( $simple_block['inner_blocks'] );
			}

			$blocks[] = $block;
		}

		return $blocks;
	}
}
