<?php
/**
 * Elementor page builder implementation.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders
 */

/**
 * Elementor builder class (PRO feature).
 *
 * @since 1.0.0
 */
class Respira_Builder_Elementor extends Respira_Builder_Interface {

	/**
	 * Detect if Elementor is active.
	 *
	 * @since 1.0.0
	 * @return bool True if Elementor is active.
	 */
	public function detect() {
		return defined( 'ELEMENTOR_VERSION' ) || class_exists( '\Elementor\Plugin' );
	}

	/**
	 * Get builder name.
	 *
	 * @since 1.0.0
	 * @return string Builder name.
	 */
	public function get_name() {
		return 'Elementor';
	}

	/**
	 * Get builder version.
	 *
	 * @since 1.0.0
	 * @return string Builder version.
	 */
	public function get_version() {
		return defined( 'ELEMENTOR_VERSION' ) ? ELEMENTOR_VERSION : 'unknown';
	}

	/**
	 * Check if post uses Elementor.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if uses Elementor.
	 */
	public function is_builder_used( $post_id ) {
		if ( class_exists( '\Elementor\Plugin' ) ) {
			$document = \Elementor\Plugin::$instance->documents->get( $post_id );
			return $document && $document->is_built_with_elementor();
		}

		return (bool) get_post_meta( $post_id, '_elementor_edit_mode', true );
	}

	/**
	 * Extract content from post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Extracted elements.
	 */
	public function extract_content( $post_id ) {
		$start = microtime( true );

		$data = get_post_meta( $post_id, '_elementor_data', true );

		if ( empty( $data ) ) {
			return array();
		}

		$elements = json_decode( $data, true );

		$result = $this->simplify_structure( $elements );

		$this->log_performance( 'extract', $post_id, microtime( true ) - $start );

		return $result;
	}

	/**
	 * Inject content into post.
	 *
	 * @since 1.0.0
	 * @param int   $post_id Post ID.
	 * @param array $content Simplified content.
	 * @return bool|WP_Error True on success.
	 */
	public function inject_content( $post_id, $content ) {
		$start = microtime( true );

		// Validate layout before injection.
		if ( class_exists( 'Respira_Elementor_Validator' ) ) {
			$validator = new Respira_Elementor_Validator();
			$validation = $validator->validate_layout( $content );
			if ( ! $validation['valid'] ) {
				return new WP_Error(
					'respira_elementor_validation_failed',
					__( 'Elementor layout validation failed:', 'respira-for-wordpress' ) . ' ' . implode( ' ', $validation['errors'] ),
					$validation['errors']
				);
			}
		}

		$elements = $this->complexify_structure( $content );

		update_post_meta( $post_id, '_elementor_data', wp_json_encode( $elements ) );
		update_post_meta( $post_id, '_elementor_edit_mode', 'builder' );
		update_post_meta( $post_id, '_elementor_version', ELEMENTOR_VERSION );

		// Clear cache.
		if ( class_exists( '\Elementor\Plugin' ) ) {
			\Elementor\Plugin::$instance->files_manager->clear_cache();
		}

		$this->log_performance( 'inject', $post_id, microtime( true ) - $start );

		return true;
	}

	/**
	 * Create a code block.
	 *
	 * @since 1.0.0
	 * @param int    $post_id Post ID.
	 * @param string $html    HTML content.
	 * @param string $css     CSS content.
	 * @param string $js      JavaScript content.
	 * @return bool|WP_Error True on success.
	 */
	public function create_code_block( $post_id, $html, $css = '', $js = '' ) {
		$data = get_post_meta( $post_id, '_elementor_data', true );
		$elements = json_decode( $data, true ) ?: array();

		// Create HTML widget element.
		$html_widget = array(
			'id'         => uniqid( 'respira_' ),
			'elType'     => 'widget',
			'widgetType' => 'html',
			'settings'   => array(
				'html' => $html . ( $css ? "\n<style>$css</style>" : '' ) . ( $js ? "\n<script>$js</script>" : '' ),
			),
		);

		$elements[] = $html_widget;

		update_post_meta( $post_id, '_elementor_data', wp_json_encode( $elements ) );

		return true;
	}

	/**
	 * Get documentation.
	 *
	 * @since 1.0.0
	 * @return array Documentation.
	 */
	public function get_documentation() {
		$documentation = array(
			'name'        => 'Elementor',
			'description' => 'Popular drag-and-drop page builder plugin',
			'overview'    => 'Elementor uses sections, columns, and widgets. Data is stored as JSON in post meta.',
			'resources'   => array(
				'https://developers.elementor.com/',
			),
		);

		// Add widget catalog if available.
		if ( class_exists( 'Respira_Elementor_Widget_Registry' ) ) {
			$all_widgets = Respira_Elementor_Widget_Registry::get_all_widgets();
			$documentation['widgets'] = array(
				'total'     => count( $all_widgets ),
				'available' => array_column( $all_widgets, 'name' ),
			);
		}

		// Add patterns if available.
		if ( function_exists( 'respira_get_elementor_patterns' ) ) {
			$patterns = respira_get_elementor_patterns();
			$documentation['patterns'] = array(
				'count'    => count( $patterns ),
				'available' => array_keys( $patterns ),
			);
		}

		return $documentation;
	}

	/**
	 * Get available modules.
	 *
	 * @since 1.0.0
	 * @return array Available widgets.
	 */
	public function get_available_modules() {
		// Use dynamic widget registry if available.
		if ( class_exists( 'Respira_Elementor_Widget_Registry' ) ) {
			return Respira_Elementor_Widget_Registry::get_all_widgets();
		}

		// Fallback to basic enumeration.
		if ( ! class_exists( '\Elementor\Plugin' ) ) {
			return array();
		}

		$widgets_manager = \Elementor\Plugin::$instance->widgets_manager;
		$widget_types = $widgets_manager->get_widget_types();

		$available = array();

		foreach ( $widget_types as $widget_type ) {
			$available[] = array(
				'name'  => $widget_type->get_name(),
				'title' => $widget_type->get_title(),
			);
		}

		return $available;
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since  1.3.0
	 * @param  array $widgets_used Optional. Array of widget names used on the page.
	 * @return array Builder schema with widget information.
	 */
	public function get_builder_schema( $widgets_used = array() ) {
		// Use Elementor Intelligence if available.
		if ( class_exists( 'Respira_Elementor_Widget_Schema' ) ) {
			$schema_generator = new Respira_Elementor_Widget_Schema();
			return $schema_generator->get_builder_schema( $widgets_used );
		}

		// Fallback to base implementation.
		return parent::get_builder_schema( $widgets_used );
	}

	/**
	 * Simplify Elementor structure.
	 *
	 * @since  1.0.0
	 * @param  array $elements Elementor elements.
	 * @return array Simplified structure.
	 */
	protected function simplify_structure( $elements ) {
		return array_map(
			function( $element ) {
				return array(
					'type'     => $element['elType'] ?? 'unknown',
					'widget'   => $element['widgetType'] ?? null,
					'settings' => $element['settings'] ?? array(),
					'elements' => isset( $element['elements'] ) ? $this->simplify_structure( $element['elements'] ) : array(),
				);
			},
			$elements
		);
	}

	/**
	 * Convert simplified structure back to Elementor format.
	 *
	 * @since  1.0.0
	 * @param  array $simplified Simplified structure.
	 * @return array Elementor format.
	 */
	protected function complexify_structure( $simplified ) {
		return array_map(
			function( $item ) {
				$element = array(
					'id'       => uniqid(),
					'elType'   => $item['type'] ?? 'widget',
					'settings' => $item['settings'] ?? array(),
				);

				if ( isset( $item['widget'] ) ) {
					$element['widgetType'] = $item['widget'];
				}

				if ( ! empty( $item['elements'] ) ) {
					$element['elements'] = $this->complexify_structure( $item['elements'] );
				}

				return $element;
			},
			$simplified
		);
	}
}
