<?php
/**
 * Divi page builder implementation.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders
 */

/**
 * Divi builder class.
 *
 * @since 1.0.0
 */
class Respira_Builder_Divi extends Respira_Builder_Interface {

	/**
	 * Detect if Divi is active.
	 *
	 * @since 1.0.0
	 * @return bool True if Divi is active.
	 */
	public function detect() {
		// Check if Divi theme or Divi Builder plugin is active.
		$theme = wp_get_theme();

		return ( 'Divi' === $theme->get( 'Name' ) ||
				'Divi' === $theme->get_template() ||
				defined( 'ET_BUILDER_PLUGIN_VERSION' ) ||
				function_exists( 'et_pb_is_pagebuilder_used' ) );
	}

	/**
	 * Get builder name.
	 *
	 * @since 1.0.0
	 * @return string Builder name.
	 */
	public function get_name() {
		return 'Divi';
	}

	/**
	 * Get builder version.
	 *
	 * @since 1.0.0
	 * @return string Builder version.
	 */
	public function get_version() {
		if ( defined( 'ET_BUILDER_PLUGIN_VERSION' ) ) {
			return ET_BUILDER_PLUGIN_VERSION;
		}

		$theme = wp_get_theme();
		if ( 'Divi' === $theme->get( 'Name' ) || 'Divi' === $theme->get_template() ) {
			return $theme->get( 'Version' );
		}

		return 'unknown';
	}

	/**
	 * Check if post uses Divi builder.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if uses Divi.
	 */
	public function is_builder_used( $post_id ) {
		// Check if Divi builder is enabled for this post.
		$enabled = get_post_meta( $post_id, '_et_pb_use_builder', true );

		if ( 'on' === $enabled ) {
			return true;
		}

		// Also check if content contains Divi shortcodes.
		$post = get_post( $post_id );
		if ( $post && preg_match( '/\[et_pb_/', $post->post_content ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Extract content from post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Extracted modules.
	 */
	public function extract_content( $post_id ) {
		$start = microtime( true );

		$post = get_post( $post_id );

		if ( ! $post ) {
			return array();
		}

		// Parse Divi shortcodes.
		$modules = $this->parse_divi_shortcodes( $post->post_content );

		// Resolve global modules if any.
		if ( class_exists( 'Respira_Divi_Global_Modules' ) ) {
			$modules = Respira_Divi_Global_Modules::resolve_global_modules( $modules );
		}

		$result = $this->simplify_structure( $modules );

		// Performance monitoring: log if extraction takes > 2 seconds.
		$time = microtime( true ) - $start;
		if ( $time > 2.0 ) {
			error_log(
				sprintf(
					/* translators: 1: post ID, 2: time in seconds */
					'Respira: Slow Divi extract - Post ID %d took %.2fs',
					$post_id,
					$time
				)
			);
		}

		return $result;
	}

	/**
	 * Inject content into post.
	 *
	 * @since 1.0.0
	 * @param int   $post_id Post ID.
	 * @param array $content Simplified module content.
	 * @return bool|WP_Error True on success.
	 */
	public function inject_content( $post_id, $content ) {
		$start = microtime( true );

		// Validate layout before injection.
		if ( class_exists( 'Respira_Divi_Validator' ) ) {
			$validation = Respira_Divi_Validator::validate_layout( $content );
			if ( ! $validation['valid'] ) {
				return new WP_Error(
					'respira_divi_validation_failed',
					__( 'Divi layout validation failed:', 'respira-for-wordpress' ) . ' ' . implode( ' ', $validation['errors'] ),
					$validation['errors']
				);
			}
		}

		// Convert simplified structure back to Divi shortcodes.
		$shortcodes = $this->build_divi_shortcodes( $content );

		// Update post content.
		$result = wp_update_post(
			array(
				'ID'           => $post_id,
				'post_content' => $shortcodes,
			),
			true
		);

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		// Ensure Divi builder is enabled.
		update_post_meta( $post_id, '_et_pb_use_builder', 'on' );
		
		// Get dynamic builder version instead of hardcoding.
		$builder_version = $this->get_version();
		if ( 'unknown' !== $builder_version ) {
			// Extract major version (e.g., "4.0" from "4.0.1").
			$version_parts = explode( '.', $builder_version );
			$major_version = $version_parts[0] . '.' . ( $version_parts[1] ?? '0' );
			update_post_meta( $post_id, '_et_pb_builder_version', $major_version );
		} else {
		update_post_meta( $post_id, '_et_pb_builder_version', '4.0' );
		}

		// Performance monitoring: log if injection takes > 2 seconds.
		$time = microtime( true ) - $start;
		if ( $time > 2.0 ) {
			error_log(
				sprintf(
					/* translators: 1: post ID, 2: time in seconds */
					'Respira: Slow Divi inject - Post ID %d took %.2fs',
					$post_id,
					$time
				)
			);
		}

		return true;
	}

	/**
	 * Create a code block.
	 *
	 * @since 1.0.0
	 * @param int    $post_id Post ID.
	 * @param string $html    HTML content.
	 * @param string $css     CSS content.
	 * @param string $js      JavaScript content.
	 * @return bool|WP_Error True on success.
	 */
	public function create_code_block( $post_id, $html, $css = '', $js = '' ) {
		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'respira_post_not_found',
				__( 'Post not found.', 'respira-for-wordpress' )
			);
		}

		// Combine code.
		$combined_code = $html;

		if ( ! empty( $css ) ) {
			$combined_code .= "\n\n<style>\n" . $css . "\n</style>";
		}

		if ( ! empty( $js ) ) {
			$combined_code .= "\n\n<script>\n" . $js . "\n</script>";
		}

		// Create Divi code module shortcode.
		$code_module = '[et_pb_code]' . base64_encode( $combined_code ) . '[/et_pb_code]';

		// Append to existing content.
		$updated_content = $post->post_content . "\n" . $code_module;

		$result = wp_update_post(
			array(
				'ID'           => $post_id,
				'post_content' => $updated_content,
			),
			true
		);

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return true;
	}

	/**
	 * Get documentation.
	 *
	 * @since 1.0.0
	 * @return array Documentation.
	 */
	public function get_documentation() {
		$documentation = array(
			'name'        => 'Divi Builder',
			'description' => 'Premium drag-and-drop page builder by Elegant Themes',
			'overview'    => 'Divi uses a hierarchical structure: Sections > Rows > Modules. Content is stored as shortcodes.',
			'structure'   => array(
				'section' => 'Top-level container (et_pb_section)',
				'row'     => 'Layout container with columns (et_pb_row)',
				'module'  => 'Content element (et_pb_text, et_pb_image, etc.)',
			),
			'specialty_sections' => array(
				'description' => 'Specialty sections allow complex column layouts within sections',
				'usage'       => 'Set specialty="on" attribute on et_pb_section to enable specialty layout',
				'structure'   => 'Specialty sections can contain rows with different column structures',
			),
			'global_modules' => array(
				'description' => 'Global modules are reusable modules stored separately',
				'usage'       => 'Use global_module="123" attribute to reference a global module',
				'benefits'    => 'Update once, changes reflect everywhere the module is used',
			),
			'resources'   => array(
				'https://www.elegantthemes.com/documentation/divi/',
			),
		);

		// Add module catalog if available.
		if ( class_exists( 'Respira_Divi_Module_Registry' ) ) {
			$all_modules = Respira_Divi_Module_Registry::get_all_modules();
			$documentation['modules'] = array(
				'total'     => count( $all_modules ),
				'available' => array_column( $all_modules, 'name' ),
			);
		}

		// Add layout patterns if available (from Divi Intelligence).
		$patterns_file = RESPIRA_PLUGIN_DIR . 'includes/divi-intelligence/divi-patterns.php';
		if ( file_exists( $patterns_file ) ) {
			require_once $patterns_file;
			if ( function_exists( 'respira_get_divi_patterns' ) ) {
				$patterns = respira_get_divi_patterns();
				$documentation['patterns'] = array(
					'count'    => count( $patterns ),
					'available' => array_keys( $patterns ),
				);
			}
		}

		return $documentation;
	}

	/**
	 * Get available modules.
	 *
	 * @since 1.0.0
	 * @param int|null $tier Deprecated. All modules are now returned regardless of tier.
	 * @return array Available Divi modules.
	 */
	public function get_available_modules( $tier = null ) {
		// Use dynamic module registry - returns ALL modules (full support).
		if ( class_exists( 'Respira_Divi_Module_Registry' ) ) {
			return Respira_Divi_Module_Registry::get_all_modules();
		}

		// Fallback to basic list if registry not available.
		return array(
			array(
				'name'        => 'et_pb_section',
				'title'       => 'Section',
				'description' => 'Top-level container',
			),
			array(
				'name'        => 'et_pb_row',
				'title'       => 'Row',
				'description' => 'Column layout container',
			),
			array(
				'name'        => 'et_pb_text',
				'title'       => 'Text',
				'description' => 'Text content module',
			),
			array(
				'name'        => 'et_pb_image',
				'title'       => 'Image',
				'description' => 'Image module',
			),
			array(
				'name'        => 'et_pb_button',
				'title'       => 'Button',
				'description' => 'Button module',
			),
		);
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since 1.0.0
	 * @param array $modules_used Optional. Array of module names used on the page.
	 * @return array Builder schema with module information.
	 */
	public function get_builder_schema( $modules_used = array() ) {
		$schema = array(
			'builder'          => 'divi',
			'builder_version'  => $this->get_version(),
			'available_modules' => array(),
			'quick_reference'  => array(
				'colors'  => 'Use hex format: #0000FF',
				'spacing' => 'Format: 10px|20px|10px|20px (top|right|bottom|left)',
				'sizing'  => 'Use px, em, %, or vw',
				'fonts'   => 'Font names from Google Fonts or system fonts',
			),
		);

		// Get schemas for modules used on the page (or all modules if none specified).
		if ( empty( $modules_used ) ) {
			// Get all available modules for full support.
			$all_modules = $this->get_available_modules();
			$modules_used  = array_column( $all_modules, 'name' );
		}

		if ( class_exists( 'Respira_Divi_Module_Schema' ) ) {
			$module_schemas = Respira_Divi_Module_Schema::get_module_schemas( $modules_used );
			foreach ( $module_schemas as $module_name => $module_schema ) {
				$schema['available_modules'][ $module_name ] = array(
					'title'       => $module_schema['title'] ?? '',
					'description' => $module_schema['description'] ?? '',
					'common_attributes' => array(),
				);

				// Include common attributes (limit to most important for performance).
				if ( isset( $module_schema['attributes'] ) ) {
					$common_attrs = array( 'text_color', 'background_color', 'content', 'button_text', 'button_url', 'src', 'url' );
					foreach ( $common_attrs as $attr_name ) {
						if ( isset( $module_schema['attributes'][ $attr_name ] ) ) {
							$schema['available_modules'][ $module_name ]['common_attributes'][ $attr_name ] = array(
								'type'    => $module_schema['attributes'][ $attr_name ]['type'] ?? 'string',
								'format'  => $module_schema['attributes'][ $attr_name ]['format'] ?? 'text',
								'example' => $module_schema['attributes'][ $attr_name ]['example'] ?? '',
							);
						}
					}
				}
			}
		}

		return $schema;
	}

	/**
	 * Parse Divi shortcodes into structured array.
	 *
	 * Uses WordPress's shortcode regex for accurate parsing while handling
	 * nested shortcodes recursively.
	 *
	 * @since  1.0.0
	 * @param  string $content Post content with Divi shortcodes.
	 * @return array  Structured array of modules.
	 */
	private function parse_divi_shortcodes( $content ) {
		$modules = array();

		if ( empty( $content ) ) {
			return $modules;
		}

		// Get all Divi modules dynamically for parsing.
		$divi_modules = array();
		if ( class_exists( 'Respira_Divi_Module_Registry' ) ) {
			$all_modules = Respira_Divi_Module_Registry::get_all_modules();
			$divi_modules = array_column( $all_modules, 'name' );
		}
		
		// If no modules found, use generic pattern that matches all et_pb_* shortcodes.
		if ( empty( $divi_modules ) ) {
			// Use generic pattern that matches any et_pb_* shortcode.
			$pattern = '/\[et_pb_(\w+)([^\]]*)\](.*?)\[\/et_pb_\1\]/s';
		} else {
			// Use WordPress's shortcode regex with all detected modules.
			$pattern = get_shortcode_regex( $divi_modules );
		}

		// If pattern is empty, fall back to generic Divi pattern.
		if ( empty( $pattern ) ) {
			$pattern = '/\[et_pb_(\w+)([^\]]*)\](.*?)\[\/et_pb_\1\]/s';
		}

		// Match all Divi shortcodes.
		preg_match_all( $pattern, $content, $matches, PREG_SET_ORDER );

		foreach ( $matches as $match ) {
			// Extract module type.
			$module_type = isset( $match[2] ) ? $match[2] : ( isset( $match[1] ) ? 'et_pb_' . $match[1] : '' );
			
			// Handle both WordPress shortcode format and our custom format.
			if ( strpos( $module_type, 'et_pb_' ) !== 0 ) {
				$module_type = 'et_pb_' . $module_type;
			}

			// Extract attributes string.
			$attr_string = isset( $match[3] ) ? $match[3] : ( isset( $match[2] ) ? $match[2] : '' );
			
			// Extract content.
			$module_content = isset( $match[5] ) ? $match[5] : ( isset( $match[3] ) ? $match[3] : '' );

			$module = array(
				'type'    => $module_type,
				'attrs'   => $this->parse_shortcode_attrs( $attr_string ),
				'content' => $module_content,
			);

			// Check for nested modules recursively.
			if ( ! empty( $module['content'] ) && preg_match( '/\[et_pb_/', $module['content'] ) ) {
				$nested_modules = $this->parse_divi_shortcodes( $module['content'] );
				
				// Special handling for rows: if children are columns, flatten structure.
				if ( 'et_pb_row' === $module_type ) {
					$column_modules = array();
					$column_widths = array();
					
					foreach ( $nested_modules as $nested ) {
						if ( 'et_pb_column' === $nested['type'] ) {
							// Extract column width from type attribute (e.g., "4_12").
							$column_type = $nested['attrs']['type'] ?? '12_12';
							if ( preg_match( '/(\d+)_12/', $column_type, $matches ) ) {
								$column_widths[] = $matches[1];
							} else {
								$column_widths[] = '12';
							}
							
							// Get modules inside this column.
							if ( ! empty( $nested['children'] ) ) {
								$column_modules = array_merge( $column_modules, $nested['children'] );
							} elseif ( ! empty( $nested['content'] ) && preg_match( '/\[et_pb_/', $nested['content'] ) ) {
								// Parse column content if it contains modules.
								$column_modules = array_merge( $column_modules, $this->parse_divi_shortcodes( $nested['content'] ) );
							}
						} else {
							// Direct child (not a column) - keep as is.
							$column_modules[] = $nested;
						}
					}
					
					// Set column_structure attribute if we found columns.
					if ( ! empty( $column_widths ) ) {
						$module['attrs']['column_structure'] = implode( '_', $column_widths );
					}
					
					$module['children'] = $column_modules;
				} else {
					$module['children'] = $nested_modules;
				}
			}

			$modules[] = $module;
		}

		// If no matches with WordPress pattern, try fallback regex.
		if ( empty( $modules ) ) {
			preg_match_all( '/\[et_pb_(\w+)([^\]]*)\](.*?)\[\/et_pb_\1\]/s', $content, $matches, PREG_SET_ORDER );

		foreach ( $matches as $match ) {
			$module = array(
				'type'    => 'et_pb_' . $match[1],
				'attrs'   => $this->parse_shortcode_attrs( $match[2] ),
				'content' => $match[3],
			);

			// Check for nested modules.
			if ( preg_match( '/\[et_pb_/', $module['content'] ) ) {
				$nested_modules = $this->parse_divi_shortcodes( $module['content'] );
				
				// Special handling for rows: if children are columns, flatten structure.
				if ( 'et_pb_row' === $module['type'] ) {
					$column_modules = array();
					$column_widths = array();
					
					foreach ( $nested_modules as $nested ) {
						if ( 'et_pb_column' === $nested['type'] ) {
							// Extract column width from type attribute (e.g., "4_12").
							$column_type = $nested['attrs']['type'] ?? '12_12';
							if ( preg_match( '/(\d+)_12/', $column_type, $matches ) ) {
								$column_widths[] = $matches[1];
							} else {
								$column_widths[] = '12';
							}
							
							// Get modules inside this column.
							if ( ! empty( $nested['children'] ) ) {
								$column_modules = array_merge( $column_modules, $nested['children'] );
							} elseif ( ! empty( $nested['content'] ) && preg_match( '/\[et_pb_/', $nested['content'] ) ) {
								// Parse column content if it contains modules.
								$column_modules = array_merge( $column_modules, $this->parse_divi_shortcodes( $nested['content'] ) );
							}
						} else {
							// Direct child (not a column) - keep as is.
							$column_modules[] = $nested;
						}
					}
					
					// Set column_structure attribute if we found columns.
					if ( ! empty( $column_widths ) ) {
						$module['attrs']['column_structure'] = implode( '_', $column_widths );
					}
					
					$module['children'] = $column_modules;
				} else {
					$module['children'] = $nested_modules;
				}
			}

			$modules[] = $module;
			}
		}

		return $modules;
	}

	/**
	 * Parse shortcode attributes.
	 *
	 * Handles attributes with special characters, JSON-encoded values,
	 * and preserves attribute order.
	 *
	 * @since  1.0.0
	 * @param  string $attr_string Attribute string.
	 * @return array  Attributes array.
	 */
	private function parse_shortcode_attrs( $attr_string ) {
		$attrs = array();

		if ( empty( $attr_string ) ) {
			return $attrs;
		}

		// Use WordPress's shortcode_atts parsing if available, but we need to handle
		// the raw attribute string first.
		// Pattern: key="value" or key='value' or key=value
		// Handle both quoted and unquoted values.
		preg_match_all( '/(\w+)(?:\s*=\s*(?:"([^"]*)"|\'([^\']*)\'|([^\s]+)))?/s', $attr_string, $matches, PREG_SET_ORDER );

		foreach ( $matches as $match ) {
			$key = $match[1];
			
			// Get value from whichever capture group matched.
			$value = '';
			if ( ! empty( $match[2] ) ) {
				$value = $match[2]; // Double-quoted.
			} elseif ( ! empty( $match[3] ) ) {
				$value = $match[3]; // Single-quoted.
			} elseif ( ! empty( $match[4] ) ) {
				$value = $match[4]; // Unquoted.
			}

			// Decode HTML entities.
			$value = html_entity_decode( $value, ENT_QUOTES | ENT_HTML5, 'UTF-8' );

			// Try to decode JSON if it looks like JSON.
			if ( ! empty( $value ) && ( $value[0] === '{' || $value[0] === '[' ) ) {
				$json_decoded = json_decode( $value, true );
				if ( json_last_error() === JSON_ERROR_NONE ) {
					$value = $json_decoded;
				}
			}

			$attrs[ $key ] = $value;
		}

		return $attrs;
	}

	/**
	 * Build Divi shortcodes from structured array.
	 *
	 * Properly escapes attributes and handles complex attribute values
	 * including arrays and special characters.
	 *
	 * @since  1.0.0
	 * @param  array $modules Structured modules array.
	 * @return string Divi shortcodes.
	 */
	private function build_divi_shortcodes( $modules ) {
		$shortcodes = '';

		foreach ( $modules as $module ) {
			$type     = $module['type'] ?? 'et_pb_text';
			$attrs    = $module['attrs'] ?? array();
			$content  = $module['content'] ?? '';
			$children = $module['children'] ?? array();

			// Special handling for rows - always create columns if children exist.
			if ( 'et_pb_row' === $type && ! empty( $children ) ) {
				// Determine column structure.
				$column_structure = '';
				$column_widths = array();

				if ( ! empty( $attrs['column_structure'] ) ) {
					// Use provided column structure (e.g., "4_4_4" = 3 columns of equal width).
					$column_structure = $attrs['column_structure'];
					$column_widths = explode( '_', $column_structure );
				} else {
					// Default: create one column for all children.
					$column_widths = array( '12' );
					$column_structure = '12';
				}

				$num_columns = count( $column_widths );

				// Remove column_structure from row attributes before building attribute string.
				$row_attrs = $attrs;
				unset( $row_attrs['column_structure'] );

				// Build row attribute string.
				$attr_string = '';
				foreach ( $row_attrs as $key => $value ) {
					if ( is_array( $value ) || is_object( $value ) ) {
						$value = wp_json_encode( $value );
					}
					$value = (string) $value;
					$value = esc_attr( $value );
					$attr_string .= ' ' . sanitize_key( $key ) . '="' . $value . '"';
				}

				// Add column_structure as a real attribute for Divi.
				if ( ! empty( $column_structure ) ) {
					$attr_string .= ' column_structure="' . esc_attr( $column_structure ) . '"';
				}

				// Build row opening tag.
				$shortcodes .= '[et_pb_row' . $attr_string . ']';

				// Distribute children across columns.
				// If we have exactly N children and N columns, put one per column.
				// Otherwise, distribute evenly (round-robin).
				$children_per_column = array();
				$num_children = count( $children );

				if ( $num_children === $num_columns ) {
					// One child per column.
					foreach ( $children as $index => $child ) {
						$children_per_column[ $index ] = array( $child );
					}
				} else {
					// Distribute evenly across columns.
					$current_column = 0;
					foreach ( $children as $child ) {
						if ( ! isset( $children_per_column[ $current_column ] ) ) {
							$children_per_column[ $current_column ] = array();
						}
						$children_per_column[ $current_column ][] = $child;
						$current_column = ( $current_column + 1 ) % $num_columns;
					}
				}

				// Create column modules and add children.
				foreach ( $children_per_column as $column_index => $column_children ) {
					$column_width = isset( $column_widths[ $column_index ] ) ? $column_widths[ $column_index ] : '12';
					
					// Create column module with width attribute (format: "4_12" means 4/12 = 1/3 width).
					$shortcodes .= '[et_pb_column type="' . esc_attr( $column_width ) . '_12"]';

					// Add children to this column.
					if ( ! empty( $column_children ) ) {
						$shortcodes .= $this->build_divi_shortcodes( $column_children );
					}

					$shortcodes .= '[/et_pb_column]';
				}

				// Close row.
				$shortcodes .= '[/et_pb_row]' . "\n";
				continue;
			}

			// Build attribute string with proper escaping.
			$attr_string = '';
			foreach ( $attrs as $key => $value ) {
				// Handle array/object values (encode as JSON).
				if ( is_array( $value ) || is_object( $value ) ) {
					$value = wp_json_encode( $value );
				}

				// Convert to string if not already.
				$value = (string) $value;

				// Escape for HTML attribute context.
				$value = esc_attr( $value );

				$attr_string .= ' ' . sanitize_key( $key ) . '="' . $value . '"';
			}

			// Build shortcode opening tag.
			$shortcodes .= '[' . sanitize_key( $type ) . $attr_string . ']';

			// Add children if present (nested modules).
			if ( ! empty( $children ) ) {
				$shortcodes .= $this->build_divi_shortcodes( $children );
			} else {
				// Add content (may contain HTML, scripts, etc.).
				// Content is already in the format Divi expects.
				$shortcodes .= $content;
			}

			// Close shortcode tag.
			$shortcodes .= '[/' . sanitize_key( $type ) . ']' . "\n";
		}

		return $shortcodes;
	}

	/**
	 * Simplify Divi structure for AI.
	 *
	 * @since  1.0.0
	 * @param  array $modules Parsed modules.
	 * @return array Simplified structure.
	 */
	protected function simplify_structure( $modules ) {
		// Divi structure is already relatively simple, just ensure consistency.
		return array_map(
			function( $module ) {
				return array(
					'type'       => $module['type'],
					'attributes' => $module['attrs'],
					'content'    => $module['content'],
					'children'   => isset( $module['children'] ) ? $this->simplify_structure( $module['children'] ) : array(),
				);
			},
			$modules
		);
	}
}
