<?php
/**
 * Brizy page builder implementation.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders
 * @since      1.3.0
 */

/**
 * Brizy builder class (PRO feature).
 *
 * @since 1.3.0
 */
class Respira_Builder_Brizy extends Respira_Builder_Interface {

	/**
	 * Detect if Brizy is active.
	 *
	 * @since 1.3.0
	 * @return bool True if Brizy is active.
	 */
	public function detect() {
		return defined( 'BRIZY_VERSION' ) || class_exists( 'Brizy_Editor' );
	}

	/**
	 * Get builder name.
	 *
	 * @since 1.3.0
	 * @return string Builder name.
	 */
	public function get_name() {
		return 'Brizy';
	}

	/**
	 * Get builder version.
	 *
	 * @since 1.3.0
	 * @return string Builder version.
	 */
	public function get_version() {
		if ( defined( 'BRIZY_VERSION' ) ) {
			return BRIZY_VERSION;
		}
		return 'unknown';
	}

	/**
	 * Check if post uses Brizy.
	 *
	 * @since 1.3.0
	 * @param int $post_id Post ID.
	 * @return bool True if uses Brizy.
	 */
	public function is_builder_used( $post_id ) {
		// Check for Brizy meta keys.
		$compiled = get_post_meta( $post_id, 'brizy_compiled_html', true );
		$data     = get_post_meta( $post_id, 'brizy', true );

		return ! empty( $compiled ) || ! empty( $data );
	}

	/**
	 * Extract content from post.
	 *
	 * @since 1.3.0
	 * @param int $post_id Post ID.
	 * @return array Extracted elements.
	 */
	public function extract_content( $post_id ) {
		$start = microtime( true );

		// Brizy stores data in multiple meta keys.
		$data = get_post_meta( $post_id, 'brizy', true );

		if ( empty( $data ) ) {
			return array();
		}

		// Brizy data is stored as JSON.
		$elements = json_decode( $data, true );

		if ( ! is_array( $elements ) ) {
			return array();
		}

		// Conservative extraction - focus on known elements.
		$result = $this->simplify_structure( $elements );

		$this->log_performance( 'extract', $post_id, microtime( true ) - $start );

		return $result;
	}

	/**
	 * Inject content into post.
	 *
	 * @since 1.3.0
	 * @param int   $post_id Post ID.
	 * @param array $content Simplified content.
	 * @return bool|WP_Error True on success.
	 */
	public function inject_content( $post_id, $content ) {
		$start = microtime( true );

		// Validate with intelligence if available.
		if ( class_exists( 'Respira_Brizy_Validator' ) ) {
			$validator = new Respira_Brizy_Validator();
			$validation = $validator->validate_layout( $content );

			if ( ! $validation['valid'] ) {
				$this->log_performance( 'inject', $post_id, microtime( true ) - $start );
				return new WP_Error(
					'validation_failed',
					__( 'Content validation failed.', 'respira-for-wordpress' ),
					array( 'errors' => $validation['errors'] )
				);
			}
		}

		// Conservative approach - only modify known structures.
		$elements = $this->complexify_structure( $content );

		// Update Brizy meta.
		update_post_meta( $post_id, 'brizy', wp_json_encode( $elements ) );

		// Clear compiled HTML cache.
		delete_post_meta( $post_id, 'brizy_compiled_html' );

		$this->log_performance( 'inject', $post_id, microtime( true ) - $start );

		return true;
	}

	/**
	 * Create a code block.
	 *
	 * @since 1.3.0
	 * @param int    $post_id Post ID.
	 * @param string $html    HTML content.
	 * @param string $css     CSS content.
	 * @param string $js      JavaScript content.
	 * @return bool|WP_Error True on success.
	 */
	public function create_code_block( $post_id, $html, $css = '', $js = '' ) {
		$data = get_post_meta( $post_id, 'brizy', true );
		$elements = json_decode( $data, true ) ?: array();

		// Combine code.
		$combined_code = $html;
		if ( ! empty( $css ) ) {
			$combined_code .= "\n\n<style>\n" . $css . "\n</style>";
		}
		if ( ! empty( $js ) ) {
			$combined_code .= "\n\n<script>\n" . $js . "\n</script>";
		}

		// Create code element (conservative - basic structure).
		$code_element = array(
			'type' => 'Html',
			'value' => array(
				'html' => $combined_code,
			),
		);

		$elements[] = $code_element;

		update_post_meta( $post_id, 'brizy', wp_json_encode( $elements ) );

		return true;
	}

	/**
	 * Get documentation.
	 *
	 * @since 1.3.0
	 * @return array Documentation.
	 */
	public function get_documentation() {
		$docs = array(
			'name'        => 'Brizy',
			'description' => 'Next-gen page builder with strong UX focus',
			'overview'    => 'Brizy stores builder data in postmeta. Limited public API - conservative support.',
			'structure'   => array(
				'sections' => 'Structural containers',
				'elements' => 'UI elements',
			),
			'resources'   => array(
				'https://wordpress.org/plugins/brizy/',
			),
			'note'        => __( 'Limited support due to undocumented API. Safe text-level edits only.', 'respira-for-wordpress' ),
		);

		// Add patterns if intelligence is available.
		if ( function_exists( 'respira_get_brizy_patterns' ) ) {
			$docs['patterns'] = respira_get_brizy_patterns();
		}

		// Add element catalog if intelligence is available.
		if ( class_exists( 'Respira_Brizy_Element_Registry' ) ) {
			$docs['elements'] = Respira_Brizy_Element_Registry::get_all_elements();
		}

		return $docs;
	}

	/**
	 * Get available modules.
	 *
	 * @since 1.3.0
	 * @return array Available elements.
	 */
	public function get_available_modules() {
		// Use dynamic element registry if available.
		if ( class_exists( 'Respira_Brizy_Element_Registry' ) ) {
			return Respira_Brizy_Element_Registry::get_all_elements();
		}

		// Fallback to basic list (known elements).
		return array(
			array(
				'name'        => 'Text',
				'title'       => 'Text',
				'description' => 'Text content element',
			),
			array(
				'name'        => 'Image',
				'title'       => 'Image',
				'description' => 'Image element',
			),
			array(
				'name'        => 'Button',
				'title'       => 'Button',
				'description' => 'Button element',
			),
			array(
				'name'        => 'Html',
				'title'       => 'HTML',
				'description' => 'Custom HTML/CSS/JS',
			),
		);
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since 1.4.0
	 * @param array $elements_used Optional. Array of element names used on the page.
	 * @return array Builder schema with element information.
	 */
	public function get_builder_schema( $elements_used = array() ) {
		if ( class_exists( 'Respira_Brizy_Element_Schema' ) ) {
			$schema_generator = new Respira_Brizy_Element_Schema();
			return $schema_generator->get_builder_schema( $elements_used );
		}

		// Fallback to basic schema.
		return array(
			'builder'         => 'brizy',
			'builder_version' => $this->get_version(),
			'elements'        => $this->get_available_modules(),
		);
	}

	/**
	 * Check if intelligence is available.
	 *
	 * @since 1.4.0
	 * @return bool True if intelligence is available.
	 */
	public function is_intelligence_available() {
		return class_exists( 'Respira_Brizy_Intelligence_Loader' ) &&
			   class_exists( 'Respira_Brizy_Element_Registry' ) &&
			   class_exists( 'Respira_Brizy_Element_Schema' ) &&
			   class_exists( 'Respira_Brizy_Validator' );
	}

	/**
	 * Simplify Brizy structure.
	 *
	 * @since  1.3.0
	 * @param  array $elements Brizy elements.
	 * @return array Simplified structure.
	 */
	protected function simplify_structure( $elements ) {
		return array_map(
			function( $element ) {
				$type = $element['type'] ?? 'unknown';
				$value = $element['value'] ?? array();

				return array(
					'type'       => $type,
					'attributes' => $value,
					'content'    => isset( $value['html'] ) ? $value['html'] : ( isset( $value['text'] ) ? $value['text'] : '' ),
					'children'   => isset( $element['elements'] ) ? $this->simplify_structure( $element['elements'] ) : array(),
				);
			},
			$elements
		);
	}

	/**
	 * Convert simplified structure back to Brizy format.
	 *
	 * @since  1.3.0
	 * @param  array $simplified Simplified structure.
	 * @return array Brizy format.
	 */
	protected function complexify_structure( $simplified ) {
		return array_map(
			function( $item ) {
				$element = array(
					'type'  => $item['type'] ?? 'Text',
					'value' => $item['attributes'] ?? array(),
				);

				if ( ! empty( $item['children'] ) ) {
					$element['elements'] = $this->complexify_structure( $item['children'] );
				}

				return $element;
			},
			$simplified
		);
	}
}

