<?php
/**
 * Oxygen Builder validator.
 *
 * Validates Oxygen Builder component structure before injection.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/oxygen-intelligence
 * @since      1.4.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-validator-base.php';

/**
 * Oxygen Builder validator class.
 *
 * @since 1.4.0
 */
class Respira_Oxygen_Validator extends Respira_Builder_Validator_Base {

	/**
	 * Validate a layout structure.
	 *
	 * @since  1.4.0
	 * @param  array $content Content structure to validate.
	 * @return array Validation result with 'valid' boolean and 'errors' array.
	 */
	public function validate_layout( $content ) {
		$this->clear_errors();

		if ( ! is_array( $content ) ) {
			$this->add_error( __( 'Content must be an array.', 'respira-for-wordpress' ) );
			return array(
				'valid'  => false,
				'errors' => $this->get_errors(),
			);
		}

		// Collect all component IDs to check for uniqueness.
		$component_ids = array();

		// Validate tree structure.
		$this->validate_tree( $content, $component_ids );

		// Check for duplicate IDs.
		$id_counts = array_count_values( $component_ids );
		foreach ( $id_counts as $id => $count ) {
			if ( $count > 1 ) {
				$this->add_error(
					sprintf(
						/* translators: %s: component ID */
						__( 'Duplicate component ID found: %s', 'respira-for-wordpress' ),
						$id
					)
				);
			}
		}

		return array(
			'valid'  => empty( $this->errors ),
			'errors' => $this->get_errors(),
		);
	}

	/**
	 * Validate tree structure recursively.
	 *
	 * @since  1.4.0
	 * @param  array $tree          Tree structure.
	 * @param  array &$component_ids Reference to array of component IDs.
	 * @param  int   $depth         Current depth in tree.
	 */
	private function validate_tree( $tree, &$component_ids = array(), $depth = 0 ) {
		// Oxygen uses a tree structure with 'code' key at root.
		if ( $depth === 0 && isset( $tree['code'] ) ) {
			$tree = json_decode( $tree['code'], true );
			if ( json_last_error() !== JSON_ERROR_NONE ) {
				$this->add_error(
					sprintf(
						/* translators: %s: JSON error message */
						__( 'Invalid JSON in tree: %s', 'respira-for-wordpress' ),
						json_last_error_msg()
					)
				);
				return;
			}
		}

		if ( ! is_array( $tree ) ) {
			return;
		}

		// Validate each component in the tree.
		foreach ( $tree as $key => $component ) {
			if ( is_array( $component ) ) {
				$this->validate_component( $component, $component_ids );
			}
		}
	}

	/**
	 * Validate a single component.
	 *
	 * @since  1.4.0
	 * @param  array $component      Component structure.
	 * @param  array &$component_ids Reference to array of component IDs.
	 */
	private function validate_component( $component, &$component_ids = array() ) {
		// Check required fields.
		if ( ! isset( $component['id'] ) ) {
			$this->add_error( __( 'Component must have an id.', 'respira-for-wordpress' ) );
			return;
		}

		if ( ! isset( $component['name'] ) ) {
			$this->add_error(
				sprintf(
					/* translators: %s: component ID */
					__( 'Component %s must have a name (type).', 'respira-for-wordpress' ),
					$component['id']
				)
			);
			return;
		}

		// Track component ID.
		$component_ids[] = $component['id'];

		$component_type = $component['name'];

		// Validate component type is known.
		$known_component = Respira_Oxygen_Component_Registry::get_component( $component_type );
		if ( ! $known_component ) {
			// Unknown component - allow it (may be from extension).
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( "Respira: Unknown Oxygen component type: {$component_type}" );
			}
		}

		// Validate options if present.
		if ( isset( $component['options'] ) ) {
			if ( ! is_array( $component['options'] ) ) {
				$this->add_error(
					sprintf(
						/* translators: %s: component ID */
						__( 'Component options must be an array for component %s.', 'respira-for-wordpress' ),
						$component['id']
					)
				);
			} else {
				$this->validate_component_options( $component_type, $component['options'], $component['id'] );
			}
		}

		// Validate children recursively.
		if ( isset( $component['children'] ) ) {
			if ( ! is_array( $component['children'] ) ) {
				$this->add_error(
					sprintf(
						/* translators: %s: component ID */
						__( 'Component children must be an array for component %s.', 'respira-for-wordpress' ),
						$component['id']
					)
				);
			} else {
				foreach ( $component['children'] as $child ) {
					$this->validate_component( $child, $component_ids );
				}
			}
		}
	}

	/**
	 * Validate component options.
	 *
	 * @since  1.4.0
	 * @param  string $component_type Component type.
	 * @param  array  $options        Component options.
	 * @param  string $component_id   Component ID.
	 */
	private function validate_component_options( $component_type, $options, $component_id ) {
		foreach ( $options as $option_name => $option_value ) {
			// Validate color properties.
			if ( ( strpos( $option_name, 'color' ) !== false || strpos( $option_name, '-color' ) !== false ) && is_string( $option_value ) ) {
				if ( ! empty( $option_value ) && ! $this->validate_color( $option_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: option name, 2: component type, 3: component ID */
							__( 'Invalid color format for %1$s in %2$s component %3$s.', 'respira-for-wordpress' ),
							$option_name,
							$component_type,
							$component_id
						)
					);
				}
			}

			// Validate URL properties.
			if ( $option_name === 'url' || $option_name === 'button_url' ) {
				if ( ! empty( $option_value ) && is_string( $option_value ) && ! $this->validate_url( $option_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: option name, 2: component type, 3: component ID */
							__( 'Invalid URL format for %1$s in %2$s component %3$s.', 'respira-for-wordpress' ),
							$option_name,
							$component_type,
							$component_id
						)
					);
				}
			}

			// Validate email properties.
			if ( $option_name === 'email_to' && ! empty( $option_value ) && is_string( $option_value ) ) {
				if ( ! is_email( $option_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: component type, 2: component ID */
							__( 'Invalid email format for email_to in %1$s component %2$s.', 'respira-for-wordpress' ),
							$component_type,
							$component_id
						)
					);
				}
			}

			// Validate numeric properties.
			if ( in_array( $option_name, array( 'zoom', 'speed', 'start', 'end', 'duration', 'percent', 'rating', 'posts_per_page', 'active_tab', 'active_item', 'mobile_breakpoint', 'rows', 'column-count', 'columns' ), true ) ) {
				if ( ! empty( $option_value ) && ! is_numeric( $option_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: option name, 2: component type, 3: component ID */
							__( 'Invalid numeric value for %1$s in %2$s component %3$s.', 'respira-for-wordpress' ),
							$option_name,
							$component_type,
							$component_id
						)
					);
				}
			}

			// Validate boolean properties.
			if ( in_array( $option_name, array( 'autoplay', 'loop', 'controls', 'multiple_open', 'open', 'arrows', 'dots', 'overlay', 'marker', 'required' ), true ) ) {
				if ( ! is_bool( $option_value ) && ! in_array( $option_value, array( 'true', 'false', '1', '0', 1, 0 ), true ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: option name, 2: component type, 3: component ID */
							__( 'Property %1$s must be a boolean in %2$s component %3$s.', 'respira-for-wordpress' ),
							$option_name,
							$component_type,
							$component_id
						)
					);
				}
			}

			// Validate array properties.
			if ( in_array( $option_name, array( 'items', 'slides', 'tabs', 'images', 'fields', 'features' ), true ) ) {
				if ( ! empty( $option_value ) && ! is_array( $option_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: option name, 2: component type, 3: component ID */
							__( 'Property %1$s must be an array in %2$s component %3$s.', 'respira-for-wordpress' ),
							$option_name,
							$component_type,
							$component_id
						)
					);
				}
			}
		}
	}

	/**
	 * Validate Oxygen tree structure.
	 *
	 * @since  1.4.0
	 * @param  mixed $data Data to validate.
	 * @return bool True if valid structure.
	 */
	public function validate_tree_structure( $data ) {
		if ( ! is_array( $data ) ) {
			$this->add_error( __( 'Oxygen data must be an array.', 'respira-for-wordpress' ) );
			return false;
		}

		// Check if it's wrapped in 'code' key.
		if ( isset( $data['code'] ) ) {
			$tree = json_decode( $data['code'], true );
			if ( json_last_error() !== JSON_ERROR_NONE ) {
				$this->add_error(
					sprintf(
						/* translators: %s: JSON error message */
						__( 'Invalid JSON in tree structure: %s', 'respira-for-wordpress' ),
						json_last_error_msg()
					)
				);
				return false;
			}
			$data = $tree;
		}

		// Each item should be a component with id and name.
		foreach ( $data as $component ) {
			if ( is_array( $component ) ) {
				if ( ! isset( $component['id'] ) ) {
					$this->add_error( __( 'Component missing id.', 'respira-for-wordpress' ) );
					return false;
				}

				if ( ! isset( $component['name'] ) ) {
					$this->add_error(
						sprintf(
							/* translators: %s: component ID */
							__( 'Component %s missing name (type).', 'respira-for-wordpress' ),
							$component['id']
						)
					);
					return false;
				}
			}
		}

		return true;
	}
}
