<?php
/**
 * Elementor widget registry.
 *
 * Dynamic widget detection from Elementor's registry.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/elementor-intelligence
 * @since      1.3.0
 */

/**
 * Elementor widget registry class.
 *
 * @since 1.3.0
 */
class Respira_Elementor_Widget_Registry {

	/**
	 * Get all registered widgets with metadata.
	 *
	 * @since  1.3.0
	 * @return array Array of widget information.
	 */
	public static function get_all_widgets() {
		if ( ! class_exists( '\Elementor\Plugin' ) ) {
			return array();
		}

		// Check cache first.
		$cached = get_transient( 'respira_elementor_widgets' );
		if ( false !== $cached ) {
			return $cached;
		}

		$widgets_manager = \Elementor\Plugin::$instance->widgets_manager;
		$widget_types    = $widgets_manager->get_widget_types();

		$widget_list = array();

		foreach ( $widget_types as $widget_type ) {
			$widget_info = array(
				'name'         => $widget_type->get_name(),
				'title'        => $widget_type->get_title(),
				'icon'         => $widget_type->get_icon(),
				'categories'   => $widget_type->get_categories(),
				'keywords'     => $widget_type->get_keywords(),
				'controls'     => array(),
			);

			// Extract widget controls if available.
			if ( method_exists( $widget_type, 'get_controls' ) ) {
				$widget_info['controls'] = $widget_type->get_controls();
			}

			$widget_list[] = $widget_info;
		}

		// Cache for 24 hours.
		set_transient( 'respira_elementor_widgets', $widget_list, DAY_IN_SECONDS );

		return $widget_list;
	}

	/**
	 * Get widget by name.
	 *
	 * @since  1.3.0
	 * @param  string $widget_name Widget name.
	 * @return array|null Widget information or null if not found.
	 */
	public static function get_widget( $widget_name ) {
		$widgets = self::get_all_widgets();

		foreach ( $widgets as $widget ) {
			if ( $widget['name'] === $widget_name ) {
				return $widget;
			}
		}

		return null;
	}

	/**
	 * Get widgets by category.
	 *
	 * @since  1.3.0
	 * @param  string $category Category name.
	 * @return array Widgets in category.
	 */
	public static function get_widgets_by_category( $category ) {
		$widgets = self::get_all_widgets();

		return array_filter(
			$widgets,
			function( $widget ) use ( $category ) {
				return in_array( $category, $widget['categories'], true );
			}
		);
	}

	/**
	 * Clear widget cache.
	 *
	 * @since 1.3.0
	 */
	public static function clear_cache() {
		delete_transient( 'respira_elementor_widgets' );
	}
}

