<?php
/**
 * Divi Intelligence - Layout Validator.
 *
 * Validates Divi module structures and attributes before injection.
 * Part of Divi Intelligence addon for Respira for WordPress.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/divi-intelligence
 */

/**
 * Divi validator class.
 *
 * @since 1.0.0
 */
class Respira_Divi_Validator {

	/**
	 * Validate module structure.
	 *
	 * Checks for required parent modules and proper nesting.
	 *
	 * @since 1.0.0
	 * @param array $modules Array of modules to validate.
	 * @return array Array with 'valid' (bool) and 'errors' (array).
	 */
	public static function validate_structure( $modules ) {
		$errors = array();

		foreach ( $modules as $index => $module ) {
			$module_type = $module['type'] ?? '';

			// Validate module type.
			if ( empty( $module_type ) || strpos( $module_type, 'et_pb_' ) !== 0 ) {
				$errors[] = sprintf(
					/* translators: %d: module index */
					__( 'Module at index %d has invalid type.', 'respira-for-wordpress' ),
					$index
				);
				continue;
			}

			// Check nesting rules.
			$nesting_errors = self::validate_nesting( $module, $index );
			$errors         = array_merge( $errors, $nesting_errors );

			// Validate children recursively.
			if ( ! empty( $module['children'] ) ) {
				$child_result = self::validate_structure( $module['children'] );
				if ( ! $child_result['valid'] ) {
					$errors = array_merge( $errors, $child_result['errors'] );
				}
			}
		}

		return array(
			'valid'  => empty( $errors ),
			'errors' => $errors,
		);
	}

	/**
	 * Validate module nesting rules.
	 *
	 * @since 1.0.0
	 * @param array $module Module to validate.
	 * @param int   $index  Module index.
	 * @return array Array of error messages.
	 */
	private static function validate_nesting( $module, $index ) {
		$errors = array();
		$type   = $module['type'] ?? '';

		// Rows must be inside sections.
		if ( 'et_pb_row' === $type ) {
			// This check would need parent context, so we'll check at a higher level.
			// For now, we just ensure rows have valid structure.
		}

		// Modules (except sections and rows) should typically be inside rows.
		// But we allow flexibility for edge cases.

		// Check for maximum nesting depth (prevent infinite recursion).
		$depth = self::calculate_depth( $module );
		if ( $depth > 10 ) {
			$errors[] = sprintf(
				/* translators: 1: module type, 2: depth */
				__( 'Module %1$s at index %2$d exceeds maximum nesting depth.', 'respira-for-wordpress' ),
				$type,
				$index
			);
		}

		return $errors;
	}

	/**
	 * Calculate nesting depth of a module.
	 *
	 * @since 1.0.0
	 * @param array $module Module to check.
	 * @return int Nesting depth.
	 */
	private static function calculate_depth( $module, $current_depth = 0 ) {
		if ( $current_depth > 10 ) {
			return $current_depth; // Prevent infinite recursion.
		}

		if ( empty( $module['children'] ) ) {
			return $current_depth;
		}

		$max_child_depth = $current_depth;
		foreach ( $module['children'] as $child ) {
			$child_depth = self::calculate_depth( $child, $current_depth + 1 );
			if ( $child_depth > $max_child_depth ) {
				$max_child_depth = $child_depth;
			}
		}

		return $max_child_depth;
	}

	/**
	 * Validate module attributes against schema.
	 *
	 * @since 1.0.0
	 * @param string $module_name Module name (e.g., 'et_pb_text').
	 * @param array  $attributes   Attributes to validate.
	 * @return array Array with 'valid' (bool) and 'errors' (array).
	 */
	public static function validate_attributes( $module_name, $attributes ) {
		$errors = array();

		if ( ! class_exists( 'Respira_Divi_Module_Schema' ) ) {
			// Can't validate without schema.
			return array(
				'valid'  => true,
				'errors' => array(),
			);
		}

		$schema = Respira_Divi_Module_Schema::get_module_schema( $module_name );
		if ( ! $schema ) {
			// No schema available, skip validation.
			return array(
				'valid'  => true,
				'errors' => array(),
			);
		}

		$schema_attrs = $schema['attributes'] ?? array();

		// Check required attributes.
		foreach ( $schema_attrs as $attr_name => $attr_def ) {
			if ( ! empty( $attr_def['required'] ) && ! isset( $attributes[ $attr_name ] ) ) {
				$errors[] = sprintf(
					/* translators: 1: attribute name, 2: module name */
					__( 'Required attribute "%1$s" missing for module %2$s.', 'respira-for-wordpress' ),
					$attr_name,
					$module_name
				);
			}
		}

		// Validate attribute formats.
		foreach ( $attributes as $attr_name => $attr_value ) {
			if ( ! isset( $schema_attrs[ $attr_name ] ) ) {
				// Unknown attribute - allow it (Divi has many dynamic attributes).
				continue;
			}

			$attr_def = $schema_attrs[ $attr_name ];
			$format   = $attr_def['format'] ?? '';

			// Validate color format.
			if ( 'color' === $attr_def['type'] && 'hex' === $format ) {
				if ( ! preg_match( '/^#[0-9A-Fa-f]{6}$/', $attr_value ) ) {
					$errors[] = sprintf(
						/* translators: 1: attribute name, 2: module name */
						__( 'Attribute "%1$s" for module %2$s must be a valid hex color (e.g., #000000).', 'respira-for-wordpress' ),
						$attr_name,
						$module_name
					);
				}
			}

			// Validate URL format.
			if ( 'url' === $format && ! empty( $attr_value ) ) {
				if ( ! filter_var( $attr_value, FILTER_VALIDATE_URL ) ) {
					$errors[] = sprintf(
						/* translators: 1: attribute name, 2: module name */
						__( 'Attribute "%1$s" for module %2$s must be a valid URL.', 'respira-for-wordpress' ),
						$attr_name,
						$module_name
					);
				}
			}
		}

		return array(
			'valid'  => empty( $errors ),
			'errors' => $errors,
		);
	}

	/**
	 * Validate complete layout before injection.
	 *
	 * @since 1.0.0
	 * @param array $modules Array of modules to validate.
	 * @return array Array with 'valid' (bool) and 'errors' (array).
	 */
	public static function validate_layout( $modules ) {
		$errors = array();

		// Validate structure.
		$structure_result = self::validate_structure( $modules );
		if ( ! $structure_result['valid'] ) {
			$errors = array_merge( $errors, $structure_result['errors'] );
		}

		// Validate attributes for each module.
		foreach ( $modules as $module ) {
			$module_type = $module['type'] ?? '';
			$attributes  = $module['attributes'] ?? array();

			if ( ! empty( $module_type ) ) {
				$attr_result = self::validate_attributes( $module_type, $attributes );
				if ( ! $attr_result['valid'] ) {
					$errors = array_merge( $errors, $attr_result['errors'] );
				}
			}

			// Validate children recursively.
			if ( ! empty( $module['children'] ) ) {
				$child_result = self::validate_layout( $module['children'] );
				if ( ! $child_result['valid'] ) {
					$errors = array_merge( $errors, $child_result['errors'] );
				}
			}
		}

		return array(
			'valid'  => empty( $errors ),
			'errors' => $errors,
		);
	}
}

