<?php
/**
 * Divi Intelligence - Module Registry.
 *
 * Dynamically detects and catalogs all available Divi modules.
 * Part of Divi Intelligence addon for Respira for WordPress.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/divi-intelligence
 */

/**
 * Divi module registry class.
 *
 * @since 1.0.0
 */
class Respira_Divi_Module_Registry {

	/**
	 * Cache key for module list.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const CACHE_KEY = 'respira_divi_modules';

	/**
	 * Cache expiration (24 hours).
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const CACHE_EXPIRATION = 86400;

	/**
	 * Get all registered Divi modules.
	 *
	 * @since 1.0.0
	 * @return array Array of module definitions.
	 */
	public static function get_all_modules() {
		// Try cache first.
		$cached = get_transient( self::CACHE_KEY );
		if ( false !== $cached ) {
			return $cached;
		}

		$modules = self::detect_modules();

		// Cache for 24 hours.
		set_transient( self::CACHE_KEY, $modules, self::CACHE_EXPIRATION );

		return $modules;
	}

	/**
	 * Detect all Divi modules.
	 *
	 * @since 1.0.0
	 * @return array Array of module definitions.
	 */
	private static function detect_modules() {
		$modules = array();

		// Strategy 1: Try to use Divi's module registry if available.
		if ( class_exists( 'ET_Builder_Module' ) ) {
			$modules = self::detect_from_divi_registry();
		}

		// Strategy 2: Scan WordPress shortcode registry.
		if ( empty( $modules ) ) {
			$modules = self::detect_from_shortcode_registry();
		}

		// Strategy 3: Fallback to comprehensive static list.
		if ( empty( $modules ) ) {
			$modules = self::get_static_module_list();
		}

		return $modules;
	}

	/**
	 * Detect modules from Divi's internal registry.
	 *
	 * @since 1.0.0
	 * @return array Array of module definitions.
	 */
	private static function detect_from_divi_registry() {
		$modules = array();

		// Check if ET_Builder_Module has a registry method.
		if ( class_exists( 'ET_Builder_Module' ) && method_exists( 'ET_Builder_Module', 'get_modules' ) ) {
			$divi_modules = ET_Builder_Module::get_modules();
			foreach ( $divi_modules as $module_name => $module_class ) {
				if ( strpos( $module_name, 'et_pb_' ) === 0 ) {
					$modules[] = array(
						'name'        => $module_name,
						'title'       => self::get_module_title( $module_name ),
						'description' => self::get_module_description( $module_name ),
						'category'    => self::get_module_category( $module_name ),
					);
				}
			}
		}

		return $modules;
	}

	/**
	 * Detect modules from WordPress shortcode registry.
	 *
	 * @since 1.0.0
	 * @return array Array of module definitions.
	 */
	private static function detect_from_shortcode_registry() {
		global $shortcode_tags;
		$modules = array();

		if ( ! empty( $shortcode_tags ) ) {
			foreach ( $shortcode_tags as $tag => $callback ) {
				if ( strpos( $tag, 'et_pb_' ) === 0 ) {
					$modules[] = array(
						'name'        => $tag,
						'title'       => self::get_module_title( $tag ),
						'description' => self::get_module_description( $tag ),
						'category'    => self::get_module_category( $tag ),
					);
				}
			}
		}

		return $modules;
	}

	/**
	 * Get comprehensive static list of all known Divi modules.
	 *
	 * @since 1.0.0
	 * @return array Array of module definitions.
	 */
	private static function get_static_module_list() {
		// Load module definitions from data file.
		$module_file = RESPIRA_PLUGIN_DIR . 'includes/divi-intelligence/divi-modules.php';
		if ( file_exists( $module_file ) ) {
			require_once $module_file;
			if ( function_exists( 'respira_get_divi_module_definitions' ) ) {
				return respira_get_divi_module_definitions();
			}
		}

		// Fallback: Try to detect from Divi's internal structure.
		return self::detect_from_divi_internal();
	}

	/**
	 * Detect modules from Divi's internal structure.
	 *
	 * @since 1.0.0
	 * @return array Array of module definitions.
	 */
	private static function detect_from_divi_internal() {
		$modules = array();

		// Try to use Divi's module loader if available.
		if ( class_exists( 'ET_Builder_Module' ) ) {
			// Use reflection to get all module classes.
			$builder_dir = '';
			if ( defined( 'ET_BUILDER_DIR' ) ) {
				$builder_dir = ET_BUILDER_DIR . 'modules/';
			} elseif ( defined( 'ET_CORE_PATH' ) ) {
				$builder_dir = ET_CORE_PATH . 'app/models/builder/module/';
			}

			if ( ! empty( $builder_dir ) && is_dir( $builder_dir ) ) {
				$module_files = glob( $builder_dir . '*.php' );
				foreach ( $module_files as $file ) {
					$module_name = basename( $file, '.php' );
					if ( strpos( $module_name, 'et_pb_' ) === 0 ) {
						$modules[] = array(
							'name'        => $module_name,
							'title'       => self::get_module_title( $module_name ),
							'description' => self::get_module_description( $module_name ),
							'category'    => self::get_module_category( $module_name ),
						);
					}
				}
			}
		}

		// If still empty, return comprehensive list from data file.
		if ( empty( $modules ) ) {
			$module_file = RESPIRA_PLUGIN_DIR . 'includes/divi-intelligence/divi-modules.php';
			if ( file_exists( $module_file ) ) {
				require_once $module_file;
				if ( function_exists( 'respira_get_divi_module_definitions' ) ) {
					$modules = respira_get_divi_module_definitions();
					// Remove tier from all modules.
					foreach ( $modules as &$module ) {
						unset( $module['tier'] );
					}
				}
			}
		}

		return $modules;
	}

	/**
	 * Get module title.
	 *
	 * @since 1.0.0
	 * @param string $module_name Module name.
	 * @return string Module title.
	 */
	private static function get_module_title( $module_name ) {
		// Remove et_pb_ prefix and format.
		$title = str_replace( 'et_pb_', '', $module_name );
		$title = str_replace( '_', ' ', $title );
		return ucwords( $title );
	}

	/**
	 * Get module description.
	 *
	 * @since 1.0.0
	 * @param string $module_name Module name.
	 * @return string Module description.
	 */
	private static function get_module_description( $module_name ) {
		// Default description - can be enhanced with lookup table.
		return sprintf(
			/* translators: %s: module name */
			__( 'Divi %s module', 'respira-for-wordpress' ),
			self::get_module_title( $module_name )
		);
	}

	/**
	 * Get module category.
	 *
	 * @since 1.0.0
	 * @param string $module_name Module name.
	 * @return string Module category.
	 */
	private static function get_module_category( $module_name ) {
		// Categorize based on module name patterns.
		if ( strpos( $module_name, 'section' ) !== false || strpos( $module_name, 'row' ) !== false ) {
			return 'layout';
		}
		if ( strpos( $module_name, 'image' ) !== false || strpos( $module_name, 'video' ) !== false || strpos( $module_name, 'gallery' ) !== false ) {
			return 'media';
		}
		if ( strpos( $module_name, 'form' ) !== false || strpos( $module_name, 'contact' ) !== false ) {
			return 'form';
		}
		return 'content';
	}

	/**
	 * Get module tier (deprecated - all modules are now equal priority).
	 *
	 * @since 1.0.0
	 * @param string $module_name Module name.
	 * @return int Always returns 1 (all modules equal priority).
	 */
	private static function get_module_tier( $module_name ) {
		// All modules are now equal priority - full support for all.
		return 1;
	}

	/**
	 * Get modules by tier (deprecated - kept for backward compatibility).
	 *
	 * @since 1.0.0
	 * @param int $tier Tier number (1, 2, or 3).
	 * @return array Array of module definitions.
	 */
	public static function get_modules_by_tier( $tier ) {
		// Return all modules regardless of tier - full support.
		return self::get_all_modules();
	}

	/**
	 * Clear module cache.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public static function clear_cache() {
		delete_transient( self::CACHE_KEY );
	}
}

