<?php
/**
 * Plugin updater class.
 *
 * Handles automatic updates from GitHub releases via custom update server.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes
 */

/**
 * Plugin updater class.
 *
 * @since 1.1.0
 */
class Respira_Updater {

	/**
	 * Update server URL.
	 *
	 * @since 1.1.0
	 * @var string
	 */
	private $update_server_url = 'https://respira.press/api/plugin-updates';

	/**
	 * GitHub repository owner.
	 *
	 * @since 1.1.0
	 * @var string
	 */
	private $github_owner = 'webmyc';

	/**
	 * GitHub repository name.
	 *
	 * @since 1.1.0
	 * @var string
	 */
	private $github_repo = 'respira-wordpress';

	/**
	 * Plugin slug.
	 *
	 * @since 1.1.0
	 * @var string
	 */
	private $plugin_slug = 'respira-for-wordpress';

	/**
	 * Plugin basename.
	 *
	 * @since 1.1.0
	 * @var string
	 */
	private $plugin_basename;

	/**
	 * Current plugin version.
	 *
	 * @since 1.1.0
	 * @var string
	 */
	private $current_version;

	/**
	 * Transient cache key.
	 *
	 * @since 1.1.0
	 * @var string
	 */
	private $cache_key = 'respira_update_info';

	/**
	 * Cache expiration (4 hours).
	 *
	 * @since 1.1.0
	 * @var int
	 */
	private $cache_expiration = 14400;

	/**
	 * Initialize the updater.
	 *
	 * @since 1.1.0
	 */
	public function __construct() {
		$this->plugin_basename = RESPIRA_PLUGIN_BASENAME;
		$this->current_version = RESPIRA_VERSION;

		// Hook into WordPress update system.
		add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'check_for_updates' ), 10, 1 );
		add_filter( 'plugins_api', array( $this, 'plugin_information' ), 10, 3 );
		add_filter( 'upgrader_post_install', array( $this, 'post_install' ), 10, 3 );
		add_action( 'upgrader_process_complete', array( $this, 'clear_update_cache' ), 10, 2 );
	}

	/**
	 * Check for plugin updates.
	 *
	 * @since 1.1.0
	 * @param object $transient The update_plugins transient object.
	 * @return object Modified transient object.
	 */
	public function check_for_updates( $transient ) {
		// If we've already checked recently, use cached data.
		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		// Get latest release info.
		$release_info = $this->get_latest_release();

		if ( ! $release_info || is_wp_error( $release_info ) ) {
			return $transient;
		}

		// Compare versions.
		if ( version_compare( $this->current_version, $release_info['version'], '<' ) ) {
			// Ensure download URL is properly formatted
			$download_url = $release_info['download_url'];
			
			// If it's a GitHub URL and might need authentication, use proxy
			if ( strpos( $download_url, 'github.com/repos' ) !== false || strpos( $download_url, 'githubusercontent.com' ) !== false ) {
				// Use proxy endpoint for reliable downloads
				$download_url = add_query_arg(
					array( 'version' => $release_info['version'] ),
					'https://respira.press/api/plugin-updates/download'
				);
			}
			
			$transient->response[ $this->plugin_basename ] = (object) array(
				'slug'        => $this->plugin_slug,
				'plugin'      => $this->plugin_basename,
				'new_version'  => $release_info['version'],
				'url'          => $release_info['url'],
				'package'     => $download_url,
				'icons'       => array(),
				'banners'     => array(),
				'banners_rtl' => array(),
			);
		}

		return $transient;
	}

	/**
	 * Get plugin information for the "View details" popup.
	 *
	 * @since 1.1.0
	 * @param false|object|array $result The result object or array.
	 * @param string             $action The type of information being requested.
	 * @param object             $args Plugin API arguments.
	 * @return object|false Plugin information or false.
	 */
	public function plugin_information( $result, $action, $args ) {
		// Only respond to our plugin.
		if ( 'plugin_information' !== $action || $this->plugin_slug !== $args->slug ) {
			return $result;
		}

		// Get latest release info.
		$release_info = $this->get_latest_release();

		if ( ! $release_info || is_wp_error( $release_info ) ) {
			return $result;
		}

		// Build response object.
		$response                = new stdClass();
		$response->name          = 'Respira for WordPress';
		$response->slug          = $this->plugin_slug;
		$response->version       = $release_info['version'];
		$response->author        = '<a href="https://respira.press">Respira</a>';
		$response->homepage      = 'https://respira.press';
		$response->requires      = '5.8';
		$response->tested        = '6.7';
		$response->requires_php  = '7.4';
		$response->last_updated  = $release_info['published_at'];
		$response->download_link = $release_info['download_url'];
		$response->sections       = array(
			'description' => $release_info['description'],
			'changelog'   => $release_info['changelog'],
		);

		return $response;
	}

	/**
	 * Get latest release from update server or GitHub.
	 *
	 * @since 1.1.0
	 * @return array|WP_Error Release information or error.
	 */
	protected function get_latest_release() {
		// Check if force-check is requested (e.g., from WordPress Updates page)
		$force_check = isset( $_GET['force-check'] ) && '1' === $_GET['force-check'];
		if ( $force_check ) {
			delete_transient( $this->cache_key );
		}
		
		// Check cache first.
		$cached = get_transient( $this->cache_key );
		if ( false !== $cached && ! $force_check ) {
			return $cached;
		}

		// Try custom update server first (works for private repos).
		$release_info = $this->get_release_from_update_server();
		
		// Fallback to GitHub API if update server fails (works for public repos).
		if ( is_wp_error( $release_info ) ) {
			$release_info = $this->get_release_from_github();
		}

		if ( is_wp_error( $release_info ) || ! $release_info ) {
			return $release_info;
		}

		// Cache the result.
		set_transient( $this->cache_key, $release_info, $this->cache_expiration );

		return $release_info;
	}
	
	/**
	 * Clear update cache manually.
	 * Can be called to force immediate update check.
	 *
	 * @since 1.8.7
	 * @return bool True on success, false on failure.
	 */
	public static function clear_cache() {
		return delete_transient( 'respira_update_info' );
	}

	/**
	 * Get release info from custom update server.
	 *
	 * @since 1.1.0
	 * @return array|WP_Error Release information or error.
	 */
	private function get_release_from_update_server() {
		$url = add_query_arg(
			array(
				'plugin' => $this->plugin_slug,
				'version' => $this->current_version,
			),
			$this->update_server_url
		);

		$response = wp_remote_get(
			$url,
			array(
				'timeout' => 15,
				'headers' => array(
					'Accept' => 'application/json',
					'User-Agent' => 'Respira-WordPress-Plugin',
				),
			)
		);

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$code = wp_remote_retrieve_response_code( $response );
		if ( 200 !== $code ) {
			$body = wp_remote_retrieve_body( $response );
			return new WP_Error( 
				'update_server_error', 
				sprintf( 
					__( 'Update server returned an error (HTTP %d): %s', 'respira-for-wordpress' ),
					$code,
					$body
				)
			);
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( ! $data || ! isset( $data['version'] ) ) {
			return new WP_Error( 'invalid_response', __( 'Invalid response from update server.', 'respira-for-wordpress' ) );
		}

		return array(
			'version'      => $data['version'],
			'url'          => $data['url'] ?? '',
			'download_url' => $data['download_url'] ?? '',
			'published_at' => $data['published_at'] ?? '',
			'description'  => $data['description'] ?? '',
			'changelog'    => $data['changelog'] ?? '',
		);
	}

	/**
	 * Get latest release from GitHub API (fallback for public repos).
	 *
	 * @since 1.1.0
	 * @return array|WP_Error Release information or error.
	 */
	private function get_release_from_github() {
		// Build GitHub API URL.
		$api_url = sprintf(
			'https://api.github.com/repos/%s/%s/releases/latest',
			$this->github_owner,
			$this->github_repo
		);

		// Build headers.
		$headers = array(
			'Accept'     => 'application/vnd.github.v3+json',
			'User-Agent' => 'Respira-WordPress-Plugin',
		);

		// Add GitHub token for private repos (optional).
		$github_token = get_option( 'respira_github_token', '' );
		if ( ! empty( $github_token ) ) {
			$headers['Authorization'] = 'Bearer ' . $github_token;
		}

		// Make API request.
		$response = wp_remote_get(
			$api_url,
			array(
				'timeout' => 15,
				'headers' => $headers,
			)
		);

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$code = wp_remote_retrieve_response_code( $response );
		if ( 200 !== $code ) {
			return new WP_Error( 'github_api_error', __( 'Failed to fetch release information from GitHub.', 'respira-for-wordpress' ) );
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( ! $data || ! isset( $data['tag_name'] ) ) {
			return new WP_Error( 'invalid_response', __( 'Invalid response from GitHub API.', 'respira-for-wordpress' ) );
		}

		// Extract version from tag (remove 'v' prefix if present).
		$version = ltrim( $data['tag_name'], 'v' );

		// Find the plugin ZIP download URL.
		$download_url = '';
		if ( isset( $data['assets'] ) && is_array( $data['assets'] ) ) {
			// First, try to find versioned file (e.g., respira-for-wordpress-1.7.0.zip).
			foreach ( $data['assets'] as $asset ) {
				if ( isset( $asset['browser_download_url'] ) && preg_match( '/respira-for-wordpress.*\.zip$/i', $asset['name'] ) ) {
					// Prefer exact version match.
					if ( strpos( $asset['name'], $version ) !== false ) {
						$download_url = $asset['browser_download_url'];
						break;
					}
				}
			}
			
			// If no versioned file found, try latest.zip.
			if ( empty( $download_url ) ) {
				foreach ( $data['assets'] as $asset ) {
					if ( isset( $asset['browser_download_url'] ) && preg_match( '/respira-for-wordpress.*latest.*\.zip$/i', $asset['name'] ) ) {
						$download_url = $asset['browser_download_url'];
						break;
					}
				}
			}
			
			// Last resort: any respira-for-wordpress zip file.
			if ( empty( $download_url ) ) {
				foreach ( $data['assets'] as $asset ) {
					if ( isset( $asset['browser_download_url'] ) && preg_match( '/respira-for-wordpress.*\.zip$/i', $asset['name'] ) ) {
						$download_url = $asset['browser_download_url'];
						break;
					}
				}
			}
		}

		if ( empty( $download_url ) ) {
			return new WP_Error( 
				'no_download', 
				sprintf( 
					__( 'Download URL not found in release assets. Release: %s, Assets: %s', 'respira-for-wordpress' ),
					$version,
					isset( $data['assets'] ) ? count( $data['assets'] ) : 0
				)
			);
		}

		// Parse changelog from release body.
		$changelog = $this->parse_changelog( $data['body'] ?? '' );

		// Build release info.
		$release_info = array(
			'version'      => $version,
			'url'          => $data['html_url'] ?? '',
			'download_url' => $download_url,
			'published_at' => $data['published_at'] ?? '',
			'description'  => $data['body'] ?? '',
			'changelog'    => $changelog,
		);

		return $release_info;
	}

	/**
	 * Parse changelog from release body.
	 *
	 * @since 1.1.0
	 * @param string $body Release body text.
	 * @return string Formatted changelog.
	 */
	private function parse_changelog( $body ) {
		if ( empty( $body ) ) {
			return __( 'No changelog available.', 'respira-for-wordpress' );
		}

		// Convert markdown-style headers to HTML.
		$changelog = $body;
		$changelog = preg_replace( '/^##\s+(.+)$/m', '<h3>$1</h3>', $changelog );
		$changelog = preg_replace( '/^###\s+(.+)$/m', '<h4>$1</h4>', $changelog );
		$changelog = preg_replace( '/^\*\s+(.+)$/m', '<li>$1</li>', $changelog );
		$changelog = nl2br( $changelog );

		return $changelog;
	}

	/**
	 * Clear update cache after successful update.
	 *
	 * @since 1.1.0
	 * @param WP_Upgrader $upgrader Upgrader instance.
	 * @param array        $options  Options array.
	 */
	public function clear_update_cache( $upgrader, $options ) {
		if ( isset( $options['plugins'] ) && in_array( $this->plugin_basename, $options['plugins'], true ) ) {
			delete_transient( $this->cache_key );
		}
	}

	/**
	 * Post-install hook to ensure plugin is activated.
	 *
	 * @since 1.1.0
	 * @param bool  $response   Installation response.
	 * @param array $hook_extra Extra arguments.
	 * @param array $result      Installation result.
	 * @return bool Response.
	 */
	public function post_install( $response, $hook_extra, $result ) {
		if ( isset( $hook_extra['plugin'] ) && $hook_extra['plugin'] === $this->plugin_basename ) {
			// Clear update cache.
			delete_transient( $this->cache_key );
		}

		return $response;
	}

	/**
	 * Check if an update is available.
	 *
	 * @since 1.8.7
	 * @return array|false Array with update info if available, false otherwise.
	 *                    Array contains: 'available' (bool), 'version' (string), 'url' (string), 'changelog' (string)
	 */
	public static function check_update_available() {
		// Check if class constants are defined
		if ( ! defined( 'RESPIRA_VERSION' ) ) {
			return false;
		}
		
		try {
			// Create instance to access protected methods
			$updater = new self();
			
			// Get latest release info (uses 4-hour cache)
			$release_info = $updater->get_latest_release();
			
			if ( is_wp_error( $release_info ) || ! $release_info || ! isset( $release_info['version'] ) ) {
				return false;
			}
			
			// Compare versions
			$current_version = RESPIRA_VERSION;
			$latest_version = $release_info['version'];
			
			if ( version_compare( $current_version, $latest_version, '<' ) ) {
				return array(
					'available' => true,
					'version'  => $latest_version,
					'url'      => $release_info['url'] ?? 'https://respira.press/releases',
					'changelog' => $release_info['changelog'] ?? '',
				);
			}
		} catch ( Exception $e ) {
			// Silently fail - don't break the site if update check fails
			return false;
		}
		
		return false;
	}
}
