<?php
/**
 * Theme detector class.
 *
 * Detects active theme, parent theme, version, and theme-specific features.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes
 * @since      1.3.0
 */

/**
 * Theme detector class.
 *
 * @since 1.3.0
 */
class Respira_Theme_Detector {

	/**
	 * Get theme information.
	 *
	 * @since  1.3.0
	 * @return array Theme information array.
	 */
	public static function get_theme_info() {
		// Check transient first.
		$cached = get_transient( 'respira_theme_info' );
		if ( false !== $cached ) {
			return $cached;
		}

		$theme = wp_get_theme();

		$info = array(
			'name'         => $theme->get( 'Name' ),
			'version'      => $theme->get( 'Version' ),
			'description'  => $theme->get( 'Description' ),
			'author'       => $theme->get( 'Author' ),
			'author_uri'   => $theme->get( 'AuthorURI' ),
			'theme_uri'    => $theme->get( 'ThemeURI' ),
			'template'     => $theme->get_template(),
			'stylesheet'   => $theme->get_stylesheet(),
			'is_child'     => is_child_theme(),
			'parent_theme' => null,
		);

		// Get parent theme if child theme.
		if ( $info['is_child'] ) {
			$parent = $theme->parent();
			if ( $parent ) {
				$info['parent_theme'] = array(
					'name'        => $parent->get( 'Name' ),
					'version'     => $parent->get( 'Version' ),
					'description' => $parent->get( 'Description' ),
					'author'      => $parent->get( 'Author' ),
					'template'    => $parent->get_template(),
				);
			}
		}

		// Detect theme-specific features.
		$info['features'] = self::detect_theme_features( $info );

		// Cache for 1 hour.
		set_transient( 'respira_theme_info', $info, HOUR_IN_SECONDS );

		return $info;
	}

	/**
	 * Detect theme-specific features.
	 *
	 * @since  1.3.0
	 * @param  array $theme_info Theme information.
	 * @return array Array of detected features.
	 */
	private static function detect_theme_features( $theme_info ) {
		$features = array();

		$theme_name = strtolower( $theme_info['name'] );
		$template   = strtolower( $theme_info['template'] );

		// Detect popular themes.
		if ( 'divi' === $theme_name || 'divi' === $template ) {
			$features[] = 'divi_builder';
		}

		if ( 'astra' === $theme_name || 'astra' === $template ) {
			$features[] = 'astra';
		}

		if ( 'generatepress' === $theme_name || 'generatepress' === $template ) {
			$features[] = 'generatepress';
		}

		if ( 'oceanwp' === $theme_name || 'oceanwp' === $template ) {
			$features[] = 'oceanwp';
		}

		if ( 'neve' === $theme_name || 'neve' === $template ) {
			$features[] = 'neve';
		}

		// Detect if theme supports block editor.
		if ( current_theme_supports( 'block-templates' ) || current_theme_supports( 'block-template-parts' ) ) {
			$features[] = 'block_theme';
		}

		// Detect if theme supports customizer.
		if ( current_theme_supports( 'customize-selective-refresh-widgets' ) ) {
			$features[] = 'customizer';
		}

		// Detect if theme supports post thumbnails.
		if ( current_theme_supports( 'post-thumbnails' ) ) {
			$features[] = 'post_thumbnails';
		}

		// Detect if theme supports menus.
		if ( current_theme_supports( 'menus' ) ) {
			$features[] = 'menus';
		}

		// Detect if theme supports widgets.
		if ( current_theme_supports( 'widgets' ) ) {
			$features[] = 'widgets';
		}

		return $features;
	}

	/**
	 * Clear theme info cache.
	 *
	 * @since 1.3.0
	 */
	public static function clear_cache() {
		delete_transient( 'respira_theme_info' );
	}
}

