<?php
/**
 * Plugin Manager
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes
 */

/**
 * Manages WordPress plugins (install, activate, deactivate, update, delete).
 *
 * EXPERIMENTAL FEATURE - Must be explicitly enabled in settings.
 *
 * @since 1.0.0
 */
class Respira_Plugin_Manager {

	/**
	 * Check if plugin management is enabled
	 *
	 * @since 1.0.0
	 * @return bool True if enabled.
	 */
	public function is_enabled() {
		return (bool) get_option( 'respira_enable_plugin_management', false );
	}

	/**
	 * List all plugins with their status
	 *
	 * @since 1.0.0
	 * @return array Plugin list with status.
	 */
	public function list_plugins() {
		if ( ! $this->is_enabled() ) {
			return array(
				'success' => false,
				'message' => 'Plugin management is disabled. Enable it in Respira settings.',
			);
		}

		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$all_plugins    = get_plugins();
		$active_plugins = get_option( 'active_plugins', array() );
		$plugins_list   = array();

		foreach ( $all_plugins as $plugin_file => $plugin_data ) {
			$is_active = in_array( $plugin_file, $active_plugins, true );

			// Check for updates.
			$update_available = false;
			$new_version      = null;
			$update_plugins   = get_site_transient( 'update_plugins' );
			if ( isset( $update_plugins->response[ $plugin_file ] ) ) {
				$update_available = true;
				$new_version      = $update_plugins->response[ $plugin_file ]->new_version;
			}

			$plugins_list[] = array(
				'slug'             => dirname( $plugin_file ),
				'file'             => $plugin_file,
				'name'             => $plugin_data['Name'],
				'version'          => $plugin_data['Version'],
				'author'           => $plugin_data['Author'],
				'description'      => $plugin_data['Description'],
				'is_active'        => $is_active,
				'update_available' => $update_available,
				'new_version'      => $new_version,
			);
		}

		return array(
			'success'      => true,
			'plugins'      => $plugins_list,
			'total'        => count( $plugins_list ),
			'active_count' => count( $active_plugins ),
			'warning'      => 'EXPERIMENTAL: Plugin management feature is active',
		);
	}

	/**
	 * Install a plugin from WordPress.org or ZIP URL
	 *
	 * @since 1.0.0
	 * @param string $slug_or_url Plugin slug or ZIP URL.
	 * @param string $source Source type ('wordpress.org' or 'url').
	 * @return array Installation result.
	 */
	public function install_plugin( $slug_or_url, $source = 'wordpress.org' ) {
		if ( ! $this->is_enabled() ) {
			return array(
				'success' => false,
				'message' => 'Plugin management is disabled. Enable it in Respira settings.',
			);
		}

		if ( ! current_user_can( 'install_plugins' ) ) {
			return array(
				'success' => false,
				'message' => 'You do not have permission to install plugins.',
			);
		}

		// Log action.
		$this->log_action( 'install', $slug_or_url, $source );

		// Load WordPress plugin installer.
		if ( ! class_exists( 'Plugin_Upgrader' ) ) {
			require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		}
		if ( ! class_exists( 'WP_Ajax_Upgrader_Skin' ) ) {
			require_once ABSPATH . 'wp-admin/includes/class-wp-ajax-upgrader-skin.php';
		}

		$skin     = new WP_Ajax_Upgrader_Skin();
		$upgrader = new Plugin_Upgrader( $skin );

		if ( $source === 'wordpress.org' ) {
			// Install from WordPress.org.
			$api = plugins_api(
				'plugin_information',
				array(
					'slug'   => $slug_or_url,
					'fields' => array( 'sections' => false ),
				)
			);

			if ( is_wp_error( $api ) ) {
				return array(
					'success' => false,
					'message' => 'Plugin not found on WordPress.org: ' . $api->get_error_message(),
				);
			}

			$download_link = $api->download_link;
		} else {
			// Install from ZIP URL.
			$download_link = $slug_or_url;
		}

		$result = $upgrader->install( $download_link );

		if ( is_wp_error( $result ) ) {
			return array(
				'success' => false,
				'message' => 'Installation failed: ' . $result->get_error_message(),
			);
		}

		if ( ! $result ) {
			return array(
				'success' => false,
				'message' => 'Installation failed.',
			);
		}

		// Get plugin file.
		$plugin_file = $upgrader->plugin_info();

		return array(
			'success' => true,
			'message' => 'Plugin installed successfully',
			'plugin'  => array(
				'file' => $plugin_file,
				'slug' => dirname( $plugin_file ),
			),
			'warning' => 'EXPERIMENTAL: Plugin installed but not activated. Use activate endpoint to activate.',
		);
	}

	/**
	 * Activate a plugin
	 *
	 * @since 1.0.0
	 * @param string $plugin_slug Plugin slug or file.
	 * @return array Activation result.
	 */
	public function activate_plugin( $plugin_slug ) {
		if ( ! $this->is_enabled() ) {
			return array(
				'success' => false,
				'message' => 'Plugin management is disabled. Enable it in Respira settings.',
			);
		}

		if ( ! current_user_can( 'activate_plugins' ) ) {
			return array(
				'success' => false,
				'message' => 'You do not have permission to activate plugins.',
			);
		}

		// Get plugin file.
		$plugin_file = $this->get_plugin_file( $plugin_slug );
		if ( ! $plugin_file ) {
			return array(
				'success' => false,
				'message' => 'Plugin not found: ' . $plugin_slug,
			);
		}

		// Log action.
		$this->log_action( 'activate', $plugin_file );

		// Check if already active.
		if ( is_plugin_active( $plugin_file ) ) {
			return array(
				'success' => true,
				'message' => 'Plugin is already active',
				'plugin'  => $plugin_file,
			);
		}

		// Activate plugin.
		$result = activate_plugin( $plugin_file );

		if ( is_wp_error( $result ) ) {
			return array(
				'success' => false,
				'message' => 'Activation failed: ' . $result->get_error_message(),
			);
		}

		return array(
			'success' => true,
			'message' => 'Plugin activated successfully',
			'plugin'  => $plugin_file,
			'warning' => 'EXPERIMENTAL: Plugin activated. Monitor your site for any issues.',
		);
	}

	/**
	 * Deactivate a plugin
	 *
	 * @since 1.0.0
	 * @param string $plugin_slug Plugin slug or file.
	 * @return array Deactivation result.
	 */
	public function deactivate_plugin( $plugin_slug ) {
		if ( ! $this->is_enabled() ) {
			return array(
				'success' => false,
				'message' => 'Plugin management is disabled. Enable it in Respira settings.',
			);
		}

		if ( ! current_user_can( 'activate_plugins' ) ) {
			return array(
				'success' => false,
				'message' => 'You do not have permission to deactivate plugins.',
			);
		}

		// Get plugin file.
		$plugin_file = $this->get_plugin_file( $plugin_slug );
		if ( ! $plugin_file ) {
			return array(
				'success' => false,
				'message' => 'Plugin not found: ' . $plugin_slug,
			);
		}

		// Prevent deactivating Respira itself.
		if ( strpos( $plugin_file, 'respira-for-wordpress' ) !== false ) {
			return array(
				'success' => false,
				'message' => 'Cannot deactivate Respira plugin via API for safety reasons.',
			);
		}

		// Log action.
		$this->log_action( 'deactivate', $plugin_file );

		// Check if already inactive.
		if ( ! is_plugin_active( $plugin_file ) ) {
			return array(
				'success' => true,
				'message' => 'Plugin is already inactive',
				'plugin'  => $plugin_file,
			);
		}

		// Deactivate plugin.
		deactivate_plugins( $plugin_file );

		return array(
			'success' => true,
			'message' => 'Plugin deactivated successfully',
			'plugin'  => $plugin_file,
		);
	}

	/**
	 * Update a plugin
	 *
	 * @since 1.0.0
	 * @param string $plugin_slug Plugin slug or file.
	 * @return array Update result.
	 */
	public function update_plugin( $plugin_slug ) {
		if ( ! $this->is_enabled() ) {
			return array(
				'success' => false,
				'message' => 'Plugin management is disabled. Enable it in Respira settings.',
			);
		}

		if ( ! current_user_can( 'update_plugins' ) ) {
			return array(
				'success' => false,
				'message' => 'You do not have permission to update plugins.',
			);
		}

		// Get plugin file.
		$plugin_file = $this->get_plugin_file( $plugin_slug );
		if ( ! $plugin_file ) {
			return array(
				'success' => false,
				'message' => 'Plugin not found: ' . $plugin_slug,
			);
		}

		// Log action.
		$this->log_action( 'update', $plugin_file );

		// Check for updates.
		$update_plugins = get_site_transient( 'update_plugins' );
		if ( ! isset( $update_plugins->response[ $plugin_file ] ) ) {
			return array(
				'success' => false,
				'message' => 'No update available for this plugin',
				'plugin'  => $plugin_file,
			);
		}

		// Load WordPress plugin upgrader.
		if ( ! class_exists( 'Plugin_Upgrader' ) ) {
			require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
		}
		if ( ! class_exists( 'WP_Ajax_Upgrader_Skin' ) ) {
			require_once ABSPATH . 'wp-admin/includes/class-wp-ajax-upgrader-skin.php';
		}

		$skin     = new WP_Ajax_Upgrader_Skin();
		$upgrader = new Plugin_Upgrader( $skin );

		$result = $upgrader->upgrade( $plugin_file );

		if ( is_wp_error( $result ) ) {
			return array(
				'success' => false,
				'message' => 'Update failed: ' . $result->get_error_message(),
			);
		}

		if ( ! $result ) {
			return array(
				'success' => false,
				'message' => 'Update failed.',
			);
		}

		return array(
			'success' => true,
			'message' => 'Plugin updated successfully',
			'plugin'  => $plugin_file,
			'warning' => 'EXPERIMENTAL: Plugin updated. Monitor your site for any issues.',
		);
	}

	/**
	 * Delete a plugin
	 *
	 * @since 1.0.0
	 * @param string $plugin_slug Plugin slug or file.
	 * @return array Deletion result.
	 */
	public function delete_plugin( $plugin_slug ) {
		if ( ! $this->is_enabled() ) {
			return array(
				'success' => false,
				'message' => 'Plugin management is disabled. Enable it in Respira settings.',
			);
		}

		if ( ! current_user_can( 'delete_plugins' ) ) {
			return array(
				'success' => false,
				'message' => 'You do not have permission to delete plugins.',
			);
		}

		// Get plugin file.
		$plugin_file = $this->get_plugin_file( $plugin_slug );
		if ( ! $plugin_file ) {
			return array(
				'success' => false,
				'message' => 'Plugin not found: ' . $plugin_slug,
			);
		}

		// Prevent deleting Respira itself.
		if ( strpos( $plugin_file, 'respira-for-wordpress' ) !== false ) {
			return array(
				'success' => false,
				'message' => 'Cannot delete Respira plugin via API for safety reasons.',
			);
		}

		// Check if plugin is active.
		if ( is_plugin_active( $plugin_file ) ) {
			return array(
				'success' => false,
				'message' => 'Plugin must be deactivated before deletion',
				'plugin'  => $plugin_file,
			);
		}

		// Log action.
		$this->log_action( 'delete', $plugin_file );

		// Delete plugin.
		$result = delete_plugins( array( $plugin_file ) );

		if ( is_wp_error( $result ) ) {
			return array(
				'success' => false,
				'message' => 'Deletion failed: ' . $result->get_error_message(),
			);
		}

		if ( ! $result ) {
			return array(
				'success' => false,
				'message' => 'Deletion failed.',
			);
		}

		return array(
			'success' => true,
			'message' => 'Plugin deleted successfully',
			'plugin'  => $plugin_file,
			'warning' => 'EXPERIMENTAL: Plugin deleted permanently.',
		);
	}

	/**
	 * Get plugin file from slug
	 *
	 * @param string $plugin_slug Plugin slug or file.
	 * @return string|null Plugin file or null if not found.
	 */
	private function get_plugin_file( $plugin_slug ) {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$all_plugins = get_plugins();

		// Check if it's already a full plugin file.
		if ( isset( $all_plugins[ $plugin_slug ] ) ) {
			return $plugin_slug;
		}

		// Try to find plugin by slug.
		foreach ( $all_plugins as $plugin_file => $plugin_data ) {
			if ( dirname( $plugin_file ) === $plugin_slug ) {
				return $plugin_file;
			}
		}

		return null;
	}

	/**
	 * Log plugin management action
	 *
	 * @param string $action Action performed.
	 * @param string $plugin Plugin file or slug.
	 * @param mixed  $extra Extra data.
	 */
	private function log_action( $action, $plugin, $extra = null ) {
		$log_entry = array(
			'timestamp' => current_time( 'mysql' ),
			'action'    => $action,
			'plugin'    => $plugin,
			'user_id'   => get_current_user_id(),
			'extra'     => $extra,
		);

		// Get existing log.
		$log = get_option( 'respira_plugin_management_log', array() );

		// Add new entry.
		$log[] = $log_entry;

		// Keep only last 100 entries.
		if ( count( $log ) > 100 ) {
			$log = array_slice( $log, -100 );
		}

		// Save log.
		update_option( 'respira_plugin_management_log', $log, false );
	}

	/**
	 * Get plugin management log
	 *
	 * @since 1.0.0
	 * @return array Log entries.
	 */
	public function get_log() {
		if ( ! $this->is_enabled() ) {
			return array(
				'success' => false,
				'message' => 'Plugin management is disabled.',
			);
		}

		$log = get_option( 'respira_plugin_management_log', array() );

		return array(
			'success' => true,
			'log'     => $log,
			'total'   => count( $log ),
		);
	}
}
