<?php
/**
 * PRO License Management
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes
 */

/**
 * PRO License Management class.
 *
 * Handles license activation, validation, and PRO feature access.
 *
 * @since 1.0.0
 */
class Respira_License {

	/**
	 * License API endpoint.
	 *
	 * @var string
	 */
	const LICENSE_SERVER = 'https://respira.press/api/license';

	/**
	 * Check if PRO is active.
	 *
	 * @since 1.0.0
	 * @return bool True if PRO license is active.
	 */
	public static function is_pro_active() {
		$license_key = get_option( 'respira_license_key' );
		$license_status = get_option( 'respira_license_status' );

		if ( empty( $license_key ) || ( 'active' !== $license_status && 'trial' !== $license_status ) ) {
			return false;
		}

		// Check if license needs revalidation (every 7 days).
		$last_check = get_option( 'respira_license_last_check', 0 );
		if ( ( time() - $last_check ) > ( 7 * DAY_IN_SECONDS ) ) {
			$is_valid = self::validate_license( $license_key );
			if ( ! $is_valid ) {
				return false;
			}
		}

		// Always validate site URL matches on every check
		$current_status = get_option( 'respira_license_status' );
		if ( 'active' !== $current_status && 'trial' !== $current_status ) {
			return false;
		}

		return true;
	}

	/**
	 * Activate a license key.
	 *
	 * @since 1.0.0
	 * @param string $license_key The license key to activate.
	 * @return true|WP_Error True on success, WP_Error on failure.
	 */
	public static function activate_license( $license_key ) {
		// Get detected page builders for this site.
		$detected_builders = array();
		
		// Load Respira_Context if available (for builder detection).
		if ( ! class_exists( 'Respira_Context' ) ) {
			require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-context.php';
		}
		
		if ( class_exists( 'Respira_Context' ) ) {
			$builder_info = Respira_Context::get_builder_info();
			if ( isset( $builder_info['builders'] ) && is_array( $builder_info['builders'] ) ) {
				foreach ( $builder_info['builders'] as $builder_name => $builder_data ) {
					if ( isset( $builder_data['detected'] ) && $builder_data['detected'] ) {
						$detected_builders[] = $builder_name;
					}
				}
			}
		}

		$response = wp_remote_post(
			self::LICENSE_SERVER . '/activate',
			array(
				'body'    => json_encode(
					array(
						'license_key'     => $license_key,
						'site_url'        => get_site_url(),
						'detected_builders' => $detected_builders,
					)
				),
				'headers' => array(
					'Content-Type' => 'application/json',
				),
				'timeout' => 15,
			)
		);

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( isset( $data['success'] ) && $data['success'] ) {
			update_option( 'respira_license_key', $license_key );
			update_option( 'respira_license_status', 'active' );
			update_option( 'respira_license_last_check', time() );
			update_option( 'respira_license_data', $data );

			return true;
		}

		return new WP_Error(
			'respira_activation_failed',
			$data['error'] ?? __( 'License activation failed.', 'respira-for-wordpress' )
		);
	}

	/**
	 * Deactivate the current license.
	 *
	 * @since 1.0.0
	 * @return true|WP_Error True on success, WP_Error on failure.
	 */
	public static function deactivate_license() {
		$license_key = get_option( 'respira_license_key' );

		if ( empty( $license_key ) ) {
			return new WP_Error(
				'respira_no_license',
				__( 'No license key found.', 'respira-for-wordpress' )
			);
		}

		$response = wp_remote_post(
			self::LICENSE_SERVER . '/deactivate',
			array(
				'body' => array(
					'license_key' => $license_key,
					'site_url'    => get_site_url(),
				),
			)
		);

		// Clear license data regardless of API response.
		delete_option( 'respira_license_key' );
		delete_option( 'respira_license_status' );
		delete_option( 'respira_license_last_check' );
		delete_option( 'respira_license_data' );

		// Disable all API keys linked to this license.
		self::disable_license_api_keys( $license_key );

		return true;
	}

	/**
	 * Disable all API keys linked to a specific license.
	 *
	 * @since 1.3.0
	 * @param string $license_key The license key.
	 * @return void
	 */
	private static function disable_license_api_keys( $license_key ) {
		global $wpdb;

		if ( empty( $license_key ) ) {
			return;
		}

		$table_name = $wpdb->prefix . 'respira_api_keys';
		$wpdb->update(
			$table_name,
			array( 'is_active' => 0 ),
			array( 'license_key' => $license_key ),
			array( '%d' ),
			array( '%s' )
		);
	}

	/**
	 * Validate an existing license.
	 *
	 * @since 1.0.0
	 * @param string $license_key The license key to validate.
	 * @return bool True if valid, false otherwise.
	 */
	public static function validate_license( $license_key ) {
		$url = add_query_arg(
			array(
				'license_key' => $license_key,
				'site_url'    => get_site_url(),
			),
			self::LICENSE_SERVER . '/validate'
		);

		$response = wp_remote_get(
			$url,
			array(
				'timeout' => 10,
			)
		);

		if ( is_wp_error( $response ) ) {
			return false;
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		// Check if validation failed due to domain mismatch
		if ( isset( $data['domain_mismatch'] ) && $data['domain_mismatch'] ) {
			update_option( 'respira_license_status', 'inactive' );
			update_option( 'respira_license_last_check', time() );
			self::disable_license_api_keys( $license_key );
			
			// Show admin notice about domain mismatch
			add_action( 'admin_notices', function() use ( $data ) {
				?>
				<div class="notice notice-error">
					<p>
						<strong><?php esc_html_e( 'Respira License Error:', 'respira-for-wordpress' ); ?></strong>
						<?php echo esc_html( $data['error'] ?? __( 'This license is not valid for the current website domain.', 'respira-for-wordpress' ) ); ?>
						<a href="<?php echo esc_url( admin_url( 'admin.php?page=respira-license' ) ); ?>" class="button button-primary" style="margin-left: 10px;">
							<?php esc_html_e( 'Update License', 'respira-for-wordpress' ); ?>
						</a>
					</p>
				</div>
				<?php
			});
			
			return false;
		}

		$status = isset( $data['valid'] ) && $data['valid'] ? 'active' : 'inactive';

		update_option( 'respira_license_status', $status );
		update_option( 'respira_license_last_check', time() );

		// If license became inactive, disable all API keys linked to it.
		if ( 'inactive' === $status ) {
			self::disable_license_api_keys( $license_key );
		}

		return 'active' === $status || ( 'trial' === $status && isset( $data['is_trial'] ) && $data['is_trial'] );
	}

	/**
	 * Check if a specific PRO feature is available.
	 *
	 * @since 1.0.0
	 * @param string $feature Feature name.
	 * @return bool True if feature is available.
	 */
	public static function has_feature( $feature ) {
		if ( ! self::is_pro_active() ) {
			return false;
		}

		$pro_features = array(
			'all_page_builders',    // Elementor, Oxygen, Bricks, etc.
			'all_ai_tools',         // Codex, Windsurf, Cline, etc.
			'multi_site',           // Up to 20 WordPress sites (Agency plan).
			'visual_preview',       // AI sees changes.
			'version_history',      // Rollback functionality.
			'git_integration',      // Auto-commit to repos.
			'priority_support',     // Email + AI chat support.
		);

		return in_array( $feature, $pro_features, true );
	}

	/**
	 * Get license information.
	 *
	 * @since 1.0.0
	 * @return array License information.
	 */
	public static function get_license_info() {
		return array(
			'key'        => get_option( 'respira_license_key', '' ),
			'status'     => get_option( 'respira_license_status', 'inactive' ),
			'last_check' => get_option( 'respira_license_last_check', 0 ),
			'data'       => get_option( 'respira_license_data', array() ),
			'is_pro'     => self::is_pro_active(),
		);
	}

	/**
	 * Display PRO upgrade notice.
	 *
	 * @since 1.0.0
	 * @param string $feature Feature name.
	 * @return string HTML upgrade notice.
	 */
	public static function upgrade_notice( $feature = '' ) {
		ob_start();
		?>
		<div class="notice notice-info respira-pro-notice">
			<h3><?php esc_html_e( 'Respira PRO Required', 'respira-for-wordpress' ); ?></h3>
			<p>
				<?php
				if ( $feature ) {
					echo wp_kses_post(
						sprintf(
							/* translators: %s: feature name */
							__( 'The <strong>%s</strong> feature requires Respira PRO.', 'respira-for-wordpress' ),
							$feature
						)
					);
				} else {
					esc_html_e( 'This feature requires Respira PRO.', 'respira-for-wordpress' );
				}
				?>
			</p>
			<p>
				<a href="https://respira.press/dashboard/billing" class="button button-primary" target="_blank">
					<?php esc_html_e( 'Upgrade Account →', 'respira-for-wordpress' ); ?>
				</a>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=respira-license' ) ); ?>" class="button">
					<?php esc_html_e( 'Enter License Key', 'respira-for-wordpress' ); ?>
				</a>
			</p>
		</div>
		<?php
		return ob_get_clean();
	}
}
