<?php
/**
 * The core plugin class.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * @since 1.0.0
 */
class Respira_Core {

	/**
	 * The loader that's responsible for maintaining and registering all hooks.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    Respira_Loader $loader Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    string $plugin_name The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    string $version The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->version     = RESPIRA_VERSION;
		$this->plugin_name = 'respira-for-wordpress';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_api_hooks();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * @since  1.0.0
	 * @access private
	 */
	private function load_dependencies() {
		/**
		 * The class responsible for orchestrating the actions and filters.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-loader.php';

		/**
		 * The class responsible for defining internationalization functionality.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-i18n.php';

		/**
		 * The class responsible for authentication.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-auth.php';

		/**
		 * The class responsible for license management.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-license.php';

		/**
		 * The class responsible for REST API endpoints.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-api.php';

		/**
		 * The class responsible for security validation.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-security.php';

		/**
		 * The class responsible for site context extraction.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-context.php';

		/**
		 * The class responsible for theme detection.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-theme-detector.php';

		/**
		 * The class responsible for page duplication.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-duplicator.php';

		/**
		 * The class responsible for page/post approval.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-approver.php';

		/**
		 * The class responsible for content filtering and script preservation.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-content-filter.php';

		/**
		 * The class responsible for page builder abstraction.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/class-builder-interface.php';

		/**
		 * The class responsible for defining all actions in the admin area.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'admin/class-respira-admin.php';

		/**
		 * The class responsible for plugin updates from GitHub.
		 */
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-updater.php';

		$this->loader = new Respira_Loader();

		// Initialize content filter for script preservation.
		Respira_Content_Filter::init();

		// Initialize updater.
		new Respira_Updater();
	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * @since  1.0.0
	 * @access private
	 */
	private function set_locale() {
		$plugin_i18n = new Respira_i18n();
		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );
	}

	/**
	 * Register all of the hooks related to the admin area functionality.
	 *
	 * @since  1.0.0
	 * @access private
	 */
	private function define_admin_hooks() {
		$plugin_admin = new Respira_Admin( $this->get_plugin_name(), $this->get_version() );

		// Register AJAX handlers early on admin_init.
		$this->loader->add_action( 'admin_init', $plugin_admin, 'register_ajax_handlers' );
		
		// Enqueue styles with high priority (999) to load after theme styles
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles', 999 );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts', 999 );
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'add_plugin_admin_menu' );
	}

	/**
	 * Register all of the hooks related to the REST API functionality.
	 *
	 * @since  1.0.0
	 * @access private
	 */
	private function define_api_hooks() {
		$plugin_api = new Respira_API();

		$this->loader->add_action( 'rest_api_init', $plugin_api, 'register_routes' );
		$this->loader->add_action( 'wp_head', $plugin_api, 'output_page_custom_css', 100 );
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since 1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it.
	 *
	 * @since  1.0.0
	 * @return string The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks.
	 *
	 * @since  1.0.0
	 * @return Respira_Loader Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since  1.0.0
	 * @return string The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}
}
