<?php
/**
 * REST API endpoints.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes
 */

/**
 * REST API endpoints.
 *
 * Registers and handles all REST API endpoints for the plugin.
 *
 * @since 1.0.0
 */
class Respira_API {

	/**
	 * Format Respira-branded message.
	 *
	 * @since 1.9.0
	 * @param string $message The message to format.
	 * @param string $type Optional. Message type: 'success', 'error', 'info', 'warning'. Default 'info'.
	 * @return string The formatted message with "Respira says:" prefix.
	 */
	private function format_respira_message( $message, $type = 'info' ) {
		$prefix = __( 'Respira says:', 'respira-for-wordpress' );
		
		// Add emoji based on type
		$emoji = '';
		switch ( $type ) {
			case 'success':
				$emoji = '✅';
				break;
			case 'error':
				$emoji = '⚠️';
				break;
			case 'warning':
				$emoji = '🛡️';
				break;
			case 'info':
			default:
				$emoji = '💬';
				break;
		}
		
		return sprintf( '%s %s %s', $prefix, $emoji, $message );
	}

	/**
	 * Create a Respira-branded WP_Error.
	 *
	 * @since 1.9.0
	 * @param string $code Error code.
	 * @param string $message Error message (will be formatted with Respira branding).
	 * @param array  $data Optional. Error data.
	 * @param string $type Optional. Message type: 'error', 'warning', 'info'. Default 'error'.
	 * @return WP_Error The formatted error.
	 */
	private function respira_error( $code, $message, $data = array(), $type = 'error' ) {
		$formatted_message = $this->format_respira_message( $message, $type );
		$data['respira_message'] = $formatted_message;
		
		// Add helpful hints for common errors.
		if ( 'respira_page_not_found' === $code || 'respira_post_not_found' === $code ) {
			$data['hint'] = $this->format_respira_message( __( 'Double-check the ID? Use GET /pages or GET /posts to list available items.', 'respira-for-wordpress' ), 'info' );
		}
		
		return new WP_Error( $code, $formatted_message, $data );
	}

	/**
	 * Add footer to success messages.
	 *
	 * @since 1.8.0
	 * @param string $message The original success message.
	 * @return string The message with footer appended.
	 */
	private function add_success_footer( $message ) {
		$footer = __( 'Thank you for using Respira for WordPress | respira.press', 'respira-for-wordpress' );
		return $message . "\n\n" . $footer;
	}

	/**
	 * Register REST API routes.
	 *
	 * @since 1.0.0
	 */
	public function register_routes() {
		// Authentication endpoints.
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/auth/generate-key',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'generate_api_key' ),
				'permission_callback' => array( $this, 'admin_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/auth/validate-key',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'validate_api_key' ),
				'permission_callback' => '__return_true',
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/auth/revoke-key/(?P<key_id>\d+)',
			array(
				'methods'             => 'DELETE',
				'callback'            => array( $this, 'revoke_api_key' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		// Site context endpoints.
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/context/site-info',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_site_info' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/context/theme-docs',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_theme_docs' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/context/builder-info',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_builder_info' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		// Page endpoints.
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/pages',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'list_pages' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
				'args'                => $this->get_list_args(),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/pages/(?P<id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_page' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/pages/(?P<id>\d+)/duplicate',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'duplicate_page' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
				'args'                => array(
					'suffix' => array(
						'required'          => false,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
				),
			)
		);

		// Register PUT method for page updates
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/pages/(?P<id>\d+)',
			array(
				'methods'             => 'PUT',
				'callback'            => array( $this, 'update_page' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
				'args'                => array(
					'title'              => array(
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'content'            => array(
						'type' => 'string',
					),
					'status'             => array(
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'custom_css'         => array(
						'type'              => 'string',
						'sanitize_callback' => array( $this, 'sanitize_custom_css' ),
						'description'       => 'Custom CSS for the page (stored in Divi _et_pb_custom_css meta field)',
					),
					'seo'                => array(
						'type'        => 'object',
						'description' => 'SEO meta tags (title, description, OG tags, Twitter cards, etc.)',
					),
					'featured_image'     => array(
						'type'        => 'object',
						'description' => 'Featured image data (url or id, plus alt and title)',
					),
					'meta'               => array(
						'type' => 'object',
					),
					'force'              => array(
						'type'    => 'boolean',
						'default' => false,
					),
					'skip_security_check' => array(
						'type'    => 'boolean',
						'default' => false,
					),
				),
			)
		);

		// Register POST method for page updates (compatibility with WordPress REST API)
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/pages/(?P<id>\d+)',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_page' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
				'args'                => array(
					'title'              => array(
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'content'            => array(
						'type' => 'string',
					),
					'status'             => array(
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'custom_css'         => array(
						'type'              => 'string',
						'sanitize_callback' => array( $this, 'sanitize_custom_css' ),
						'description'       => 'Custom CSS for the page (stored in Divi _et_pb_custom_css meta field)',
					),
					'seo'                => array(
						'type'        => 'object',
						'description' => 'SEO meta tags (title, description, OG tags, Twitter cards, etc.)',
					),
					'featured_image'     => array(
						'type'        => 'object',
						'description' => 'Featured image data (url or id, plus alt and title)',
					),
					'meta'               => array(
						'type' => 'object',
					),
					'force'              => array(
						'type'    => 'boolean',
						'default' => false,
					),
					'skip_security_check' => array(
						'type'    => 'boolean',
						'default' => false,
					),
				),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/pages/(?P<id>\d+)',
			array(
				'methods'             => 'DELETE',
				'callback'            => array( $this, 'delete_page' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		// Post endpoints (similar to pages).
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/posts',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'list_posts' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
				'args'                => $this->get_list_args(),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/posts/(?P<id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_post' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/posts/(?P<id>\d+)/duplicate',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'duplicate_post' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		// Register PUT method for post updates
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/posts/(?P<id>\d+)',
			array(
				'methods'             => 'PUT',
				'callback'            => array( $this, 'update_post' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
				'args'                => array(
					'title'              => array(
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'content'            => array(
						'type' => 'string',
					),
					'status'             => array(
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'seo'                => array(
						'type'        => 'object',
						'description' => 'SEO meta tags (title, description, OG tags, Twitter cards, etc.)',
					),
					'featured_image'     => array(
						'type'        => 'object',
						'description' => 'Featured image data (url or id, plus alt and title)',
					),
					'meta'               => array(
						'type' => 'object',
					),
					'force'              => array(
						'type'    => 'boolean',
						'default' => false,
					),
					'skip_security_check' => array(
						'type'    => 'boolean',
						'default' => false,
					),
				),
			)
		);

		// Register POST method for post updates (compatibility with WordPress REST API)
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/posts/(?P<id>\d+)',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_post' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
				'args'                => array(
					'title'              => array(
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'content'            => array(
						'type' => 'string',
					),
					'status'             => array(
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'seo'                => array(
						'type'        => 'object',
						'description' => 'SEO meta tags (title, description, OG tags, Twitter cards, etc.)',
					),
					'featured_image'     => array(
						'type'        => 'object',
						'description' => 'Featured image data (url or id, plus alt and title)',
					),
					'meta'               => array(
						'type' => 'object',
					),
					'force'              => array(
						'type'    => 'boolean',
						'default' => false,
					),
					'skip_security_check' => array(
						'type'    => 'boolean',
						'default' => false,
					),
				),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/posts/(?P<id>\d+)',
			array(
				'methods'             => 'DELETE',
				'callback'            => array( $this, 'delete_post' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		// Media endpoints.
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/media',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'list_media' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
				'args'                => $this->get_list_args(),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/media/upload',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'upload_media' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		// Page builder endpoints.
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/builder/(?P<builder>[a-z_]+)/extract/(?P<page_id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'extract_builder_content' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/builder/(?P<builder>[a-z_]+)/inject/(?P<page_id>\d+)',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'inject_builder_content' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		// Validation endpoints.
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/validate/security',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'validate_security' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		// Audit log endpoint.
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/audit/logs',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_audit_logs' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
				'args'                => $this->get_list_args(),
			)
		);

		// Page Speed Analysis endpoints.
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/analyze/performance/(?P<page_id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'analyze_performance' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/analyze/core-web-vitals/(?P<page_id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_core_web_vitals' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/analyze/images/(?P<page_id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'analyze_images' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		// SEO Analysis endpoints.
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/analyze/seo/(?P<page_id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'analyze_seo' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/analyze/seo-issues/(?P<page_id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'check_seo_issues' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/analyze/readability/(?P<page_id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'analyze_readability' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		// AEO Analysis endpoints.
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/analyze/aeo/(?P<page_id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'analyze_aeo' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/analyze/structured-data/(?P<page_id>\d+)',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'check_structured_data' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		// Plugin Management endpoints (Experimental).
		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/plugins',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'list_plugins' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/plugins/install',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'install_plugin' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
				'args'                => array(
					'slug_or_url' => array(
						'required'          => true,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'source'      => array(
						'required' => false,
						'type'     => 'string',
						'default'  => 'wordpress.org',
						'enum'     => array( 'wordpress.org', 'url' ),
					),
				),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/plugins/(?P<slug>[a-zA-Z0-9_-]+)/activate',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'activate_plugin' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/plugins/(?P<slug>[a-zA-Z0-9_-]+)/deactivate',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'deactivate_plugin' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/plugins/(?P<slug>[a-zA-Z0-9_-]+)/update',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_plugin' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

		register_rest_route(
			RESPIRA_REST_NAMESPACE,
			'/plugins/(?P<slug>[a-zA-Z0-9_-]+)',
			array(
				'methods'             => 'DELETE',
				'callback'            => array( $this, 'delete_plugin' ),
				'permission_callback' => array( $this, 'api_key_permission_check' ),
			)
		);

	}

	/**
	 * Permission callback for admin users.
	 *
	 * @since 1.0.0
	 * @return bool True if user can manage options.
	 */
	public function admin_permission_check() {
		return current_user_can( 'manage_options' );
	}

	/**
	 * Permission callback that validates API key from header.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return bool|WP_Error True if valid, WP_Error otherwise.
	 */
	public function api_key_permission_check( $request ) {
		$api_key = $request->get_header( 'X-Respira-API-Key' );

		// Also check Authorization header (Bearer token format).
		if ( empty( $api_key ) ) {
			$auth_header = $request->get_header( 'Authorization' );
			if ( ! empty( $auth_header ) && preg_match( '/Bearer\s+(.*)$/i', $auth_header, $matches ) ) {
				$api_key = $matches[1];
			}
		}

		$validation = Respira_Auth::validate_api_key( $api_key );

		if ( is_wp_error( $validation ) ) {
			return $validation;
		}

		// Store key data in request for later use.
		$request->set_param( '_respira_key_data', $validation );

		return true;
	}

	/**
	 * Generate API key endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function generate_api_key( $request ) {
		$name = $request->get_param( 'name' );
		$api_key = Respira_Auth::generate_api_key( get_current_user_id(), $name );

		if ( is_wp_error( $api_key ) ) {
			return $api_key;
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'api_key' => $api_key,
				'message' => $this->add_success_footer( $this->format_respira_message( __( 'API key generated successfully! Save this key securely - it won\'t be shown again.', 'respira-for-wordpress' ), 'success' ) ),
			),
			201
		);
	}

	/**
	 * Validate API key endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function validate_api_key( $request ) {
		$api_key    = $request->get_param( 'api_key' );
		$validation = Respira_Auth::validate_api_key( $api_key );

		if ( is_wp_error( $validation ) ) {
			return $validation;
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'valid'   => true,
				'message' => __( 'API key is valid.', 'respira-for-wordpress' ),
			),
			200
		);
	}

	/**
	 * Revoke API key endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function revoke_api_key( $request ) {
		$key_id = $request->get_param( 'key_id' );
		$result = Respira_Auth::revoke_api_key( $key_id );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

			return new WP_REST_Response(
				array(
					'success' => true,
					'message' => $this->add_success_footer( $this->format_respira_message( __( 'API key revoked successfully. That key is now inactive.', 'respira-for-wordpress' ), 'success' ) ),
				),
				200
			);
	}

	/**
	 * Get site information endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response Response object.
	 */
	public function get_site_info( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'get_site_info', 'context', null );

		$context = Respira_Context::get_site_info();

		return new WP_REST_Response( $context, 200 );
	}

	/**
	 * Get theme documentation endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response Response object.
	 */
	public function get_theme_docs( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'get_theme_docs', 'context', null );

		$docs = Respira_Context::get_theme_docs();

		return new WP_REST_Response( $docs, 200 );
	}

	/**
	 * Get page builder information endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response Response object.
	 */
	public function get_builder_info( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'get_builder_info', 'context', null );

		$builder_info = Respira_Context::get_builder_info();

		return new WP_REST_Response( $builder_info, 200 );
	}

	/**
	 * List pages endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response Response object.
	 */
	public function list_pages( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );

		$args = array(
			'post_type'      => 'page',
			'post_status'    => $request->get_param( 'status' ) ?? array( 'publish', 'draft', 'pending' ),
			'posts_per_page' => $request->get_param( 'per_page' ) ?? 20,
			'paged'          => $request->get_param( 'page' ) ?? 1,
			's'              => $request->get_param( 'search' ) ?? '',
		);

		$query = new WP_Query( $args );
		$pages = array();

		foreach ( $query->posts as $post ) {
			$pages[] = $this->format_post_response( $post );
		}

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'list_pages', 'page', null );

		return new WP_REST_Response(
			array(
				'pages'       => $pages,
				'total'       => $query->found_posts,
				'total_pages' => $query->max_num_pages,
			),
			200
		);
	}

	/**
	 * Get single page endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function get_page( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		$page_id  = $request->get_param( 'id' );

		$post = get_post( $page_id );

		if ( ! $post || 'page' !== $post->post_type ) {
			return $this->respira_error(
				'respira_page_not_found',
				__( 'Oops! That page doesn\'t exist. Double-check the ID?', 'respira-for-wordpress' ),
				array( 'status' => 404 ),
				'error'
			);
		}

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'get_page', 'page', $page_id );

		return new WP_REST_Response( $this->format_post_response( $post, true ), 200 );
	}

	/**
	 * Duplicate page endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function duplicate_page( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		$page_id  = $request->get_param( 'id' );
		$suffix   = $request->get_param( 'suffix' ) ?? '_duplicate_' . time();

		$duplicate_id = Respira_Duplicator::duplicate_post( $page_id, $suffix );

		if ( is_wp_error( $duplicate_id ) ) {
			return $duplicate_id;
		}

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'duplicate_page', 'page', $duplicate_id );

		$duplicate_post = get_post( $duplicate_id );

		return new WP_REST_Response(
			array(
				'success'   => true,
				'duplicate' => $this->format_post_response( $duplicate_post, true ),
				'message'   => $this->add_success_footer( $this->format_respira_message( __( 'Page duplicated successfully! You can now edit the duplicate safely.', 'respira-for-wordpress' ), 'success' ) ),
			),
			201
		);
	}

	/**
	 * Update page endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function update_page( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		$page_id  = $request->get_param( 'id' );

		// Security check: ensure this is a duplicate page.
		$post = get_post( $page_id );
		if ( ! $post ) {
			return new WP_Error(
				'respira_page_not_found',
				__( 'Page not found.', 'respira-for-wordpress' ),
				array( 'status' => 404 )
			);
		}

		// Check if page was created by Respira (has duplicate marker in meta).
		$is_duplicate = get_post_meta( $page_id, '_respira_duplicate', true );
		$force        = $request->get_param( 'force' );
		$allow_direct_edit = get_option( 'respira_allow_direct_edit', 0 );
		$post_type_label = 'page' === $post->post_type ? __( 'page', 'respira-for-wordpress' ) : __( 'post', 'respira-for-wordpress' );
		$auto_duplicate_created = false;
		$original_id = $page_id;

		// If not a duplicate, ALWAYS create a duplicate first (safety requirement).
		if ( ! $is_duplicate ) {
			// Check if a duplicate already exists.
			$existing_duplicates = Respira_Duplicator::get_duplicates( $page_id );
			
			if ( ! empty( $existing_duplicates ) ) {
				// Use the first existing duplicate.
				$duplicate_id = $existing_duplicates[0]->ID;
				$error_data = Respira_Approver::get_approval_error_data( $page_id );
				return new WP_Error(
					'respira_duplicate_exists',
					sprintf(
						/* translators: 1: post type label, 2: page ID, 3: duplicate ID */
						__( 'Cannot edit original %1$s (ID: %2$d). A duplicate already exists (ID: %3$d). Please edit the duplicate instead.', 'respira-for-wordpress' ),
						$post_type_label,
						$page_id,
						$duplicate_id
					),
					array_merge( $error_data, array(
						'duplicate_id' => $duplicate_id,
						'instructions' => array(
							sprintf(
								/* translators: 1: duplicate ID */
								__( 'Edit the duplicate %1$s (ID: %2$d) instead of the original.', 'respira-for-wordpress' ),
								$post_type_label,
								$duplicate_id
							),
							__( 'After editing, approve the duplicate in WordPress admin to replace the original.', 'respira-for-wordpress' ),
						),
					) )
				);
			}

			// If force=true is requested, require 3 confirmations before allowing direct edit.
			if ( $force && $allow_direct_edit ) {
				$confirmation_key = 'respira_force_confirm_' . $page_id . '_' . $key_data['id'];
				$confirmations = get_transient( $confirmation_key );
				
				if ( false === $confirmations ) {
					$confirmations = 0;
				}
				
				$confirmations++;
				set_transient( $confirmation_key, $confirmations, 300 ); // 5 minute window
				
				if ( $confirmations < 3 ) {
					$remaining = 3 - $confirmations;
					$error_data = Respira_Approver::get_approval_error_data( $page_id );
					return new WP_Error(
						'respira_force_confirmation_required',
						sprintf(
							/* translators: 1: number of remaining confirmations, 2: post type label, 3: page ID */
							_n(
								'⚠️ SAFETY CHECK: You must confirm %1$d more time(s) before directly editing live %2$s (ID: %3$d). This is a LIVE page that visitors can see. Are you absolutely sure you want to edit it directly without creating a duplicate first?',
								'⚠️ SAFETY CHECK: You must confirm %1$d more time(s) before directly editing live %2$s (ID: %3$d). This is a LIVE page that visitors can see. Are you absolutely sure you want to edit it directly without creating a duplicate first?',
								$remaining,
								'respira-for-wordpress'
							),
							$remaining,
							$post_type_label,
							$page_id
						),
						array_merge( $error_data, array(
							'confirmations_required' => 3,
							'confirmations_received' => $confirmations,
							'confirmations_remaining' => $remaining,
							'instructions' => array(
								__( '⚠️ WARNING: You are attempting to edit a LIVE page/post directly.', 'respira-for-wordpress' ),
								__( 'For safety, Respira requires 3 separate confirmations before allowing direct edits to live content.', 'respira-for-wordpress' ),
								sprintf(
									/* translators: 1: number of remaining confirmations */
									_n(
										'Please confirm %1$d more time(s) by calling this endpoint again with force=true.',
										'Please confirm %1$d more time(s) by calling this endpoint again with force=true.',
										$remaining,
										'respira-for-wordpress'
									),
									$remaining
								),
								__( 'RECOMMENDED: Create a duplicate first using wordpress_create_page_duplicate, then edit the duplicate.', 'respira-for-wordpress' ),
							),
						) )
					);
				}
				
				// All 3 confirmations received - allow direct edit but log warning.
				Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'force_edit_page', 'page', $page_id );
				delete_transient( $confirmation_key );
			} else {
				// No force or direct editing disabled - automatically create duplicate.
				$suffix = '_duplicate_' . time();
				$duplicate_id = Respira_Duplicator::duplicate_post( $page_id, $suffix );
				
				if ( is_wp_error( $duplicate_id ) ) {
					return $duplicate_id;
				}
				
				// Log the automatic duplication.
				Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'auto_duplicate_page', 'page', $duplicate_id );
				
				// Switch to editing the duplicate instead of the original.
				$page_id = $duplicate_id;
				$post = get_post( $page_id );
				$is_duplicate = true;
				$auto_duplicate_created = true;
				
				// Continue with the update below - we'll add a note in the response.
			}
		}

		// Validate content if security checks are enabled and not skipped.
		$skip_security = $request->get_param( 'skip_security_check' );
		if ( get_option( 'respira_security_checks', 1 ) && ! $skip_security ) {
			$content = $request->get_param( 'content' );
			if ( $content ) {
				$security_check = Respira_Security::validate_content( $content );
				if ( is_wp_error( $security_check ) ) {
					return $security_check;
				}
			}
		}

		// Update the post.
		$update_data = array(
			'ID' => $page_id,
		);

		if ( $request->get_param( 'title' ) ) {
			$update_data['post_title'] = sanitize_text_field( $request->get_param( 'title' ) );
		}

		if ( $request->get_param( 'content' ) ) {
			// Use API-specific sanitization that allows scripts and modern web features.
			$update_data['post_content'] = $this->sanitize_api_content( $request->get_param( 'content' ) );
		}

		if ( $request->get_param( 'status' ) ) {
			$update_data['post_status'] = sanitize_text_field( $request->get_param( 'status' ) );
		}

		// Enable API update context to preserve script tags.
		// This tells our content filter to preserve scripts in Divi modules.
		Respira_Content_Filter::set_api_update_context( true );

		// Temporarily remove ALL WordPress content sanitization filters.
		// We've already sanitized content with our custom allowed HTML in sanitize_api_content().
		// This is safe because:
		// 1. Request is authenticated via API key
		// 2. Content was validated by Respira_Security::validate_content() above
		// 3. We use wp_kses() with a custom whitelist in sanitize_api_content()
		$removed_filters = Respira_Content_Filter::remove_content_filters();

		// Update the post with preserved content.
		$result = wp_update_post( $update_data, true );

		// Restore WordPress filters.
		Respira_Content_Filter::restore_content_filters( $removed_filters );

		// Disable API update context.
		Respira_Content_Filter::set_api_update_context( false );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		// Update meta if provided.
		if ( $request->get_param( 'meta' ) ) {
			$meta = $request->get_param( 'meta' );
			foreach ( $meta as $key => $value ) {
				update_post_meta( $page_id, sanitize_key( $key ), $value );
			}
		}

		// Update custom CSS if provided (Divi Builder support).
		if ( $request->has_param( 'custom_css' ) ) {
			$custom_css = $request->get_param( 'custom_css' );

			// If empty string is provided, delete the custom CSS.
			if ( '' === $custom_css ) {
				delete_post_meta( $page_id, '_et_pb_custom_css' );
			} else {
				// The CSS has already been sanitized by the sanitize_callback.
				update_post_meta( $page_id, '_et_pb_custom_css', $custom_css );
			}
		}

		// Update SEO meta if provided.
		if ( $request->has_param( 'seo' ) ) {
			$seo_data = $request->get_param( 'seo' );
			if ( is_array( $seo_data ) && ! empty( $seo_data ) ) {
				$this->update_seo_meta( $page_id, $seo_data );
			}
		}

		// Update featured image if provided.
		if ( $request->has_param( 'featured_image' ) ) {
			$image_data = $request->get_param( 'featured_image' );
			if ( is_array( $image_data ) && ! empty( $image_data ) ) {
				$this->update_featured_image( $page_id, $image_data );
			}
		}

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'update_page', 'page', $page_id );

		// Track usage stat (non-blocking).
		// PRIVACY: Only aggregate statistics are sent - actual content is NEVER transmitted.
		// See class-respira-usage-tracker.php for detailed privacy documentation.
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-usage-tracker.php';
		Respira_Usage_Tracker::track( 'update_page', 'page', $page_id, $content );

		$updated_post = get_post( $page_id );
		$approvals_url = admin_url( 'admin.php?page=respira-approvals' );
		
		$response_data = array(
			'success' => true,
			'page'    => $this->format_post_response( $updated_post, true ),
			'message' => $this->add_success_footer( $this->format_respira_message( __( 'Page updated successfully! Your changes are safe and sound.', 'respira-for-wordpress' ), 'success' ) ),
		);
		
		// Add note if duplicate was auto-created.
		if ( $auto_duplicate_created ) {
			$response_data['duplicate_created'] = true;
			$response_data['duplicate_id'] = $page_id;
			$response_data['original_id'] = $original_id;
			$response_data['approval_url'] = $approvals_url;
			$response_data['message'] = $this->add_success_footer( $this->format_respira_message( sprintf(
				/* translators: 1: post type label, 2: duplicate ID */
				__( 'I created a duplicate %1$s for safety (ID: %2$d). Review it in WordPress admin before approving to make it live.', 'respira-for-wordpress' ),
				$post_type_label,
				$page_id
			), 'warning' ) );
			$response_data['instructions'] = array(
				$this->format_respira_message( __( 'A duplicate was automatically created for safety.', 'respira-for-wordpress' ), 'warning' ),
				sprintf(
					/* translators: 1: post type label, 2: duplicate ID, 3: original ID */
					__( '📋 Duplicate %1$s ID: %2$d | Original ID: %3$d', 'respira-for-wordpress' ),
					$post_type_label,
					$page_id,
					$original_id
				),
				sprintf(
					/* translators: 1: approvals URL */
					__( '👀 Review and approve: %1$s', 'respira-for-wordpress' ),
					$approvals_url
				),
				__( '💡 Tip: Changes are on the duplicate. Approve it to make them live.', 'respira-for-wordpress' ),
			);
		}

		$response = new WP_REST_Response( $response_data, 200 );
		
		// Add response headers for better debugging.
		if ( $auto_duplicate_created ) {
			$response->header( 'X-Respira-Duplicate-Created', 'true' );
			$response->header( 'X-Respira-Duplicate-ID', (string) $page_id );
			$response->header( 'X-Respira-Original-ID', (string) $original_id );
			$response->header( 'X-Respira-Approval-URL', $approvals_url );
		} else {
			$response->header( 'X-Respira-Duplicate-Created', 'false' );
		}

		return $response;
	}

	/**
	 * Delete page endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function delete_page( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		$page_id  = $request->get_param( 'id' );

		// Only allow deletion of Respira duplicates unless forced.
		$is_duplicate = get_post_meta( $page_id, '_respira_duplicate', true );
		$force        = $request->get_param( 'force' );
		$allow_direct_edit = get_option( 'respira_allow_direct_edit', 0 );

		// If not a duplicate, check if we can delete it.
		if ( ! $is_duplicate ) {
			// If force is requested but direct editing is disabled, return error with instructions.
			if ( $force && ! $allow_direct_edit ) {
				$error_data = Respira_Approver::get_approval_error_data( $page_id );
				$post = get_post( $page_id );
				return new WP_Error(
					'respira_direct_edit_disabled',
					sprintf(
						/* translators: 1: post type label */
						__( 'Cannot delete original %1$s. Direct editing of original pages is disabled in settings. Only Respira duplicates can be deleted.', 'respira-for-wordpress' ),
						$post && 'page' === $post->post_type ? __( 'page', 'respira-for-wordpress' ) : __( 'post', 'respira-for-wordpress' )
					),
					$error_data
				);
			}

			// If force is not requested, return error with instructions.
			if ( ! $force ) {
				$error_data = Respira_Approver::get_approval_error_data( $page_id );
				$post = get_post( $page_id );
				$message = sprintf(
					/* translators: 1: post type label, 2: page ID */
					__( 'Cannot delete original %1$s (ID: %2$d). Only Respira duplicates can be deleted.', 'respira-for-wordpress' ),
					$post && 'page' === $post->post_type ? __( 'page', 'respira-for-wordpress' ) : __( 'post', 'respira-for-wordpress' ),
					$page_id
				);

				return new WP_Error(
					'respira_not_duplicate',
					$message,
					$error_data
				);
			}
		}

		$result = wp_delete_post( $page_id, true );

		if ( ! $result ) {
			return new WP_Error(
				'respira_delete_failed',
				__( 'Failed to delete page.', 'respira-for-wordpress' ),
				array( 'status' => 500 )
			);
		}

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'delete_page', 'page', $page_id );

		return new WP_REST_Response(
			array(
				'success' => true,
				'message' => $this->add_success_footer( $this->format_respira_message( __( 'Page deleted successfully. It\'s gone for good!', 'respira-for-wordpress' ), 'success' ) ),
			),
			200
		);
	}

	/**
	 * List posts endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response Response object.
	 */
	public function list_posts( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );

		$args = array(
			'post_type'      => 'post',
			'post_status'    => $request->get_param( 'status' ) ?? array( 'publish', 'draft', 'pending' ),
			'posts_per_page' => $request->get_param( 'per_page' ) ?? 20,
			'paged'          => $request->get_param( 'page' ) ?? 1,
			's'              => $request->get_param( 'search' ) ?? '',
		);

		$query = new WP_Query( $args );
		$posts = array();

		foreach ( $query->posts as $post ) {
			$posts[] = $this->format_post_response( $post );
		}

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'list_posts', 'post', null );

		return new WP_REST_Response(
			array(
				'posts'       => $posts,
				'total'       => $query->found_posts,
				'total_pages' => $query->max_num_pages,
			),
			200
		);
	}

	/**
	 * Get single post endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function get_post( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		$post_id  = $request->get_param( 'id' );

		$post = get_post( $post_id );

		if ( ! $post || 'post' !== $post->post_type ) {
			return new WP_Error(
				'respira_post_not_found',
				__( 'Post not found.', 'respira-for-wordpress' ),
				array( 'status' => 404 )
			);
		}

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'get_post', 'post', $post_id );

		return new WP_REST_Response( $this->format_post_response( $post, true ), 200 );
	}

	/**
	 * Duplicate post endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function duplicate_post( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		$post_id  = $request->get_param( 'id' );
		$suffix   = $request->get_param( 'suffix' ) ?? '_duplicate_' . time();

		$duplicate_id = Respira_Duplicator::duplicate_post( $post_id, $suffix );

		if ( is_wp_error( $duplicate_id ) ) {
			return $duplicate_id;
		}

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'duplicate_post', 'post', $duplicate_id );

		$duplicate_post = get_post( $duplicate_id );

		return new WP_REST_Response(
			array(
				'success'   => true,
				'duplicate' => $this->format_post_response( $duplicate_post, true ),
				'message'   => $this->add_success_footer( $this->format_respira_message( __( 'Post duplicated successfully! You can now edit the duplicate safely.', 'respira-for-wordpress' ), 'success' ) ),
			),
			201
		);
	}

	/**
	 * Update post endpoint.
	 *
	 * @since 1.2.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function update_post( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		$post_id  = $request->get_param( 'id' );

		// Security check: ensure this is a duplicate post.
		$post = get_post( $post_id );
		if ( ! $post || 'post' !== $post->post_type ) {
			return new WP_Error(
				'respira_post_not_found',
				__( 'Post not found.', 'respira-for-wordpress' ),
				array( 'status' => 404 )
			);
		}

		// Check if post was created by Respira (has duplicate marker in meta).
		$is_duplicate = get_post_meta( $post_id, '_respira_duplicate', true );
		$force        = $request->get_param( 'force' );
		$allow_direct_edit = get_option( 'respira_allow_direct_edit', 0 );
		$post_type_label = __( 'post', 'respira-for-wordpress' );
		$auto_duplicate_created = false;
		$original_id = $post_id;

		// If not a duplicate, ALWAYS create a duplicate first (safety requirement).
		if ( ! $is_duplicate ) {
			// Check if a duplicate already exists.
			$existing_duplicates = Respira_Duplicator::get_duplicates( $post_id );
			
			if ( ! empty( $existing_duplicates ) ) {
				// Use the first existing duplicate.
				$duplicate_id = $existing_duplicates[0]->ID;
				$error_data = Respira_Approver::get_approval_error_data( $post_id );
				return new WP_Error(
					'respira_duplicate_exists',
					sprintf(
						/* translators: 1: post type label, 2: post ID, 3: duplicate ID */
						__( 'Cannot edit original %1$s (ID: %2$d). A duplicate already exists (ID: %3$d). Please edit the duplicate instead.', 'respira-for-wordpress' ),
						$post_type_label,
						$post_id,
						$duplicate_id
					),
					array_merge( $error_data, array(
						'duplicate_id' => $duplicate_id,
						'instructions' => array(
							sprintf(
								/* translators: 1: duplicate ID */
								__( 'Edit the duplicate %1$s (ID: %2$d) instead of the original.', 'respira-for-wordpress' ),
								$post_type_label,
								$duplicate_id
							),
							__( 'After editing, approve the duplicate in WordPress admin to replace the original.', 'respira-for-wordpress' ),
						),
					) )
				);
			}

			// If force=true is requested, require 3 confirmations before allowing direct edit.
			if ( $force && $allow_direct_edit ) {
				$confirmation_key = 'respira_force_confirm_' . $post_id . '_' . $key_data['id'];
				$confirmations = get_transient( $confirmation_key );
				
				if ( false === $confirmations ) {
					$confirmations = 0;
				}
				
				$confirmations++;
				set_transient( $confirmation_key, $confirmations, 300 ); // 5 minute window
				
				if ( $confirmations < 3 ) {
					$remaining = 3 - $confirmations;
					$error_data = Respira_Approver::get_approval_error_data( $post_id );
					return new WP_Error(
						'respira_force_confirmation_required',
						sprintf(
							/* translators: 1: number of remaining confirmations, 2: post type label, 3: post ID */
							_n(
								'⚠️ SAFETY CHECK: You must confirm %1$d more time(s) before directly editing live %2$s (ID: %3$d). This is a LIVE post that visitors can see. Are you absolutely sure you want to edit it directly without creating a duplicate first?',
								'⚠️ SAFETY CHECK: You must confirm %1$d more time(s) before directly editing live %2$s (ID: %3$d). This is a LIVE post that visitors can see. Are you absolutely sure you want to edit it directly without creating a duplicate first?',
								$remaining,
								'respira-for-wordpress'
							),
							$remaining,
							$post_type_label,
							$post_id
						),
						array_merge( $error_data, array(
							'confirmations_required' => 3,
							'confirmations_received' => $confirmations,
							'confirmations_remaining' => $remaining,
							'instructions' => array(
								__( '⚠️ WARNING: You are attempting to edit a LIVE page/post directly.', 'respira-for-wordpress' ),
								__( 'For safety, Respira requires 3 separate confirmations before allowing direct edits to live content.', 'respira-for-wordpress' ),
								sprintf(
									/* translators: 1: number of remaining confirmations */
									_n(
										'Please confirm %1$d more time(s) by calling this endpoint again with force=true.',
										'Please confirm %1$d more time(s) by calling this endpoint again with force=true.',
										$remaining,
										'respira-for-wordpress'
									),
									$remaining
								),
								__( 'RECOMMENDED: Create a duplicate first using wordpress_create_post_duplicate, then edit the duplicate.', 'respira-for-wordpress' ),
							),
						) )
					);
				}
				
				// All 3 confirmations received - allow direct edit but log warning.
				Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'force_edit_post', 'post', $post_id );
				delete_transient( $confirmation_key );
			} else {
				// No force or direct editing disabled - automatically create duplicate.
				$suffix = '_duplicate_' . time();
				$duplicate_id = Respira_Duplicator::duplicate_post( $post_id, $suffix );
				
				if ( is_wp_error( $duplicate_id ) ) {
					return $duplicate_id;
				}
				
				// Log the automatic duplication.
				Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'auto_duplicate_post', 'post', $duplicate_id );
				
				// Switch to editing the duplicate instead of the original.
				$post_id = $duplicate_id;
				$post = get_post( $post_id );
				$is_duplicate = true;
				$auto_duplicate_created = true;
				
				// Continue with the update below - we'll add a note in the response.
			}
		}

		// Validate content if security checks are enabled and not skipped.
		$skip_security = $request->get_param( 'skip_security_check' );
		if ( get_option( 'respira_security_checks', 1 ) && ! $skip_security ) {
			$content = $request->get_param( 'content' );
			if ( $content ) {
				$security_check = Respira_Security::validate_content( $content );
				if ( is_wp_error( $security_check ) ) {
					return $security_check;
				}
			}
		}

		// Update the post.
		$update_data = array(
			'ID' => $post_id,
		);

		if ( $request->get_param( 'title' ) ) {
			$update_data['post_title'] = sanitize_text_field( $request->get_param( 'title' ) );
		}

		if ( $request->get_param( 'content' ) ) {
			// Use API-specific sanitization that allows scripts and modern web features.
			$update_data['post_content'] = $this->sanitize_api_content( $request->get_param( 'content' ) );
		}

		if ( $request->get_param( 'status' ) ) {
			$update_data['post_status'] = sanitize_text_field( $request->get_param( 'status' ) );
		}

		// Enable API update context to preserve script tags.
		Respira_Content_Filter::set_api_update_context( true );

		// Temporarily remove ALL WordPress content sanitization filters.
		$removed_filters = Respira_Content_Filter::remove_content_filters();

		// Update the post with preserved content.
		$result = wp_update_post( $update_data, true );

		// Restore WordPress filters.
		Respira_Content_Filter::restore_content_filters( $removed_filters );

		// Disable API update context.
		Respira_Content_Filter::set_api_update_context( false );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		// Update meta if provided.
		if ( $request->get_param( 'meta' ) ) {
			$meta = $request->get_param( 'meta' );
			foreach ( $meta as $key => $value ) {
				update_post_meta( $post_id, sanitize_key( $key ), $value );
			}
		}

		// Update SEO meta if provided.
		if ( $request->has_param( 'seo' ) ) {
			$seo_data = $request->get_param( 'seo' );
			if ( is_array( $seo_data ) && ! empty( $seo_data ) ) {
				$this->update_seo_meta( $post_id, $seo_data );
			}
		}

		// Update featured image if provided.
		if ( $request->has_param( 'featured_image' ) ) {
			$image_data = $request->get_param( 'featured_image' );
			if ( is_array( $image_data ) && ! empty( $image_data ) ) {
				$this->update_featured_image( $post_id, $image_data );
			}
		}

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'update_post', 'post', $post_id );

		// Track usage stat (non-blocking).
		// PRIVACY: Only aggregate statistics are sent - actual content is NEVER transmitted.
		// See class-respira-usage-tracker.php for detailed privacy documentation.
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-usage-tracker.php';
		Respira_Usage_Tracker::track( 'update_post', 'post', $post_id, $content );

		$updated_post = get_post( $post_id );
		$approvals_url = admin_url( 'admin.php?page=respira-approvals' );
		
		$response_data = array(
			'success' => true,
			'post'    => $this->format_post_response( $updated_post, true ),
			'message' => $this->add_success_footer( $this->format_respira_message( __( 'Post updated successfully! Your changes are safe and sound.', 'respira-for-wordpress' ), 'success' ) ),
		);
		
		// Add note if duplicate was auto-created.
		if ( $auto_duplicate_created ) {
			$response_data['duplicate_created'] = true;
			$response_data['duplicate_id'] = $post_id;
			$response_data['original_id'] = $original_id;
			$response_data['approval_url'] = $approvals_url;
			$response_data['message'] = $this->add_success_footer( $this->format_respira_message( sprintf(
				/* translators: 1: post type label, 2: duplicate ID */
				__( 'I created a duplicate %1$s for safety (ID: %2$d). Review it in WordPress admin before approving to make it live.', 'respira-for-wordpress' ),
				$post_type_label,
				$post_id
			), 'warning' ) );
			$response_data['instructions'] = array(
				$this->format_respira_message( __( 'A duplicate was automatically created for safety.', 'respira-for-wordpress' ), 'warning' ),
				sprintf(
					/* translators: 1: post type label, 2: duplicate ID, 3: original ID */
					__( '📋 Duplicate %1$s ID: %2$d | Original ID: %3$d', 'respira-for-wordpress' ),
					$post_type_label,
					$post_id,
					$original_id
				),
				sprintf(
					/* translators: 1: approvals URL */
					__( '👀 Review and approve: %1$s', 'respira-for-wordpress' ),
					$approvals_url
				),
				__( '💡 Tip: Changes are on the duplicate. Approve it to make them live.', 'respira-for-wordpress' ),
			);
		}

		$response = new WP_REST_Response( $response_data, 200 );
		
		// Add response headers for better debugging.
		if ( $auto_duplicate_created ) {
			$response->header( 'X-Respira-Duplicate-Created', 'true' );
			$response->header( 'X-Respira-Duplicate-ID', (string) $post_id );
			$response->header( 'X-Respira-Original-ID', (string) $original_id );
			$response->header( 'X-Respira-Approval-URL', $approvals_url );
		} else {
			$response->header( 'X-Respira-Duplicate-Created', 'false' );
		}

		return $response;
	}

	/**
	 * Delete post endpoint.
	 *
	 * @since 1.2.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function delete_post( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		$post_id  = $request->get_param( 'id' );

		$post = get_post( $post_id );
		if ( ! $post || 'post' !== $post->post_type ) {
			return new WP_Error(
				'respira_post_not_found',
				__( 'Post not found.', 'respira-for-wordpress' ),
				array( 'status' => 404 )
			);
		}

		// Only allow deletion of Respira duplicates unless forced.
		$is_duplicate = get_post_meta( $post_id, '_respira_duplicate', true );
		$force        = $request->get_param( 'force' );
		$allow_direct_edit = get_option( 'respira_allow_direct_edit', 0 );
		$post_type_label = __( 'post', 'respira-for-wordpress' );

		// If not a duplicate, check if we can delete it.
		if ( ! $is_duplicate ) {
			// If force is requested but direct editing is disabled, return error with instructions.
			if ( $force && ! $allow_direct_edit ) {
				$error_data = Respira_Approver::get_approval_error_data( $post_id );
				return new WP_Error(
					'respira_direct_edit_disabled',
					sprintf(
						/* translators: 1: post type label */
						__( 'Cannot delete original %1$s. Direct editing of original pages is disabled in settings. Only Respira duplicates can be deleted.', 'respira-for-wordpress' ),
						$post_type_label
					),
					$error_data
				);
			}

			// If force is not requested, return error with instructions.
			if ( ! $force ) {
				$error_data = Respira_Approver::get_approval_error_data( $post_id );
				$message = sprintf(
					/* translators: 1: post type label, 2: post ID */
					__( 'Cannot delete original %1$s (ID: %2$d). Only Respira duplicates can be deleted.', 'respira-for-wordpress' ),
					$post_type_label,
					$post_id
				);

				return new WP_Error(
					'respira_not_duplicate',
					$message,
					$error_data
				);
			}
		}

		$result = wp_delete_post( $post_id, true );

		if ( ! $result ) {
			return new WP_Error(
				'respira_delete_failed',
				__( 'Failed to delete post.', 'respira-for-wordpress' ),
				array( 'status' => 500 )
			);
		}

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'delete_post', 'post', $post_id );

		return new WP_REST_Response(
			array(
				'success' => true,
				'message' => $this->add_success_footer( __( 'Post deleted successfully.', 'respira-for-wordpress' ) ),
			),
			200
		);
	}

	/**
	 * List media endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response Response object.
	 */
	public function list_media( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );

		$args = array(
			'post_type'      => 'attachment',
			'post_status'    => 'inherit',
			'posts_per_page' => $request->get_param( 'per_page' ) ?? 20,
			'paged'          => $request->get_param( 'page' ) ?? 1,
			's'              => $request->get_param( 'search' ) ?? '',
		);

		$query = new WP_Query( $args );
		$media = array();

		foreach ( $query->posts as $post ) {
			$media[] = array(
				'id'         => $post->ID,
				'title'      => $post->post_title,
				'url'        => wp_get_attachment_url( $post->ID ),
				'mime_type'  => $post->post_mime_type,
				'uploaded'   => $post->post_date,
				'file_size'  => filesize( get_attached_file( $post->ID ) ),
			);
		}

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'list_media', 'media', null );

		return new WP_REST_Response(
			array(
				'media'       => $media,
				'total'       => $query->found_posts,
				'total_pages' => $query->max_num_pages,
			),
			200
		);
	}

	/**
	 * Upload media endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function upload_media( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );

		// Handle file upload.
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/media.php';
		require_once ABSPATH . 'wp-admin/includes/image.php';

		$file = $request->get_file_params();

		if ( empty( $file['file'] ) ) {
			return $this->respira_error(
				'respira_no_file',
				__( 'No file provided. Please include a file in your request.', 'respira-for-wordpress' ),
				array( 'status' => 400 ),
				'error'
			);
		}

		$attachment_id = media_handle_upload( 'file', 0 );

		if ( is_wp_error( $attachment_id ) ) {
			return $attachment_id;
		}

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'upload_media', 'media', $attachment_id );

		return new WP_REST_Response(
			array(
				'success' => true,
				'media'   => array(
					'id'   => $attachment_id,
					'url'  => wp_get_attachment_url( $attachment_id ),
					'type' => get_post_mime_type( $attachment_id ),
				),
				'message' => $this->add_success_footer( $this->format_respira_message( __( 'Media uploaded successfully! Your file is ready to use.', 'respira-for-wordpress' ), 'success' ) ),
			),
			201
		);
	}

	/**
	 * Extract page builder content endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function extract_builder_content( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		$builder  = $request->get_param( 'builder' );
		$page_id  = $request->get_param( 'page_id' );

		$builder_instance = Respira_Builder_Interface::get_builder( $builder );

		if ( is_wp_error( $builder_instance ) ) {
			return $builder_instance;
		}

		$content = $builder_instance->extract_content( $page_id );

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'extract_builder_content', 'page', $page_id );

		return new WP_REST_Response(
			array(
				'success' => true,
				'builder' => $builder,
				'content' => $content,
			),
			200
		);
	}

	/**
	 * Inject page builder content endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response|WP_Error Response object.
	 */
	public function inject_builder_content( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		$builder  = $request->get_param( 'builder' );
		$page_id  = $request->get_param( 'page_id' );
		$content  = $request->get_param( 'content' );

		$builder_instance = Respira_Builder_Interface::get_builder( $builder );

		if ( is_wp_error( $builder_instance ) ) {
			return $builder_instance;
		}

		$result = $builder_instance->inject_content( $page_id, $content );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'inject_builder_content', 'page', $page_id );

		// Track usage stat (non-blocking).
		// PRIVACY: Only aggregate statistics are sent - actual content is NEVER transmitted.
		require_once RESPIRA_PLUGIN_DIR . 'includes/class-respira-usage-tracker.php';
		Respira_Usage_Tracker::track( 'inject_builder_content', 'page', $page_id, $content );

		return new WP_REST_Response(
			array(
				'success' => true,
				'message' => $this->add_success_footer( $this->format_respira_message( __( 'Content injected successfully! Your page builder content is now updated.', 'respira-for-wordpress' ), 'success' ) ),
			),
			200
		);
	}

	/**
	 * Validate security endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response Response object.
	 */
	public function validate_security( $request ) {
		$key_data = $request->get_param( '_respira_key_data' );
		$content  = $request->get_param( 'content' );

		$validation = Respira_Security::validate_content( $content );

		Respira_Auth::log_action( $key_data['id'], $key_data['user_id'], 'validate_security', 'validation', null );

		if ( is_wp_error( $validation ) ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'valid'   => false,
					'issues'  => $validation->get_error_messages(),
				),
				200
			);
		}

		return new WP_REST_Response(
			array(
				'success' => true,
				'valid'   => true,
				'message' => $this->format_respira_message( __( 'Content passed security validation. Looks good to go!', 'respira-for-wordpress' ), 'success' ),
			),
			200
		);
	}

	/**
	 * Get audit logs endpoint.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request The request object.
	 * @return WP_REST_Response Response object.
	 */
	public function get_audit_logs( $request ) {
		global $wpdb;

		$key_data = $request->get_param( '_respira_key_data' );
		if ( ! $key_data ) {
			return new WP_Error(
				'respira_unauthorized',
				__( 'API key required.', 'respira-for-wordpress' ),
				array( 'status' => 401 )
			);
		}

		$table_name = $wpdb->prefix . 'respira_audit_log';
		$per_page   = $request->get_param( 'per_page' ) ?? 20;
		$page       = $request->get_param( 'page' ) ?? 1;
		$offset     = ( $page - 1 ) * $per_page;

		// Filter logs by API key ID (users can only see their own logs)
		$logs = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT * FROM $table_name WHERE api_key_id = %d ORDER BY created_at DESC LIMIT %d OFFSET %d",
				$key_data['id'],
				$per_page,
				$offset
			)
		);

		$total = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM $table_name WHERE api_key_id = %d",
				$key_data['id']
			)
		);

		return new WP_REST_Response(
			array(
				'logs'        => $logs,
				'total'       => $total,
				'total_pages' => ceil( $total / $per_page ),
			),
			200
		);
	}

	/**
	 * Format post response.
	 *
	 * @since  1.0.0
	 * @param  WP_Post $post      The post object.
	 * @param  bool    $full_data Whether to include full data including meta.
	 * @return array   Formatted post data.
	 */
	private function format_post_response( $post, $full_data = false ) {
		$data = array(
			'id'             => $post->ID,
			'title'          => $post->post_title,
			'slug'           => $post->post_name,
			'status'         => $post->post_status,
			'type'           => $post->post_type,
			'author'         => $post->post_author,
			'date'           => $post->post_date,
			'modified'       => $post->post_modified,
			'url'            => get_permalink( $post->ID ),
			'is_duplicate'   => (bool) get_post_meta( $post->ID, '_respira_duplicate', true ),
			'original_id'    => get_post_meta( $post->ID, '_respira_original_id', true ),
		);

		if ( $full_data ) {
			$data['content'] = $post->post_content;
			$data['excerpt'] = $post->post_excerpt;
			$data['meta']    = get_post_meta( $post->ID );

			// Include custom CSS if it exists (Divi Builder).
			$custom_css = get_post_meta( $post->ID, '_et_pb_custom_css', true );
			if ( $custom_css ) {
				$data['custom_css'] = $custom_css;
			}

			// Include SEO meta data.
			$seo_meta = $this->get_seo_meta( $post->ID );
			if ( ! empty( array_filter( $seo_meta ) ) ) {
				$data['seo'] = $seo_meta;
			}

			// Include featured image data.
			$thumbnail_id = get_post_thumbnail_id( $post->ID );
			if ( $thumbnail_id ) {
				$data['featured_image'] = array(
					'id'  => $thumbnail_id,
					'url' => wp_get_attachment_url( $thumbnail_id ),
					'alt' => get_post_meta( $thumbnail_id, '_wp_attachment_image_alt', true ),
				);
			}

			// Detect page builder.
			$builder = Respira_Builder_Interface::detect_builder( $post->ID );
			if ( $builder ) {
				$data['builder'] = $builder->get_name();

				// Add builder-specific schema for Divi.
				if ( 'Divi' === $builder->get_name() && method_exists( $builder, 'get_builder_schema' ) ) {
					// Extract module names from content to get relevant schemas.
					$modules_used = array();
					if ( ! empty( $post->post_content ) && preg_match_all( '/\[et_pb_(\w+)/', $post->post_content, $matches ) ) {
						$modules_used = array_unique( array_map( function( $name ) {
							return 'et_pb_' . $name;
						}, $matches[1] ) );
					}

					$data['builder_data'] = $builder->get_builder_schema( $modules_used );
				}
			}
		}

		return $data;
	}

	/**
	 * Get allowed HTML tags for API content.
	 *
	 * Returns an array of allowed HTML tags and attributes for content
	 * submitted via the Respira API. Extends wp_kses_post to allow:
	 * - Script tags (for third-party embeds, analytics, etc.)
	 * - Data attributes (for modern JavaScript frameworks)
	 * - SVG elements (for scalable graphics)
	 * - Modern web components
	 *
	 * @since  1.0.3
	 * @return array Allowed HTML tags and attributes.
	 */
	private function get_allowed_html_for_api() {
		// Start with WordPress post allowed tags.
		$allowed = wp_kses_allowed_html( 'post' );

		// Allow script tags for third-party embeds (Cal.com, analytics, etc.).
		$allowed['script'] = array(
			'type'            => true,
			'src'             => true,
			'async'           => true,
			'defer'           => true,
			'charset'         => true,
			'crossorigin'     => true,
			'integrity'       => true,
			'nomodule'        => true,
			'nonce'           => true,
			'referrerpolicy'  => true,
		);

		// Allow iframe for embeds (videos, maps, etc.).
		$allowed['iframe'] = array(
			'src'             => true,
			'width'           => true,
			'height'          => true,
			'frameborder'     => true,
			'allowfullscreen' => true,
			'allow'           => true,
			'loading'         => true,
			'title'           => true,
			'name'            => true,
			'sandbox'         => true,
		);

		// Allow data attributes on all tags for modern JavaScript.
		foreach ( $allowed as $tag => $attributes ) {
			if ( is_array( $attributes ) ) {
				// Allow all data-* attributes.
				$allowed[ $tag ]['data-*'] = true;
				// Common data attributes.
				$allowed[ $tag ]['data-parallax']       = true;
				$allowed[ $tag ]['data-parallax-speed'] = true;
				$allowed[ $tag ]['data-aos']            = true;
				$allowed[ $tag ]['data-aos-delay']      = true;
				$allowed[ $tag ]['data-sal']            = true;
				$allowed[ $tag ]['data-scroll']         = true;
			}
		}

		// Allow SVG elements for scalable graphics.
		$svg_args = array(
			'class'           => true,
			'aria-hidden'     => true,
			'aria-labelledby' => true,
			'role'            => true,
			'xmlns'           => true,
			'width'           => true,
			'height'          => true,
			'viewbox'         => true,
			'fill'            => true,
			'stroke'          => true,
			'stroke-width'    => true,
			'stroke-linecap'  => true,
			'stroke-linejoin' => true,
		);

		$allowed['svg']   = $svg_args;
		$allowed['g']     = $svg_args;
		$allowed['title'] = array( 'title' => true );
		$allowed['path']  = $svg_args + array(
			'd'          => true,
			'fill-rule'  => true,
			'clip-rule'  => true,
		);
		$allowed['circle']   = $svg_args + array( 'cx' => true, 'cy' => true, 'r' => true );
		$allowed['rect']     = $svg_args + array( 'x' => true, 'y' => true, 'rx' => true, 'ry' => true );
		$allowed['line']     = $svg_args + array( 'x1' => true, 'y1' => true, 'x2' => true, 'y2' => true );
		$allowed['polygon']  = $svg_args + array( 'points' => true );
		$allowed['polyline'] = $svg_args + array( 'points' => true );

		return $allowed;
	}

	/**
	 * Preserve script tags in Divi HTML content and code blocks.
	 *
	 * Divi's et_pb_html_content and et_pb_code blocks may contain JavaScript.
	 * This method decodes any HTML entities in these blocks to preserve scripts.
	 *
	 * @since  1.0.3
	 * @param  string $content Content to process.
	 * @return string Content with decoded scripts in Divi blocks.
	 */
	private function preserve_divi_scripts( $content ) {
		// Preserve scripts in et_pb_html_content blocks.
		$content = preg_replace_callback(
			'/\[et_pb_html_content([^\]]*)\](.*?)\[\/et_pb_html_content\]/s',
			function( $matches ) {
				$attributes   = $matches[1];
				$html_content = $matches[2];

				// If it contains script tags (even encoded), decode them.
				if ( stripos( $html_content, 'script' ) !== false ) {
					$html_content = html_entity_decode( $html_content, ENT_QUOTES | ENT_HTML5, 'UTF-8' );
				}

				return '[et_pb_html_content' . $attributes . ']' . $html_content . '[/et_pb_html_content]';
			},
			$content
		);

		// Preserve scripts in et_pb_code blocks.
		$content = preg_replace_callback(
			'/\[et_pb_code([^\]]*)\](.*?)\[\/et_pb_code\]/s',
			function( $matches ) {
				$attributes   = $matches[1];
				$code_content = $matches[2];

				// If it contains script tags (even encoded), decode them.
				if ( stripos( $code_content, 'script' ) !== false ) {
					$code_content = html_entity_decode( $code_content, ENT_QUOTES | ENT_HTML5, 'UTF-8' );
				}

				return '[et_pb_code' . $attributes . ']' . $code_content . '[/et_pb_code]';
			},
			$content
		);

		return $content;
	}

	/**
	 * Sanitize content for API updates.
	 *
	 * Sanitizes HTML content while preserving script tags, modern web features,
	 * and Divi Builder shortcodes. This is more permissive than wp_kses_post
	 * but still secure for authenticated API requests.
	 *
	 * @since  1.0.3
	 * @param  string $content Content to sanitize.
	 * @return string Sanitized content.
	 */
	private function sanitize_api_content( $content ) {
		// First, preserve and decode scripts in Divi blocks.
		$content = $this->preserve_divi_scripts( $content );

		// Allow the full set of HTML tags for API content.
		$allowed_html = $this->get_allowed_html_for_api();

		// Sanitize with extended allowed tags.
		$sanitized = wp_kses( $content, $allowed_html );

		// Preserve Divi shortcodes - they may contain encoded content.
		// This ensures et_pb_html_content and et_pb_code blocks work correctly.
		return $sanitized;
	}

	/**
	 * Sanitize custom CSS.
	 *
	 * Sanitizes CSS while preserving CSS syntax. Removes potentially
	 * dangerous content but keeps valid CSS. Supports Divi Builder
	 * custom CSS meta field (_et_pb_custom_css).
	 *
	 * @since  1.0.1
	 * @param  string $css CSS content to sanitize.
	 * @return string Sanitized CSS.
	 */
	public function sanitize_custom_css( $css ) {
		// Allow empty string to clear custom CSS.
		if ( '' === $css ) {
			return '';
		}

		// Remove any HTML/PHP tags while preserving CSS.
		$css = wp_strip_all_tags( $css );

		// Remove any null bytes.
		$css = str_replace( chr( 0 ), '', $css );

		// Trim whitespace.
		$css = trim( $css );

		// Optional: Add size limit (50KB recommended).
		$max_size = apply_filters( 'respira_custom_css_max_size', 51200 ); // 50KB default.
		if ( strlen( $css ) > $max_size ) {
			return new WP_Error(
				'respira_css_too_large',
				sprintf(
					/* translators: %s: Maximum size in KB */
					__( 'Custom CSS exceeds maximum size of %s KB.', 'respira-for-wordpress' ),
					number_format( $max_size / 1024, 0 )
				),
				array( 'status' => 400 )
			);
		}

		return $css;
	}

	/**
	 * Get common list args for pagination.
	 *
	 * @since  1.0.0
	 * @return array Args array.
	 */
	private function get_list_args() {
		return array(
			'page'     => array(
				'required'          => false,
				'type'              => 'integer',
				'default'           => 1,
				'sanitize_callback' => 'absint',
			),
			'per_page' => array(
				'required'          => false,
				'type'              => 'integer',
				'default'           => 20,
				'sanitize_callback' => 'absint',
			),
			'search'   => array(
				'required'          => false,
				'type'              => 'string',
				'sanitize_callback' => 'sanitize_text_field',
			),
			'status'   => array(
				'required' => false,
				'type'     => 'string',
			),
		);
	}

	/**
	 * Detect active SEO plugin.
	 *
	 * @since  1.0.1
	 * @return string SEO plugin identifier ('yoast', 'rank_math', 'aioseo', 'core').
	 */
	private function detect_seo_plugin() {
		if ( ! function_exists( 'is_plugin_active' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		if ( is_plugin_active( 'wordpress-seo/wp-seo.php' ) || defined( 'WPSEO_VERSION' ) ) {
			return 'yoast';
		} elseif ( is_plugin_active( 'seo-by-rank-math/rank-math.php' ) || defined( 'RANK_MATH_VERSION' ) ) {
			return 'rank_math';
		} elseif ( is_plugin_active( 'all-in-one-seo-pack/all_in_one_seo_pack.php' ) || defined( 'AIOSEOP_VERSION' ) ) {
			return 'aioseo';
		}

		return 'core';
	}

	/**
	 * Update SEO meta fields.
	 *
	 * @since  1.0.1
	 * @param  int   $post_id  Post ID.
	 * @param  array $seo_data SEO data array.
	 * @return bool  True on success.
	 */
	private function update_seo_meta( $post_id, $seo_data ) {
		$seo_plugin = $this->detect_seo_plugin();

		switch ( $seo_plugin ) {
			case 'yoast':
				return $this->update_yoast_seo( $post_id, $seo_data );
			case 'rank_math':
				return $this->update_rank_math_seo( $post_id, $seo_data );
			case 'aioseo':
				return $this->update_aioseo_seo( $post_id, $seo_data );
			default:
				return $this->update_core_seo( $post_id, $seo_data );
		}
	}

	/**
	 * Update Yoast SEO meta fields.
	 *
	 * @since  1.0.1
	 * @param  int   $post_id  Post ID.
	 * @param  array $seo_data SEO data array.
	 * @return bool  True on success.
	 */
	private function update_yoast_seo( $post_id, $seo_data ) {
		$mapping = array(
			'title'              => '_yoast_wpseo_title',
			'description'        => '_yoast_wpseo_metadesc',
			'focus_keyword'      => '_yoast_wpseo_focuskw',
			'og_title'           => '_yoast_wpseo_opengraph-title',
			'og_description'     => '_yoast_wpseo_opengraph-description',
			'og_image'           => '_yoast_wpseo_opengraph-image',
			'twitter_title'      => '_yoast_wpseo_twitter-title',
			'twitter_description' => '_yoast_wpseo_twitter-description',
			'twitter_image'      => '_yoast_wpseo_twitter-image',
			'canonical_url'      => '_yoast_wpseo_canonical',
		);

		foreach ( $mapping as $key => $meta_key ) {
			if ( isset( $seo_data[ $key ] ) ) {
				$value = $this->sanitize_seo_field( $key, $seo_data[ $key ] );
				update_post_meta( $post_id, $meta_key, $value );
			}
		}

		// Handle robots directives.
		if ( isset( $seo_data['robots'] ) ) {
			$this->update_yoast_robots( $post_id, $seo_data['robots'] );
		}

		return true;
	}

	/**
	 * Update Yoast robots directives.
	 *
	 * @since  1.0.1
	 * @param  int    $post_id Post ID.
	 * @param  string $robots  Robots directives string.
	 * @return void
	 */
	private function update_yoast_robots( $post_id, $robots ) {
		$robots = strtolower( $robots );

		// Parse robots string (e.g., "index, follow" or "noindex, nofollow").
		$no_index  = strpos( $robots, 'noindex' ) !== false ? 1 : 0;
		$no_follow = strpos( $robots, 'nofollow' ) !== false ? 1 : 0;

		update_post_meta( $post_id, '_yoast_wpseo_meta-robots-noindex', $no_index );
		update_post_meta( $post_id, '_yoast_wpseo_meta-robots-nofollow', $no_follow );

		// Advanced directives (noodp, noimageindex, noarchive, nosnippet).
		$advanced = array();
		if ( strpos( $robots, 'noodp' ) !== false ) {
			$advanced[] = 'noodp';
		}
		if ( strpos( $robots, 'noimageindex' ) !== false ) {
			$advanced[] = 'noimageindex';
		}
		if ( strpos( $robots, 'noarchive' ) !== false ) {
			$advanced[] = 'noarchive';
		}
		if ( strpos( $robots, 'nosnippet' ) !== false ) {
			$advanced[] = 'nosnippet';
		}

		if ( ! empty( $advanced ) ) {
			update_post_meta( $post_id, '_yoast_wpseo_meta-robots-adv', implode( ',', $advanced ) );
		}
	}

	/**
	 * Update Rank Math SEO meta fields.
	 *
	 * @since  1.0.1
	 * @param  int   $post_id  Post ID.
	 * @param  array $seo_data SEO data array.
	 * @return bool  True on success.
	 */
	private function update_rank_math_seo( $post_id, $seo_data ) {
		$mapping = array(
			'title'              => 'rank_math_title',
			'description'        => 'rank_math_description',
			'focus_keyword'      => 'rank_math_focus_keyword',
			'og_title'           => 'rank_math_facebook_title',
			'og_description'     => 'rank_math_facebook_description',
			'twitter_title'      => 'rank_math_twitter_title',
			'twitter_description' => 'rank_math_twitter_description',
			'canonical_url'      => 'rank_math_canonical_url',
		);

		foreach ( $mapping as $key => $meta_key ) {
			if ( isset( $seo_data[ $key ] ) ) {
				$value = $this->sanitize_seo_field( $key, $seo_data[ $key ] );
				update_post_meta( $post_id, $meta_key, $value );
			}
		}

		// Handle OG/Twitter images (Rank Math uses attachment IDs).
		if ( isset( $seo_data['og_image'] ) ) {
			$image_id = $this->get_or_create_attachment( $seo_data['og_image'], $post_id );
			if ( $image_id ) {
				update_post_meta( $post_id, 'rank_math_facebook_image', $image_id );
				update_post_meta( $post_id, 'rank_math_facebook_image_id', $image_id );
			}
		}

		if ( isset( $seo_data['twitter_image'] ) ) {
			$image_id = $this->get_or_create_attachment( $seo_data['twitter_image'], $post_id );
			if ( $image_id ) {
				update_post_meta( $post_id, 'rank_math_twitter_image', $image_id );
				update_post_meta( $post_id, 'rank_math_twitter_image_id', $image_id );
			}
		}

		// Handle robots directives.
		if ( isset( $seo_data['robots'] ) ) {
			$robots_array = array_map( 'trim', explode( ',', strtolower( $seo_data['robots'] ) ) );
			update_post_meta( $post_id, 'rank_math_robots', $robots_array );
		}

		return true;
	}

	/**
	 * Update All in One SEO meta fields.
	 *
	 * @since  1.0.1
	 * @param  int   $post_id  Post ID.
	 * @param  array $seo_data SEO data array.
	 * @return bool  True on success.
	 */
	private function update_aioseo_seo( $post_id, $seo_data ) {
		$mapping = array(
			'title'         => '_aioseop_title',
			'description'   => '_aioseop_description',
			'og_title'      => '_aioseop_opengraph_settings',
			'og_description' => '_aioseop_opengraph_settings',
			'og_image'      => '_aioseop_opengraph_settings',
			'canonical_url' => '_aioseop_custom_link',
		);

		// AIOSEO stores some data in a serialized array.
		$opengraph_settings = get_post_meta( $post_id, '_aioseop_opengraph_settings', true );
		if ( ! is_array( $opengraph_settings ) ) {
			$opengraph_settings = array();
		}

		if ( isset( $seo_data['title'] ) ) {
			update_post_meta( $post_id, '_aioseop_title', sanitize_text_field( $seo_data['title'] ) );
		}

		if ( isset( $seo_data['description'] ) ) {
			update_post_meta( $post_id, '_aioseop_description', sanitize_text_field( $seo_data['description'] ) );
		}

		if ( isset( $seo_data['og_title'] ) ) {
			$opengraph_settings['aioseop_opengraph_settings_title'] = sanitize_text_field( $seo_data['og_title'] );
		}

		if ( isset( $seo_data['og_description'] ) ) {
			$opengraph_settings['aioseop_opengraph_settings_desc'] = sanitize_text_field( $seo_data['og_description'] );
		}

		if ( isset( $seo_data['og_image'] ) ) {
			$opengraph_settings['aioseop_opengraph_settings_image'] = esc_url_raw( $seo_data['og_image'] );
		}

		update_post_meta( $post_id, '_aioseop_opengraph_settings', $opengraph_settings );

		if ( isset( $seo_data['canonical_url'] ) ) {
			update_post_meta( $post_id, '_aioseop_custom_link', esc_url_raw( $seo_data['canonical_url'] ) );
		}

		return true;
	}

	/**
	 * Update core WordPress SEO fields (fallback).
	 *
	 * @since  1.0.1
	 * @param  int   $post_id  Post ID.
	 * @param  array $seo_data SEO data array.
	 * @return bool  True on success.
	 */
	private function update_core_seo( $post_id, $seo_data ) {
		// Use post excerpt for description if no SEO plugin.
		if ( isset( $seo_data['description'] ) ) {
			wp_update_post(
				array(
					'ID'           => $post_id,
					'post_excerpt' => sanitize_text_field( $seo_data['description'] ),
				)
			);
		}

		// Store basic SEO data in custom meta fields.
		if ( isset( $seo_data['title'] ) ) {
			update_post_meta( $post_id, '_seo_title', sanitize_text_field( $seo_data['title'] ) );
		}

		if ( isset( $seo_data['og_title'] ) ) {
			update_post_meta( $post_id, '_og_title', sanitize_text_field( $seo_data['og_title'] ) );
		}

		if ( isset( $seo_data['og_description'] ) ) {
			update_post_meta( $post_id, '_og_description', sanitize_text_field( $seo_data['og_description'] ) );
		}

		if ( isset( $seo_data['og_image'] ) ) {
			update_post_meta( $post_id, '_og_image', esc_url_raw( $seo_data['og_image'] ) );
		}

		return true;
	}

	/**
	 * Sanitize SEO field.
	 *
	 * @since  1.0.1
	 * @param  string $field Field name.
	 * @param  mixed  $value Field value.
	 * @return mixed  Sanitized value.
	 */
	private function sanitize_seo_field( $field, $value ) {
		// URL fields.
		if ( in_array( $field, array( 'og_image', 'twitter_image', 'canonical_url' ), true ) ) {
			return esc_url_raw( $value );
		}

		// Text fields.
		return sanitize_text_field( $value );
	}

	/**
	 * Update featured image.
	 *
	 * @since  1.0.1
	 * @param  int   $post_id    Post ID.
	 * @param  array $image_data Image data array.
	 * @return int|false Attachment ID on success, false on failure.
	 */
	private function update_featured_image( $post_id, $image_data ) {
		// If attachment ID provided, use it directly.
		if ( isset( $image_data['id'] ) && is_numeric( $image_data['id'] ) ) {
			$attachment_id = intval( $image_data['id'] );
			if ( get_post( $attachment_id ) && 'attachment' === get_post_type( $attachment_id ) ) {
				set_post_thumbnail( $post_id, $attachment_id );

				// Update alt text if provided.
				if ( isset( $image_data['alt'] ) ) {
					update_post_meta( $attachment_id, '_wp_attachment_image_alt', sanitize_text_field( $image_data['alt'] ) );
				}

				return $attachment_id;
			}
		}

		// If URL provided, download and create attachment.
		if ( isset( $image_data['url'] ) ) {
			$attachment_id = $this->get_or_create_attachment( $image_data['url'], $post_id );
			if ( $attachment_id ) {
				set_post_thumbnail( $post_id, $attachment_id );

				// Update attachment metadata.
				if ( isset( $image_data['alt'] ) ) {
					update_post_meta( $attachment_id, '_wp_attachment_image_alt', sanitize_text_field( $image_data['alt'] ) );
				}

				if ( isset( $image_data['title'] ) ) {
					wp_update_post(
						array(
							'ID'         => $attachment_id,
							'post_title' => sanitize_text_field( $image_data['title'] ),
						)
					);
				}

				return $attachment_id;
			}
		}

		return false;
	}

	/**
	 * Get or create attachment from URL.
	 *
	 * @since  1.0.1
	 * @param  string $image_url Image URL.
	 * @param  int    $post_id   Post ID to attach image to.
	 * @return int|false Attachment ID on success, false on failure.
	 */
	private function get_or_create_attachment( $image_url, $post_id = 0 ) {
		// Check if URL is valid.
		if ( ! filter_var( $image_url, FILTER_VALIDATE_URL ) ) {
			return false;
		}

		// Check if attachment already exists by URL.
		$existing_attachment = $this->get_attachment_by_url( $image_url );
		if ( $existing_attachment ) {
			return $existing_attachment;
		}

		// Download and create attachment.
		return $this->download_and_create_attachment( $image_url, $post_id );
	}

	/**
	 * Get attachment ID by URL.
	 *
	 * @since  1.0.1
	 * @param  string $url Image URL.
	 * @return int|false Attachment ID if found, false otherwise.
	 */
	private function get_attachment_by_url( $url ) {
		global $wpdb;

		$attachment = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT ID FROM {$wpdb->posts} WHERE guid = %s AND post_type = 'attachment'",
				$url
			)
		);

		return $attachment ? intval( $attachment ) : false;
	}

	/**
	 * Download image and create attachment.
	 *
	 * @since  1.0.1
	 * @param  string $image_url Image URL.
	 * @param  int    $post_id   Post ID to attach image to.
	 * @return int|false Attachment ID on success, false on failure.
	 */
	private function download_and_create_attachment( $image_url, $post_id = 0 ) {
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/media.php';
		require_once ABSPATH . 'wp-admin/includes/image.php';

		// Validate URL.
		if ( ! filter_var( $image_url, FILTER_VALIDATE_URL ) ) {
			return false;
		}

		// Download image.
		$tmp = download_url( $image_url, 300 ); // 5 minute timeout.

		if ( is_wp_error( $tmp ) ) {
			return false;
		}

		// Get file extension and validate.
		$file_array = array(
			'name'     => basename( wp_parse_url( $image_url, PHP_URL_PATH ) ),
			'tmp_name' => $tmp,
		);

		// Ensure file has a name.
		if ( empty( $file_array['name'] ) ) {
			$file_array['name'] = 'image-' . time() . '.jpg';
		}

		// Validate file type.
		$file_type = wp_check_filetype( $file_array['name'] );
		$allowed_types = array( 'image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/svg+xml' );

		if ( ! in_array( $file_type['type'], $allowed_types, true ) ) {
			@unlink( $tmp );
			return false;
		}

		// Check file size (max 10MB).
		$max_size = apply_filters( 'respira_max_image_size', 10485760 ); // 10MB default.
		if ( filesize( $tmp ) > $max_size ) {
			@unlink( $tmp );
			return false;
		}

		// Create attachment.
		$attachment_id = media_handle_sideload( $file_array, $post_id );

		if ( is_wp_error( $attachment_id ) ) {
			@unlink( $tmp );
			return false;
		}

		return $attachment_id;
	}

	/**
	 * Get SEO meta data for a post.
	 *
	 * @since  1.0.1
	 * @param  int $post_id Post ID.
	 * @return array SEO meta data.
	 */
	private function get_seo_meta( $post_id ) {
		$seo_plugin = $this->detect_seo_plugin();
		$seo_data   = array();

		switch ( $seo_plugin ) {
			case 'yoast':
				$seo_data = $this->get_yoast_seo( $post_id );
				break;
			case 'rank_math':
				$seo_data = $this->get_rank_math_seo( $post_id );
				break;
			case 'aioseo':
				$seo_data = $this->get_aioseo_seo( $post_id );
				break;
			default:
				$seo_data = $this->get_core_seo( $post_id );
		}

		// Add SEO plugin info.
		$seo_data['seo_plugin'] = $seo_plugin;

		return $seo_data;
	}

	/**
	 * Get Yoast SEO meta data.
	 *
	 * @since  1.0.1
	 * @param  int $post_id Post ID.
	 * @return array SEO meta data.
	 */
	private function get_yoast_seo( $post_id ) {
		return array(
			'title'              => get_post_meta( $post_id, '_yoast_wpseo_title', true ),
			'description'        => get_post_meta( $post_id, '_yoast_wpseo_metadesc', true ),
			'focus_keyword'      => get_post_meta( $post_id, '_yoast_wpseo_focuskw', true ),
			'og_title'           => get_post_meta( $post_id, '_yoast_wpseo_opengraph-title', true ),
			'og_description'     => get_post_meta( $post_id, '_yoast_wpseo_opengraph-description', true ),
			'og_image'           => get_post_meta( $post_id, '_yoast_wpseo_opengraph-image', true ),
			'twitter_title'      => get_post_meta( $post_id, '_yoast_wpseo_twitter-title', true ),
			'twitter_description' => get_post_meta( $post_id, '_yoast_wpseo_twitter-description', true ),
			'twitter_image'      => get_post_meta( $post_id, '_yoast_wpseo_twitter-image', true ),
			'canonical_url'      => get_post_meta( $post_id, '_yoast_wpseo_canonical', true ),
		);
	}

	/**
	 * Get Rank Math SEO meta data.
	 *
	 * @since  1.0.1
	 * @param  int $post_id Post ID.
	 * @return array SEO meta data.
	 */
	private function get_rank_math_seo( $post_id ) {
		$og_image_id      = get_post_meta( $post_id, 'rank_math_facebook_image_id', true );
		$twitter_image_id = get_post_meta( $post_id, 'rank_math_twitter_image_id', true );

		return array(
			'title'              => get_post_meta( $post_id, 'rank_math_title', true ),
			'description'        => get_post_meta( $post_id, 'rank_math_description', true ),
			'focus_keyword'      => get_post_meta( $post_id, 'rank_math_focus_keyword', true ),
			'og_title'           => get_post_meta( $post_id, 'rank_math_facebook_title', true ),
			'og_description'     => get_post_meta( $post_id, 'rank_math_facebook_description', true ),
			'og_image'           => $og_image_id ? wp_get_attachment_url( $og_image_id ) : '',
			'twitter_title'      => get_post_meta( $post_id, 'rank_math_twitter_title', true ),
			'twitter_description' => get_post_meta( $post_id, 'rank_math_twitter_description', true ),
			'twitter_image'      => $twitter_image_id ? wp_get_attachment_url( $twitter_image_id ) : '',
			'canonical_url'      => get_post_meta( $post_id, 'rank_math_canonical_url', true ),
		);
	}

	/**
	 * Get All in One SEO meta data.
	 *
	 * @since  1.0.1
	 * @param  int $post_id Post ID.
	 * @return array SEO meta data.
	 */
	private function get_aioseo_seo( $post_id ) {
		$opengraph_settings = get_post_meta( $post_id, '_aioseop_opengraph_settings', true );
		if ( ! is_array( $opengraph_settings ) ) {
			$opengraph_settings = array();
		}

		return array(
			'title'         => get_post_meta( $post_id, '_aioseop_title', true ),
			'description'   => get_post_meta( $post_id, '_aioseop_description', true ),
			'og_title'      => isset( $opengraph_settings['aioseop_opengraph_settings_title'] ) ? $opengraph_settings['aioseop_opengraph_settings_title'] : '',
			'og_description' => isset( $opengraph_settings['aioseop_opengraph_settings_desc'] ) ? $opengraph_settings['aioseop_opengraph_settings_desc'] : '',
			'og_image'      => isset( $opengraph_settings['aioseop_opengraph_settings_image'] ) ? $opengraph_settings['aioseop_opengraph_settings_image'] : '',
			'canonical_url' => get_post_meta( $post_id, '_aioseop_custom_link', true ),
		);
	}

	/**
	 * Get core WordPress SEO data (fallback).
	 *
	 * @since  1.0.1
	 * @param  int $post_id Post ID.
	 * @return array SEO meta data.
	 */
	private function get_core_seo( $post_id ) {
		$post = get_post( $post_id );

		return array(
			'title'         => get_post_meta( $post_id, '_seo_title', true ),
			'description'   => $post ? $post->post_excerpt : '',
			'og_title'      => get_post_meta( $post_id, '_og_title', true ),
			'og_description' => get_post_meta( $post_id, '_og_description', true ),
			'og_image'      => get_post_meta( $post_id, '_og_image', true ),
		);
	}

	/**
	 * Output custom CSS to page head.
	 *
	 * Outputs custom CSS stored in the _et_pb_custom_css meta field to the page's
	 * <head> section when the page is rendered on the frontend. This ensures that
	 * custom CSS set via the API is actually applied to the page.
	 *
	 * @since  1.0.2
	 * @return void
	 */
	public function output_page_custom_css() {
		// Only output on single page views.
		if ( ! is_singular( 'page' ) ) {
			return;
		}

		global $post;
		if ( ! $post || ! isset( $post->ID ) ) {
			return;
		}

		// Get custom CSS from meta field (Divi Builder field).
		$custom_css = get_post_meta( $post->ID, '_et_pb_custom_css', true );

		// If CSS exists and is not empty, output it.
		if ( ! empty( $custom_css ) ) {
			// Sanitize one more time for safety (remove any HTML tags).
			$sanitized_css = wp_strip_all_tags( $custom_css );

			// Output CSS in style tag.
			echo "\n<!-- Respira Custom CSS -->\n";
			echo '<style id="respira-page-custom-css" type="text/css">' . "\n";
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- CSS is sanitized above with wp_strip_all_tags.
			echo $sanitized_css . "\n";
			echo '</style>' . "\n";
		}
	}

	/**
	 * Add approval links to analysis response.
	 *
	 * @since  1.7.2
	 * @param  array $result Analysis result array.
	 * @param  int   $page_id Page ID being analyzed.
	 * @return array Modified result array with links.
	 */
	private function add_approval_links_to_response( $result, $page_id ) {
		if ( ! is_array( $result ) ) {
			return $result;
		}

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-respira-duplicator.php';

		// Check if this page has duplicates.
		$duplicates = Respira_Duplicator::get_duplicates( $page_id );
		$has_duplicates = ! empty( $duplicates );

		// Get approvals page URL.
		$approvals_url = admin_url( 'admin.php?page=respira-approvals' );

		// Initialize links array if it doesn't exist.
		if ( ! isset( $result['links'] ) ) {
			$result['links'] = array();
		}

		// Always include the approvals page URL.
		$result['links']['approve_edits'] = $approvals_url;

		// Add next steps if duplicates exist.
		if ( $has_duplicates ) {
			if ( ! isset( $result['next_steps'] ) ) {
				$result['next_steps'] = array();
			}

			$duplicate_count = count( $duplicates );
			$result['next_steps'][] = sprintf(
				/* translators: 1: number of duplicates, 2: approvals page URL */
				__( 'Review and approve %1$d duplicate page(s) in WordPress admin: %2$s', 'respira-for-wordpress' ),
				$duplicate_count,
				$approvals_url
			);

			// Add duplicate IDs for reference.
			$result['duplicate_ids'] = array_map(
				function( $duplicate ) {
					return $duplicate->ID;
				},
				$duplicates
			);
		}

		return $result;
	}

	/**
	 * Analyze page performance.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function analyze_performance( $request ) {
		$page_id = $request->get_param( 'page_id' );

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/analyzers/class-respira-performance-analyzer.php';
		$analyzer = new Respira_Performance_Analyzer();
		$result   = $analyzer->analyze_performance( $page_id );

		// Add approval links to response.
		$result = $this->add_approval_links_to_response( $result, $page_id );

		return new WP_REST_Response( $result, 200 );
	}

	/**
	 * Get Core Web Vitals.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function get_core_web_vitals( $request ) {
		$page_id = $request->get_param( 'page_id' );

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/analyzers/class-respira-performance-analyzer.php';
		$analyzer = new Respira_Performance_Analyzer();
		$result   = $analyzer->get_core_web_vitals( $page_id );

		// Add approval links to response.
		$result = $this->add_approval_links_to_response( $result, $page_id );

		return new WP_REST_Response( $result, 200 );
	}

	/**
	 * Analyze images on a page.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function analyze_images( $request ) {
		$page_id = $request->get_param( 'page_id' );

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/analyzers/class-respira-performance-analyzer.php';
		$analyzer = new Respira_Performance_Analyzer();
		$result   = $analyzer->analyze_images( $page_id );

		// Add approval links to response.
		$result = $this->add_approval_links_to_response( $result, $page_id );

		return new WP_REST_Response( $result, 200 );
	}

	/**
	 * Analyze SEO for a page.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function analyze_seo( $request ) {
		$page_id = $request->get_param( 'page_id' );

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/analyzers/class-respira-seo-analyzer.php';
		$analyzer = new Respira_SEO_Analyzer();
		$result   = $analyzer->analyze_seo( $page_id );

		// Add approval links to response.
		$result = $this->add_approval_links_to_response( $result, $page_id );

		return new WP_REST_Response( $result, 200 );
	}

	/**
	 * Check SEO issues for a page.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function check_seo_issues( $request ) {
		$page_id = $request->get_param( 'page_id' );

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/analyzers/class-respira-seo-analyzer.php';
		$analyzer = new Respira_SEO_Analyzer();
		$result   = $analyzer->check_seo_issues( $page_id );

		// Add approval links to response.
		$result = $this->add_approval_links_to_response( $result, $page_id );

		return new WP_REST_Response( $result, 200 );
	}

	/**
	 * Analyze readability for a page.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function analyze_readability( $request ) {
		$page_id = $request->get_param( 'page_id' );

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/analyzers/class-respira-seo-analyzer.php';
		$analyzer = new Respira_SEO_Analyzer();
		$result   = $analyzer->analyze_readability( $page_id );

		// Add approval links to response.
		$result = $this->add_approval_links_to_response( $result, $page_id );

		return new WP_REST_Response( $result, 200 );
	}

	/**
	 * Analyze AEO for a page.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function analyze_aeo( $request ) {
		$page_id = $request->get_param( 'page_id' );

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/analyzers/class-respira-aeo-analyzer.php';
		$analyzer = new Respira_AEO_Analyzer();
		$result   = $analyzer->analyze_aeo( $page_id );

		// Add approval links to response.
		$result = $this->add_approval_links_to_response( $result, $page_id );

		return new WP_REST_Response( $result, 200 );
	}

	/**
	 * Check structured data for a page.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function check_structured_data( $request ) {
		$page_id = $request->get_param( 'page_id' );

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/analyzers/class-respira-aeo-analyzer.php';
		$analyzer = new Respira_AEO_Analyzer();
		$result   = $analyzer->check_structured_data( $page_id );

		// Add approval links to response.
		$result = $this->add_approval_links_to_response( $result, $page_id );

		return new WP_REST_Response( $result, 200 );
	}

	/**
	 * List all plugins.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function list_plugins( $request ) {
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-respira-plugin-manager.php';
		$manager = new Respira_Plugin_Manager();
		$result  = $manager->list_plugins();

		return new WP_REST_Response( $result, 200 );
	}

	/**
	 * Install a plugin.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function install_plugin( $request ) {
		$slug_or_url = $request->get_param( 'slug_or_url' );
		$source      = $request->get_param( 'source' );

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-respira-plugin-manager.php';
		$manager = new Respira_Plugin_Manager();
		$result  = $manager->install_plugin( $slug_or_url, $source );

		return new WP_REST_Response( $result, $result['success'] ? 200 : 400 );
	}

	/**
	 * Activate a plugin.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function activate_plugin( $request ) {
		$slug = $request->get_param( 'slug' );

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-respira-plugin-manager.php';
		$manager = new Respira_Plugin_Manager();
		$result  = $manager->activate_plugin( $slug );

		return new WP_REST_Response( $result, $result['success'] ? 200 : 400 );
	}

	/**
	 * Deactivate a plugin.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function deactivate_plugin( $request ) {
		$slug = $request->get_param( 'slug' );

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-respira-plugin-manager.php';
		$manager = new Respira_Plugin_Manager();
		$result  = $manager->deactivate_plugin( $slug );

		return new WP_REST_Response( $result, $result['success'] ? 200 : 400 );
	}

	/**
	 * Update a plugin.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function update_plugin( $request ) {
		$slug = $request->get_param( 'slug' );

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-respira-plugin-manager.php';
		$manager = new Respira_Plugin_Manager();
		$result  = $manager->update_plugin( $slug );

		return new WP_REST_Response( $result, $result['success'] ? 200 : 400 );
	}

	/**
	 * Delete a plugin.
	 *
	 * @since  1.0.0
	 * @param  WP_REST_Request $request The request object.
	 * @return WP_REST_Response
	 */
	public function delete_plugin( $request ) {
		$slug = $request->get_param( 'slug' );

		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-respira-plugin-manager.php';
		$manager = new Respira_Plugin_Manager();
		$result  = $manager->delete_plugin( $slug );

		return new WP_REST_Response( $result, $result['success'] ? 200 : 400 );
	}

}
