<?php
/**
 * Brizy validator.
 *
 * Validates Brizy element structure before injection.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/brizy-intelligence
 * @since      1.4.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-validator-base.php';

/**
 * Brizy validator class.
 *
 * @since 1.4.0
 */
class Respira_Brizy_Validator extends Respira_Builder_Validator_Base {

	/**
	 * Validate a layout structure.
	 *
	 * @since  1.4.0
	 * @param  array $content Content structure to validate.
	 * @return array Validation result with 'valid' boolean and 'errors' array.
	 */
	public function validate_layout( $content ) {
		$this->clear_errors();

		if ( ! is_array( $content ) ) {
			$this->add_error( __( 'Content must be an array.', 'respira-for-wordpress' ) );
			return array(
				'valid'  => false,
				'errors' => $this->get_errors(),
			);
		}

		// Define nesting rules: Section > Row > Column > Elements.
		$nesting_rules = array(
			'Section' => array( 'Row', 'Wrapper' ),
			'Row'     => array( 'Column' ),
			'Column'  => array(
				'Text',
				'RichText',
				'Image',
				'Button',
				'Icon',
				'Spacer',
				'Line',
				'Map',
				'Video',
				'SoundCloud',
				'Counter',
				'Countdown',
				'ProgressBar',
				'Tabs',
				'Accordion',
				'ImageGallery',
				'Menu',
				'MenuSimple',
				'Form2',
				'IconText',
				'Html',
				'Wrapper',
			),
			'Wrapper' => array(
				'Text',
				'RichText',
				'Image',
				'Button',
				'Icon',
				'Spacer',
				'Line',
				'Map',
				'Video',
				'SoundCloud',
				'Counter',
				'Countdown',
				'ProgressBar',
				'Tabs',
				'Accordion',
				'ImageGallery',
				'Menu',
				'MenuSimple',
				'Form2',
				'IconText',
				'Html',
				'Row',
			),
		);

		// Validate structure nesting.
		$this->validate_nesting( $content, $nesting_rules );

		// Validate each element.
		foreach ( $content as $element ) {
			$this->validate_element( $element );
		}

		return array(
			'valid'  => empty( $this->errors ),
			'errors' => $this->get_errors(),
		);
	}

	/**
	 * Validate a single element.
	 *
	 * @since  1.4.0
	 * @param  array $element Element structure.
	 */
	private function validate_element( $element ) {
		// Check required fields.
		if ( ! isset( $element['type'] ) ) {
			$this->add_error( __( 'Element must have a type.', 'respira-for-wordpress' ) );
			return;
		}

		$element_type = $element['type'];

		// Validate element type is known.
		$known_element = Respira_Brizy_Element_Registry::get_element( $element_type );
		if ( ! $known_element ) {
			// Unknown element - allow it (may be from a plugin or newer version).
			// But add a warning in debug mode.
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( "Respira: Unknown Brizy element type: {$element_type}" );
			}
		}

		// Validate value/attributes if present.
		if ( isset( $element['value'] ) ) {
			if ( ! is_array( $element['value'] ) ) {
				$this->add_error(
					sprintf(
						/* translators: %s: element type */
						__( 'Element value must be an array for type %s.', 'respira-for-wordpress' ),
						$element_type
					)
				);
			} else {
				$this->validate_element_properties( $element_type, $element['value'] );
			}
		}

		// Validate attributes (alternative to value).
		if ( isset( $element['attributes'] ) ) {
			if ( ! is_array( $element['attributes'] ) ) {
				$this->add_error(
					sprintf(
						/* translators: %s: element type */
						__( 'Element attributes must be an array for type %s.', 'respira-for-wordpress' ),
						$element_type
					)
				);
			} else {
				$this->validate_element_properties( $element_type, $element['attributes'] );
			}
		}

		// Validate nested elements recursively.
		if ( ! empty( $element['children'] ) && is_array( $element['children'] ) ) {
			foreach ( $element['children'] as $nested_element ) {
				$this->validate_element( $nested_element );
			}
		}

		// Validate items array (used by container elements).
		if ( ! empty( $element['items'] ) && is_array( $element['items'] ) ) {
			foreach ( $element['items'] as $item ) {
				$this->validate_element( $item );
			}
		}

		// Validate columns array (used by Row).
		if ( ! empty( $element['columns'] ) && is_array( $element['columns'] ) ) {
			foreach ( $element['columns'] as $column ) {
				$this->validate_element( $column );
			}
		}
	}

	/**
	 * Validate element properties.
	 *
	 * @since  1.4.0
	 * @param  string $element_type Element type.
	 * @param  array  $props        Element properties.
	 */
	private function validate_element_properties( $element_type, $props ) {
		$element = Respira_Brizy_Element_Registry::get_element( $element_type );
		if ( ! $element ) {
			return; // Unknown element, skip validation.
		}

		// Get expected properties from registry.
		$expected_properties = isset( $element['properties'] ) ? $element['properties'] : array();

		// Validate each property.
		foreach ( $props as $prop_name => $prop_value ) {
			// If property is expected, validate its format.
			if ( in_array( $prop_name, $expected_properties, true ) ) {
				$this->validate_property_format( $prop_name, $prop_value, $element_type );
			}
			// Allow unknown properties (may be from newer versions or custom extensions).
		}
	}

	/**
	 * Validate property format.
	 *
	 * @since  1.4.0
	 * @param  string $prop_name   Property name.
	 * @param  mixed  $prop_value  Property value.
	 * @param  string $element_type Element type.
	 */
	private function validate_property_format( $prop_name, $prop_value, $element_type ) {
		// Validate based on property name patterns.
		if ( strpos( $prop_name, 'color' ) !== false || strpos( $prop_name, 'Color' ) !== false ) {
			if ( ! $this->validate_color( $prop_value ) ) {
				$this->add_error(
					sprintf(
						/* translators: 1: property name, 2: element type */
						__( 'Invalid color format for %1$s in element %2$s. Use hex format (#RRGGBB).', 'respira-for-wordpress' ),
						$prop_name,
						$element_type
					)
				);
			}
		} elseif ( strpos( $prop_name, 'url' ) !== false || strpos( $prop_name, 'Src' ) !== false || $prop_name === 'link' ) {
			if ( ! empty( $prop_value ) && ! $this->validate_url( $prop_value ) ) {
				$this->add_error(
					sprintf(
						/* translators: 1: property name, 2: element type */
						__( 'Invalid URL format for %1$s in element %2$s.', 'respira-for-wordpress' ),
						$prop_name,
						$element_type
					)
				);
			}
		} elseif ( strpos( $prop_name, 'width' ) !== false || strpos( $prop_name, 'Width' ) !== false ||
				   strpos( $prop_name, 'height' ) !== false || strpos( $prop_name, 'Height' ) !== false ||
				   strpos( $prop_name, 'size' ) !== false || $prop_name === 'zoom' ) {
			if ( ! is_numeric( $prop_value ) ) {
				$this->add_error(
					sprintf(
						/* translators: 1: property name, 2: element type */
						__( 'Invalid numeric value for %1$s in element %2$s.', 'respira-for-wordpress' ),
						$prop_name,
						$element_type
					)
				);
			}
		}

		// Validate specific properties.
		if ( $prop_name === 'html' && ! is_string( $prop_value ) ) {
			$this->add_error(
				sprintf(
					/* translators: %s: element type */
					__( 'HTML property must be a string in element %s.', 'respira-for-wordpress' ),
					$element_type
				)
			);
		}

		if ( ( $prop_name === 'items' || $prop_name === 'columns' || $prop_name === 'tabs' || $prop_name === 'fields' ) && ! is_array( $prop_value ) ) {
			$this->add_error(
				sprintf(
					/* translators: 1: property name, 2: element type */
					__( 'Property %1$s must be an array in element %2$s.', 'respira-for-wordpress' ),
					$prop_name,
					$element_type
				)
			);
		}
	}

	/**
	 * Validate JSON structure.
	 *
	 * @since  1.4.0
	 * @param  mixed $data Data to validate.
	 * @return bool  True if valid JSON structure.
	 */
	public function validate_json_structure( $data ) {
		if ( is_string( $data ) ) {
			// Try to decode JSON string.
			$decoded = json_decode( $data, true );
			if ( json_last_error() !== JSON_ERROR_NONE ) {
				$this->add_error(
					sprintf(
						/* translators: %s: JSON error message */
						__( 'Invalid JSON: %s', 'respira-for-wordpress' ),
						json_last_error_msg()
					)
				);
				return false;
			}
			return true;
		} elseif ( is_array( $data ) ) {
			// Already an array, that's fine.
			return true;
		}

		$this->add_error( __( 'Data must be a JSON string or array.', 'respira-for-wordpress' ) );
		return false;
	}
}
