<?php
/**
 * Brizy element registry.
 *
 * Dynamic element detection from Brizy's registry.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/brizy-intelligence
 * @since      1.4.0
 */

/**
 * Brizy element registry class.
 *
 * @since 1.4.0
 */
class Respira_Brizy_Element_Registry {

	/**
	 * Get all registered elements with metadata.
	 *
	 * @since  1.4.0
	 * @return array Array of element information.
	 */
	public static function get_all_elements() {
		// Check cache first.
		$cached = get_transient( 'respira_brizy_elements' );
		if ( false !== $cached ) {
			return $cached;
		}

		$element_list = array();

		// Try to detect elements from Brizy's internal registry.
		if ( class_exists( 'Brizy_Editor_API' ) ) {
			$element_list = self::detect_from_api();
		}

		// Fallback: use known element types from Brizy documentation.
		if ( empty( $element_list ) ) {
			$element_list = self::get_known_elements();
		}

		// Cache for 24 hours.
		set_transient( 'respira_brizy_elements', $element_list, DAY_IN_SECONDS );

		return $element_list;
	}

	/**
	 * Detect elements from Brizy API (if available).
	 *
	 * @since  1.4.0
	 * @return array Array of detected elements.
	 */
	private static function detect_from_api() {
		$elements = array();

		// Brizy has a limited public API - attempt to access element definitions.
		// This is based on reverse-engineering Brizy's structure.
		try {
			// Check for Brizy's element configuration.
			if ( class_exists( 'Brizy_Config' ) && method_exists( 'Brizy_Config', 'getElementTypes' ) ) {
				$element_types = Brizy_Config::getElementTypes();
				foreach ( $element_types as $type ) {
					$elements[] = array(
						'name'        => $type,
						'title'       => ucfirst( $type ),
						'type'        => $type,
						'description' => '',
						'properties'  => array(),
					);
				}
			}
		} catch ( Exception $e ) {
			// Silently fall back to known elements.
		}

		return $elements;
	}

	/**
	 * Get known Brizy elements from documentation.
	 *
	 * @since  1.4.0
	 * @return array Array of known elements.
	 */
	private static function get_known_elements() {
		return array(
			array(
				'name'        => 'Text',
				'title'       => 'Text',
				'type'        => 'Text',
				'description' => 'Text element with rich formatting options',
				'properties'  => array( 'text', 'typography', 'color' ),
			),
			array(
				'name'        => 'RichText',
				'title'       => 'Rich Text',
				'type'        => 'RichText',
				'description' => 'Rich text editor element',
				'properties'  => array( 'text', 'typography', 'color', 'link' ),
			),
			array(
				'name'        => 'Image',
				'title'       => 'Image',
				'type'        => 'Image',
				'description' => 'Image element with styling options',
				'properties'  => array( 'imageSrc', 'imageWidth', 'imageHeight', 'alt' ),
			),
			array(
				'name'        => 'Button',
				'title'       => 'Button',
				'type'        => 'Button',
				'description' => 'Button element with link and styling',
				'properties'  => array( 'text', 'link', 'bgColorHex', 'colorHex' ),
			),
			array(
				'name'        => 'Icon',
				'title'       => 'Icon',
				'type'        => 'Icon',
				'description' => 'Icon element from icon libraries',
				'properties'  => array( 'name', 'type', 'size', 'colorHex' ),
			),
			array(
				'name'        => 'Spacer',
				'title'       => 'Spacer',
				'type'        => 'Spacer',
				'description' => 'Spacing element for layout',
				'properties'  => array( 'height' ),
			),
			array(
				'name'        => 'Line',
				'title'       => 'Line',
				'type'        => 'Line',
				'description' => 'Horizontal line divider',
				'properties'  => array( 'width', 'borderWidth', 'borderColorHex' ),
			),
			array(
				'name'        => 'Map',
				'title'       => 'Map',
				'type'        => 'Map',
				'description' => 'Google Maps embed',
				'properties'  => array( 'address', 'zoom' ),
			),
			array(
				'name'        => 'Video',
				'title'       => 'Video',
				'type'        => 'Video',
				'description' => 'Video embed (YouTube, Vimeo)',
				'properties'  => array( 'video', 'coverImageSrc', 'ratio' ),
			),
			array(
				'name'        => 'SoundCloud',
				'title'       => 'SoundCloud',
				'type'        => 'SoundCloud',
				'description' => 'SoundCloud embed',
				'properties'  => array( 'url' ),
			),
			array(
				'name'        => 'Counter',
				'title'       => 'Counter',
				'type'        => 'Counter',
				'description' => 'Animated counter element',
				'properties'  => array( 'start', 'end', 'duration' ),
			),
			array(
				'name'        => 'Countdown',
				'title'       => 'Countdown',
				'type'        => 'Countdown',
				'description' => 'Countdown timer',
				'properties'  => array( 'date', 'timezone' ),
			),
			array(
				'name'        => 'ProgressBar',
				'title'       => 'Progress Bar',
				'type'        => 'ProgressBar',
				'description' => 'Animated progress bar',
				'properties'  => array( 'percentage', 'showPercentage', 'bgColorHex', 'barColorHex' ),
			),
			array(
				'name'        => 'Tabs',
				'title'       => 'Tabs',
				'type'        => 'Tabs',
				'description' => 'Tabbed content container',
				'properties'  => array( 'tabs' ),
			),
			array(
				'name'        => 'Accordion',
				'title'       => 'Accordion',
				'type'        => 'Accordion',
				'description' => 'Accordion/collapsible content',
				'properties'  => array( 'items' ),
			),
			array(
				'name'        => 'ImageGallery',
				'title'       => 'Image Gallery',
				'type'        => 'ImageGallery',
				'description' => 'Image gallery with lightbox',
				'properties'  => array( 'items', 'gridColumn' ),
			),
			array(
				'name'        => 'Menu',
				'title'       => 'Menu',
				'type'        => 'Menu',
				'description' => 'Navigation menu',
				'properties'  => array( 'menuSelected' ),
			),
			array(
				'name'        => 'MenuSimple',
				'title'       => 'Simple Menu',
				'type'        => 'MenuSimple',
				'description' => 'Simple navigation menu',
				'properties'  => array( 'menuSelected' ),
			),
			array(
				'name'        => 'Form2',
				'title'       => 'Form',
				'type'        => 'Form2',
				'description' => 'Contact form with fields',
				'properties'  => array( 'fields', 'submitText' ),
			),
			array(
				'name'        => 'IconText',
				'title'       => 'Icon with Text',
				'type'        => 'IconText',
				'description' => 'Icon paired with text',
				'properties'  => array( 'iconName', 'text' ),
			),
			array(
				'name'        => 'Row',
				'title'       => 'Row',
				'type'        => 'Row',
				'description' => 'Layout row container',
				'properties'  => array( 'columns' ),
			),
			array(
				'name'        => 'Column',
				'title'       => 'Column',
				'type'        => 'Column',
				'description' => 'Layout column',
				'properties'  => array( 'width', 'items' ),
			),
			array(
				'name'        => 'Section',
				'title'       => 'Section',
				'type'        => 'Section',
				'description' => 'Section container',
				'properties'  => array( 'items' ),
			),
			array(
				'name'        => 'Wrapper',
				'title'       => 'Wrapper',
				'type'        => 'Wrapper',
				'description' => 'Generic wrapper container',
				'properties'  => array( 'items' ),
			),
			array(
				'name'        => 'Html',
				'title'       => 'Custom HTML',
				'type'        => 'Html',
				'description' => 'Custom HTML/CSS/JS code',
				'properties'  => array( 'html' ),
			),
		);
	}

	/**
	 * Get element by name.
	 *
	 * @since  1.4.0
	 * @param  string $element_name Element name.
	 * @return array|null Element information or null if not found.
	 */
	public static function get_element( $element_name ) {
		$elements = self::get_all_elements();

		foreach ( $elements as $element ) {
			if ( $element['name'] === $element_name || $element['type'] === $element_name ) {
				return $element;
			}
		}

		return null;
	}

	/**
	 * Clear element cache.
	 *
	 * @since 1.4.0
	 */
	public static function clear_cache() {
		delete_transient( 'respira_brizy_elements' );
	}
}
