<?php
/**
 * Bricks Builder validator.
 *
 * Validates Bricks Builder element structure before injection.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/bricks-intelligence
 * @since      1.4.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-validator-base.php';

/**
 * Bricks Builder validator class.
 *
 * @since 1.4.0
 */
class Respira_Bricks_Validator extends Respira_Builder_Validator_Base {

	/**
	 * Validate a layout structure.
	 *
	 * @since  1.4.0
	 * @param  array $content Content structure to validate.
	 * @return array Validation result with 'valid' boolean and 'errors' array.
	 */
	public function validate_layout( $content ) {
		$this->clear_errors();

		if ( ! is_array( $content ) ) {
			$this->add_error( __( 'Content must be an array.', 'respira-for-wordpress' ) );
			return array(
				'valid'  => false,
				'errors' => $this->get_errors(),
			);
		}

		// Collect all element IDs to check for uniqueness.
		$element_ids = array();

		// Validate each element in the layout.
		foreach ( $content as $element ) {
			$this->validate_element( $element, $element_ids );
		}

		// Check for duplicate IDs.
		$id_counts = array_count_values( $element_ids );
		foreach ( $id_counts as $id => $count ) {
			if ( $count > 1 ) {
				$this->add_error(
					sprintf(
						/* translators: %s: element ID */
						__( 'Duplicate element ID found: %s', 'respira-for-wordpress' ),
						$id
					)
				);
			}
		}

		return array(
			'valid'  => empty( $this->errors ),
			'errors' => $this->get_errors(),
		);
	}

	/**
	 * Validate a single element.
	 *
	 * @since  1.4.0
	 * @param  array $element     Element structure.
	 * @param  array &$element_ids Reference to array of element IDs.
	 */
	private function validate_element( $element, &$element_ids = array() ) {
		// Check required fields.
		if ( ! isset( $element['id'] ) ) {
			$this->add_error( __( 'Element must have an id.', 'respira-for-wordpress' ) );
			return;
		}

		if ( ! isset( $element['name'] ) ) {
			$this->add_error(
				sprintf(
					/* translators: %s: element ID */
					__( 'Element %s must have a name (type).', 'respira-for-wordpress' ),
					$element['id']
				)
			);
			return;
		}

		// Track element ID.
		$element_ids[] = $element['id'];

		$element_type = $element['name'];

		// Validate element type is known.
		$known_element = Respira_Bricks_Element_Registry::get_element( $element_type );
		if ( ! $known_element ) {
			// Unknown element - allow it (may be from extension).
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( "Respira: Unknown Bricks element type: {$element_type}" );
			}
		}

		// Validate settings if present.
		if ( isset( $element['settings'] ) ) {
			if ( ! is_array( $element['settings'] ) ) {
				$this->add_error(
					sprintf(
						/* translators: %s: element ID */
						__( 'Element settings must be an array for element %s.', 'respira-for-wordpress' ),
						$element['id']
					)
				);
			} else {
				$this->validate_element_settings( $element_type, $element['settings'], $element['id'] );
			}
		}

		// Validate children recursively.
		if ( isset( $element['children'] ) ) {
			if ( ! is_array( $element['children'] ) ) {
				$this->add_error(
					sprintf(
						/* translators: %s: element ID */
						__( 'Element children must be an array for element %s.', 'respira-for-wordpress' ),
						$element['id']
					)
				);
			} else {
				foreach ( $element['children'] as $child ) {
					$this->validate_element( $child, $element_ids );
				}
			}
		}
	}

	/**
	 * Validate element settings.
	 *
	 * @since  1.4.0
	 * @param  string $element_type Element type.
	 * @param  array  $settings     Element settings.
	 * @param  string $element_id   Element ID.
	 */
	private function validate_element_settings( $element_type, $settings, $element_id ) {
		foreach ( $settings as $setting_name => $setting_value ) {
			// Validate color properties (including those prefixed with _).
			if ( strpos( $setting_name, 'color' ) !== false || strpos( $setting_name, 'Color' ) !== false || $setting_name === '_color' ) {
				if ( is_string( $setting_value ) && ! empty( $setting_value ) && ! $this->validate_color( $setting_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: setting name, 2: element type, 3: element ID */
							__( 'Invalid color format for %1$s in %2$s element %3$s.', 'respira-for-wordpress' ),
							$setting_name,
							$element_type,
							$element_id
						)
					);
				}
			}

			// Validate URL properties.
			if ( ( $setting_name === 'videoUrl' || strpos( $setting_name, 'Url' ) !== false ) && is_string( $setting_value ) ) {
				if ( ! empty( $setting_value ) && ! $this->validate_url( $setting_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: setting name, 2: element type, 3: element ID */
							__( 'Invalid URL format for %1$s in %2$s element %3$s.', 'respira-for-wordpress' ),
							$setting_name,
							$element_type,
							$element_id
						)
					);
				}
			}

			// Validate link objects.
			if ( $setting_name === 'link' && is_array( $setting_value ) ) {
				if ( isset( $setting_value['url'] ) && ! empty( $setting_value['url'] ) && ! $this->validate_url( $setting_value['url'] ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: element type, 2: element ID */
							__( 'Invalid URL in link object for %1$s element %2$s.', 'respira-for-wordpress' ),
							$element_type,
							$element_id
						)
					);
				}
			}

			// Validate numeric properties.
			if ( in_array( $setting_name, array( 'zoom', 'speed', 'start', 'end', 'duration', 'value', 'max', 'postsPerPage', 'activeTab', 'slidesPerView', 'mobileBreakpoint' ), true ) ) {
				if ( ! is_numeric( $setting_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: setting name, 2: element type, 3: element ID */
							__( 'Invalid numeric value for %1$s in %2$s element %3$s.', 'respira-for-wordpress' ),
							$setting_name,
							$element_type,
							$element_id
						)
					);
				}
			}

			// Validate boolean properties.
			if ( in_array( $setting_name, array( 'autoplay', 'loop', 'controls', 'multipleOpen', 'firstOpen', 'navigation', 'pagination', 'mobileMenu', 'showValue', 'highlight', 'lineNumbers' ), true ) ) {
				if ( ! is_bool( $setting_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: setting name, 2: element type, 3: element ID */
							__( 'Property %1$s must be a boolean in %2$s element %3$s.', 'respira-for-wordpress' ),
							$setting_name,
							$element_type,
							$element_id
						)
					);
				}
			}

			// Validate array properties.
			if ( in_array( $setting_name, array( 'items', 'images', 'fields', 'features', 'markers' ), true ) ) {
				if ( ! is_array( $setting_value ) ) {
					$this->add_error(
						sprintf(
							/* translators: 1: setting name, 2: element type, 3: element ID */
							__( 'Property %1$s must be an array in %2$s element %3$s.', 'respira-for-wordpress' ),
							$setting_name,
							$element_type,
							$element_id
						)
					);
				}
			}
		}
	}

	/**
	 * Validate Bricks data structure.
	 *
	 * @since  1.4.0
	 * @param  mixed $data Data to validate.
	 * @return bool True if valid structure.
	 */
	public function validate_data_structure( $data ) {
		if ( ! is_array( $data ) ) {
			$this->add_error( __( 'Bricks data must be an array.', 'respira-for-wordpress' ) );
			return false;
		}

		// Each item should be an element with id and name.
		foreach ( $data as $element ) {
			if ( ! is_array( $element ) ) {
				$this->add_error( __( 'Each element must be an array.', 'respira-for-wordpress' ) );
				return false;
			}

			if ( ! isset( $element['id'] ) ) {
				$this->add_error( __( 'Element missing id.', 'respira-for-wordpress' ) );
				return false;
			}

			if ( ! isset( $element['name'] ) ) {
				$this->add_error(
					sprintf(
						/* translators: %s: element ID */
						__( 'Element %s missing name (type).', 'respira-for-wordpress' ),
						$element['id']
					)
				);
				return false;
			}
		}

		return true;
	}
}
