<?php
/**
 * Bricks Builder element schema generator.
 *
 * Generates schemas for all registered elements with property definitions.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/bricks-intelligence
 * @since      1.4.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-schema-base.php';

/**
 * Bricks Builder element schema class.
 *
 * @since 1.4.0
 */
class Respira_Bricks_Element_Schema extends Respira_Builder_Schema_Base {

	/**
	 * Constructor.
	 *
	 * @since 1.4.0
	 */
	public function __construct() {
		$version = defined( 'BRICKS_VERSION' ) ? BRICKS_VERSION : 'unknown';
		parent::__construct( 'bricks', $version );
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since  1.4.0
	 * @param  array $elements_used Optional. Array of element names used on the page.
	 * @return array Builder schema with element information.
	 */
	public function get_builder_schema( $elements_used = array() ) {
		$schema = array(
			'builder'            => 'bricks',
			'builder_version'    => $this->builder_version,
			'available_elements' => array(),
			'quick_reference'    => $this->get_quick_reference(),
			'structure_notes'    => $this->get_structure_notes(),
		);

		// Get schemas for elements used on the page (or all elements if none specified).
		if ( empty( $elements_used ) ) {
			$all_elements  = Respira_Bricks_Element_Registry::get_all_elements();
			$elements_used = array_column( $all_elements, 'type' );
		}

		foreach ( $elements_used as $element_name ) {
			$element = Respira_Bricks_Element_Registry::get_element( $element_name );
			if ( ! $element ) {
				continue;
			}

			$element_schema = array(
				'title'       => $element['title'],
				'description' => $element['description'],
				'type'        => $element['type'],
				'category'    => isset( $element['category'] ) ? $element['category'] : 'general',
				'properties'  => array(),
			);

			// Generate property schemas.
			if ( ! empty( $element['properties'] ) ) {
				foreach ( $element['properties'] as $property ) {
					$element_schema['properties'][ $property ] = $this->get_property_schema( $property, $element['type'] );
				}
			}

			$schema['available_elements'][ $element['type'] ] = $element_schema;
		}

		return $schema;
	}

	/**
	 * Get property schema for an element property.
	 *
	 * @since  1.4.0
	 * @param  string $property     Property name.
	 * @param  string $element_type Element type.
	 * @return array Property schema.
	 */
	private function get_property_schema( $property, $element_type ) {
		// Bricks uses underscore prefix for styling properties.
		$property_map = array(
			// Content properties.
			'text'        => array( 'type' => 'string', 'format' => 'html', 'example' => '<p>Text content</p>' ),
			'html'        => array( 'type' => 'string', 'format' => 'html', 'example' => '<div>HTML</div>' ),
			'code'        => array( 'type' => 'string', 'format' => 'code', 'example' => 'console.log("Hello");' ),
			'title'       => array( 'type' => 'string', 'format' => 'text', 'example' => 'Title' ),
			'description' => array( 'type' => 'string', 'format' => 'text', 'example' => 'Description' ),
			'content'     => array( 'type' => 'string', 'format' => 'html', 'example' => '<p>Content</p>' ),

			// Link properties.
			'link'        => array( 'type' => 'object', 'format' => 'link_object', 'example' => array( 'url' => '#', 'newTab' => false ) ),

			// Tag properties.
			'tag'         => array( 'type' => 'string', 'format' => 'enum', 'example' => 'div', 'options' => array( 'div', 'section', 'header', 'footer', 'article', 'aside', 'nav', 'main', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'p', 'span' ) ),

			// Image properties.
			'image'       => array( 'type' => 'object', 'format' => 'image_object', 'example' => array( 'id' => 123, 'url' => 'https://example.com/image.jpg' ) ),
			'images'      => array( 'type' => 'array', 'format' => 'image_array', 'example' => array() ),
			'size'        => array( 'type' => 'string', 'format' => 'image_size', 'example' => 'large' ),
			'alt'         => array( 'type' => 'string', 'format' => 'text', 'example' => 'Image alt text' ),
			'objectFit'   => array( 'type' => 'string', 'format' => 'enum', 'example' => 'cover', 'options' => array( 'cover', 'contain', 'fill', 'none', 'scale-down' ) ),

			// Video properties.
			'videoType'   => array( 'type' => 'string', 'format' => 'enum', 'example' => 'youtube', 'options' => array( 'youtube', 'vimeo', 'media' ) ),
			'videoUrl'    => array( 'type' => 'string', 'format' => 'url', 'example' => 'https://youtube.com/watch?v=abc' ),
			'videoId'     => array( 'type' => 'string', 'format' => 'text', 'example' => 'abc123' ),
			'controls'    => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => true ),
			'autoplay'    => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => false ),
			'loop'        => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => false ),

			// Audio properties.
			'audioFile'   => array( 'type' => 'object', 'format' => 'media_object', 'example' => array( 'id' => 123, 'url' => 'https://example.com/audio.mp3' ) ),

			// Icon properties.
			'icon'        => array( 'type' => 'object', 'format' => 'icon_object', 'example' => array( 'library' => 'fontawesome', 'icon' => 'fa-heart' ) ),

			// Button properties.
			'type'        => array( 'type' => 'string', 'format' => 'enum', 'example' => 'primary' ),

			// List properties.
			'items'       => array( 'type' => 'array', 'format' => 'items_array', 'example' => array() ),

			// Language property.
			'language'    => array( 'type' => 'string', 'format' => 'text', 'example' => 'javascript' ),
			'lineNumbers' => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => true ),

			// SVG properties.
			'svg'         => array( 'type' => 'string', 'format' => 'svg', 'example' => '<svg>...</svg>' ),
			'width'       => array( 'type' => 'string', 'format' => 'size', 'example' => '100px' ),
			'height'      => array( 'type' => 'string', 'format' => 'size', 'example' => '100px' ),

			// Accordion/Tabs properties.
			'multipleOpen' => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => false ),
			'firstOpen'   => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => true ),
			'layout'      => array( 'type' => 'string', 'format' => 'enum', 'example' => 'horizontal' ),
			'activeTab'   => array( 'type' => 'number', 'format' => 'number', 'example' => 0 ),

			// Toggle properties.
			'open'        => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => false ),

			// Slider/Carousel properties.
			'speed'       => array( 'type' => 'number', 'format' => 'milliseconds', 'example' => 1000 ),
			'navigation'  => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => true ),
			'pagination'  => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => true ),
			'slidesPerView' => array( 'type' => 'number', 'format' => 'number', 'example' => 1 ),

			// Modal/Popup properties.
			'trigger'     => array( 'type' => 'string', 'format' => 'enum', 'example' => 'click' ),
			'animation'   => array( 'type' => 'string', 'format' => 'enum', 'example' => 'fade' ),

			// Form properties.
			'fields'      => array( 'type' => 'array', 'format' => 'form_fields', 'example' => array() ),
			'submitText'  => array( 'type' => 'string', 'format' => 'text', 'example' => 'Submit' ),
			'action'      => array( 'type' => 'string', 'format' => 'url', 'example' => '' ),
			'method'      => array( 'type' => 'string', 'format' => 'enum', 'example' => 'POST', 'options' => array( 'POST', 'GET' ) ),

			// Posts properties.
			'query'       => array( 'type' => 'object', 'format' => 'query_object', 'example' => array() ),
			'postsPerPage' => array( 'type' => 'number', 'format' => 'number', 'example' => 10 ),
			'orderBy'     => array( 'type' => 'string', 'format' => 'enum', 'example' => 'date' ),
			'order'       => array( 'type' => 'string', 'format' => 'enum', 'example' => 'DESC', 'options' => array( 'ASC', 'DESC' ) ),

			// Menu properties.
			'menu'        => array( 'type' => 'string', 'format' => 'menu_id', 'example' => 'primary' ),
			'mobileMenu'  => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => true ),
			'mobileBreakpoint' => array( 'type' => 'number', 'format' => 'pixels', 'example' => 768 ),

			// Sidebar properties.
			'sidebar'     => array( 'type' => 'string', 'format' => 'sidebar_id', 'example' => 'sidebar-1' ),

			// Map properties.
			'address'     => array( 'type' => 'string', 'format' => 'text', 'example' => '123 Main St, City' ),
			'zoom'        => array( 'type' => 'number', 'format' => 'number', 'example' => 12 ),
			'markers'     => array( 'type' => 'array', 'format' => 'markers_array', 'example' => array() ),

			// Counter properties.
			'start'       => array( 'type' => 'number', 'format' => 'number', 'example' => 0 ),
			'end'         => array( 'type' => 'number', 'format' => 'number', 'example' => 100 ),
			'duration'    => array( 'type' => 'number', 'format' => 'milliseconds', 'example' => 2000 ),
			'prefix'      => array( 'type' => 'string', 'format' => 'text', 'example' => '$' ),
			'suffix'      => array( 'type' => 'string', 'format' => 'text', 'example' => '+' ),

			// Progress bar properties.
			'value'       => array( 'type' => 'number', 'format' => 'number', 'example' => 75 ),
			'max'         => array( 'type' => 'number', 'format' => 'number', 'example' => 100 ),
			'showValue'   => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => true ),

			// Pricing table properties.
			'price'       => array( 'type' => 'string', 'format' => 'text', 'example' => '$99' ),
			'features'    => array( 'type' => 'array', 'format' => 'features_array', 'example' => array() ),
			'button'      => array( 'type' => 'object', 'format' => 'button_object', 'example' => array() ),
			'highlight'   => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => false ),

			// Style properties (prefixed with _).
			'_width'      => array( 'type' => 'string', 'format' => 'size', 'example' => '100%' ),
			'_height'     => array( 'type' => 'string', 'format' => 'size', 'example' => 'auto' ),
			'_padding'    => array( 'type' => 'object', 'format' => 'spacing', 'example' => array( 'top' => '20px' ) ),
			'_margin'     => array( 'type' => 'object', 'format' => 'spacing', 'example' => array( 'top' => '20px' ) ),
			'_background' => array( 'type' => 'object', 'format' => 'background', 'example' => array( 'color' => '#ffffff' ) ),
			'_border'     => array( 'type' => 'object', 'format' => 'border', 'example' => array() ),
			'_typography' => array( 'type' => 'object', 'format' => 'typography', 'example' => array() ),
			'_color'      => array( 'type' => 'string', 'format' => 'color', 'example' => '#000000' ),
			'_fontSize'   => array( 'type' => 'string', 'format' => 'size', 'example' => '16px' ),
			'_aspectRatio' => array( 'type' => 'string', 'format' => 'ratio', 'example' => '16/9' ),
		);

		// Return the schema for this property, or a generic schema if not found.
		if ( isset( $property_map[ $property ] ) ) {
			return $property_map[ $property ];
		}

		// Generic property schema.
		return array(
			'type'    => 'string',
			'format'  => 'text',
			'example' => 'value',
		);
	}

	/**
	 * Get structure notes for Bricks Builder.
	 *
	 * @since  1.4.0
	 * @return array Structure notes.
	 */
	private function get_structure_notes() {
		return array(
			'hierarchy'    => 'Container → Element → Nested Elements',
			'data_format'  => 'Bricks stores data as JSON arrays in postmeta',
			'meta_key'     => '_bricks_page_content',
			'elements'     => 'Each element has id, name (type), settings, and optional children',
			'nesting'      => 'Elements can be nested via children array',
			'styling'      => 'Style properties are prefixed with underscore (_background, _padding, etc.)',
			'note'         => 'Element IDs must be unique within the layout',
		);
	}
}
