<?php
/**
 * Beaver Builder module registry.
 *
 * Dynamic module detection from Beaver Builder's registry.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/beaver-intelligence
 * @since      1.4.0
 */

/**
 * Beaver Builder module registry class.
 *
 * @since 1.4.0
 */
class Respira_Beaver_Module_Registry {

	/**
	 * Get all registered modules with metadata.
	 *
	 * @since  1.4.0
	 * @return array Array of module information.
	 */
	public static function get_all_modules() {
		// Check cache first.
		$cached = get_transient( 'respira_beaver_modules' );
		if ( false !== $cached ) {
			return $cached;
		}

		$module_list = array();

		// Try to detect modules from Beaver Builder's registry.
		if ( class_exists( 'FLBuilderModel' ) && method_exists( 'FLBuilderModel', 'get_modules' ) ) {
			$module_list = self::detect_from_api();
		}

		// Fallback: use known module types from Beaver Builder documentation.
		if ( empty( $module_list ) ) {
			$module_list = self::get_known_modules();
		}

		// Cache for 24 hours.
		set_transient( 'respira_beaver_modules', $module_list, DAY_IN_SECONDS );

		return $module_list;
	}

	/**
	 * Detect modules from Beaver Builder API (if available).
	 *
	 * @since  1.4.0
	 * @return array Array of detected modules.
	 */
	private static function detect_from_api() {
		$modules = array();

		try {
			// Beaver Builder provides FLBuilderModel::get_modules() to get all modules.
			if ( class_exists( 'FLBuilderModel' ) ) {
				$bb_modules = FLBuilderModel::get_modules();

				foreach ( $bb_modules as $slug => $module_class ) {
					if ( class_exists( $module_class ) ) {
						$module_instance = new $module_class();
						$modules[]       = array(
							'name'        => isset( $module_instance->name ) ? $module_instance->name : $slug,
							'title'       => isset( $module_instance->name ) ? $module_instance->name : $slug,
							'type'        => $slug,
							'description' => isset( $module_instance->description ) ? $module_instance->description : '',
							'category'    => isset( $module_instance->category ) ? $module_instance->category : 'standard',
							'properties'  => array(),
						);
					}
				}
			}
		} catch ( Exception $e ) {
			// Silently fall back to known modules.
		}

		return $modules;
	}

	/**
	 * Get known Beaver Builder modules from documentation.
	 *
	 * @since  1.4.0
	 * @return array Array of known modules.
	 */
	private static function get_known_modules() {
		return array(
			// Basic modules.
			array(
				'name'        => 'Heading',
				'title'       => 'Heading',
				'type'        => 'heading',
				'description' => 'Customizable heading with styling options',
				'category'    => 'basic',
				'properties'  => array( 'heading', 'tag', 'font_family', 'font_size', 'color', 'link' ),
			),
			array(
				'name'        => 'Photo',
				'title'       => 'Photo',
				'type'        => 'photo',
				'description' => 'Single photo with alignment and linking options',
				'category'    => 'basic',
				'properties'  => array( 'photo', 'photo_src', 'align', 'link', 'link_url', 'caption' ),
			),
			array(
				'name'        => 'Rich Text',
				'title'       => 'Rich Text',
				'type'        => 'rich-text',
				'description' => 'WYSIWYG text editor',
				'category'    => 'basic',
				'properties'  => array( 'text' ),
			),
			array(
				'name'        => 'HTML',
				'title'       => 'HTML',
				'type'        => 'html',
				'description' => 'Custom HTML code',
				'category'    => 'basic',
				'properties'  => array( 'html' ),
			),
			array(
				'name'        => 'Button',
				'title'       => 'Button',
				'type'        => 'button',
				'description' => 'Customizable call-to-action button',
				'category'    => 'basic',
				'properties'  => array( 'text', 'link', 'link_target', 'style', 'width', 'align', 'bg_color', 'text_color', 'icon' ),
			),
			array(
				'name'        => 'Icon',
				'title'       => 'Icon',
				'type'        => 'icon',
				'description' => 'Single icon with styling',
				'category'    => 'basic',
				'properties'  => array( 'icon', 'size', 'align', 'color', 'link' ),
			),
			array(
				'name'        => 'Separator',
				'title'       => 'Separator',
				'type'        => 'separator',
				'description' => 'Horizontal separator line',
				'category'    => 'basic',
				'properties'  => array( 'style', 'color', 'height', 'width' ),
			),
			array(
				'name'        => 'Spacer',
				'title'       => 'Spacer',
				'type'        => 'spacer',
				'description' => 'Vertical spacing element',
				'category'    => 'basic',
				'properties'  => array( 'size' ),
			),

			// Media modules.
			array(
				'name'        => 'Video',
				'title'       => 'Video',
				'type'        => 'video',
				'description' => 'Video embed (YouTube, Vimeo, self-hosted)',
				'category'    => 'media',
				'properties'  => array( 'video_type', 'video', 'video_embed', 'poster' ),
			),
			array(
				'name'        => 'Audio',
				'title'       => 'Audio',
				'type'        => 'audio',
				'description' => 'Audio player',
				'category'    => 'media',
				'properties'  => array( 'audio', 'audio_src' ),
			),
			array(
				'name'        => 'Gallery',
				'title'       => 'Gallery',
				'type'        => 'gallery',
				'description' => 'Photo gallery with lightbox',
				'category'    => 'media',
				'properties'  => array( 'photos', 'layout', 'photo_spacing', 'show_captions', 'click_action' ),
			),
			array(
				'name'        => 'Slideshow',
				'title'       => 'Slideshow',
				'type'        => 'slideshow',
				'description' => 'Image slideshow',
				'category'    => 'media',
				'properties'  => array( 'photos', 'transition', 'speed', 'pause' ),
			),

			// Advanced modules.
			array(
				'name'        => 'Accordion',
				'title'       => 'Accordion',
				'type'        => 'accordion',
				'description' => 'Collapsible accordion items',
				'category'    => 'advanced',
				'properties'  => array( 'items', 'open_first', 'collapse', 'border_style', 'border_color' ),
			),
			array(
				'name'        => 'Tabs',
				'title'       => 'Tabs',
				'type'        => 'tabs',
				'description' => 'Tabbed content container',
				'category'    => 'advanced',
				'properties'  => array( 'tabs', 'layout', 'border_style', 'border_color' ),
			),
			array(
				'name'        => 'Contact Form',
				'title'       => 'Contact Form',
				'type'        => 'contact-form',
				'description' => 'Contact form with custom fields',
				'category'    => 'advanced',
				'properties'  => array( 'name_toggle', 'email_toggle', 'phone_toggle', 'subject_toggle', 'message_toggle', 'success_action', 'success_message' ),
			),
			array(
				'name'        => 'Testimonials',
				'title'       => 'Testimonials',
				'type'        => 'testimonials',
				'description' => 'Testimonial slider or grid',
				'category'    => 'advanced',
				'properties'  => array( 'source', 'layout', 'auto_play', 'pause' ),
			),
			array(
				'name'        => 'Pricing Table',
				'title'       => 'Pricing Table',
				'type'        => 'pricing-table',
				'description' => 'Pricing box with features',
				'category'    => 'advanced',
				'properties'  => array( 'title', 'price', 'duration', 'features', 'button_text', 'button_url', 'highlight' ),
			),
			array(
				'name'        => 'Subscribe Form',
				'title'       => 'Subscribe Form',
				'type'        => 'subscribe-form',
				'description' => 'Email subscription form',
				'category'    => 'advanced',
				'properties'  => array( 'service', 'success_action', 'button_text', 'layout' ),
			),
			array(
				'name'        => 'Icon Group',
				'title'       => 'Icon Group',
				'type'        => 'icon-group',
				'description' => 'Group of icons with links',
				'category'    => 'advanced',
				'properties'  => array( 'icons', 'view', 'spacing' ),
			),
			array(
				'name'        => 'Callout',
				'title'       => 'Callout',
				'type'        => 'callout',
				'description' => 'Call-to-action box with heading, text, and button',
				'category'    => 'advanced',
				'properties'  => array( 'title', 'text', 'button_text', 'button_url', 'cta_type', 'bg_color' ),
			),

			// WordPress modules.
			array(
				'name'        => 'Posts',
				'title'       => 'Posts',
				'type'        => 'post-grid',
				'description' => 'Grid or list of WordPress posts',
				'category'    => 'posts',
				'properties'  => array( 'layout', 'post_type', 'order', 'offset', 'posts_per_page', 'post_taxonomies' ),
			),
			array(
				'name'        => 'Post Slider',
				'title'       => 'Post Slider',
				'type'        => 'post-slider',
				'description' => 'Slider of WordPress posts',
				'category'    => 'posts',
				'properties'  => array( 'post_type', 'order', 'offset', 'posts_per_page', 'slider_transition', 'slider_speed' ),
			),
			array(
				'name'        => 'Menu',
				'title'       => 'Menu',
				'type'        => 'menu',
				'description' => 'WordPress menu',
				'category'    => 'wordpress',
				'properties'  => array( 'menu', 'mobile_toggle', 'submenu_click' ),
			),
			array(
				'name'        => 'Sidebar',
				'title'       => 'Sidebar',
				'type'        => 'sidebar',
				'description' => 'WordPress widget area',
				'category'    => 'wordpress',
				'properties'  => array( 'sidebar' ),
			),

			// Social modules.
			array(
				'name'        => 'Social Buttons',
				'title'       => 'Social Buttons',
				'type'        => 'social-buttons',
				'description' => 'Social sharing buttons',
				'category'    => 'social',
				'properties'  => array( 'platforms', 'target', 'view' ),
			),

			// Layout modules.
			array(
				'name'        => 'Map',
				'title'       => 'Map',
				'type'        => 'map',
				'description' => 'Google Maps embed',
				'category'    => 'media',
				'properties'  => array( 'address', 'height' ),
			),
			array(
				'name'        => 'Number Counter',
				'title'       => 'Number Counter',
				'type'        => 'number-counter',
				'description' => 'Animated number counter',
				'category'    => 'advanced',
				'properties'  => array( 'number', 'max_number', 'speed', 'number_prefix', 'number_suffix', 'layout' ),
			),
		);
	}

	/**
	 * Get module by name or type.
	 *
	 * @since  1.4.0
	 * @param  string $module_name Module name or type.
	 * @return array|null Module information or null if not found.
	 */
	public static function get_module( $module_name ) {
		$modules = self::get_all_modules();

		foreach ( $modules as $module ) {
			if ( $module['name'] === $module_name || $module['type'] === $module_name ) {
				return $module;
			}
		}

		return null;
	}

	/**
	 * Clear module cache.
	 *
	 * @since 1.4.0
	 */
	public static function clear_cache() {
		delete_transient( 'respira_beaver_modules' );
	}
}
