<?php
/**
 * Beaver Builder layout patterns library.
 *
 * Common layout patterns for Beaver Builder modules.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/beaver-intelligence
 * @since      1.4.0
 */

/**
 * Get Beaver Builder layout patterns.
 *
 * @since  1.4.0
 * @return array Array of patterns.
 */
function respira_get_beaver_patterns() {
	return array(
		'hero-section'          => array(
			'title'       => __( 'Hero Section', 'respira-for-wordpress' ),
			'description' => __( 'Full-width hero section with heading and CTA button', 'respira-for-wordpress' ),
			'category'    => 'headers',
			'structure'   => array(
				'row_' . uniqid() => array(
					'type'     => 'row',
					'settings' => array(
						'width'    => 'full',
						'bg_color' => '0066cc',
						'padding_top' => '80',
						'padding_bottom' => '80',
					),
					'columns'  => array( 'column_' . uniqid() ),
				),
				'column_' . uniqid() => array(
					'type'     => 'column',
					'parent'   => 'row_' . uniqid(),
					'settings' => array(
						'size' => '100',
					),
					'nodes'    => array( 'heading_' . uniqid(), 'text_' . uniqid(), 'button_' . uniqid() ),
				),
				'heading_' . uniqid() => array(
					'type'     => 'module',
					'parent'   => 'column_' . uniqid(),
					'settings' => array(
						'type'    => 'heading',
						'heading' => __( 'Welcome to Our Amazing Service', 'respira-for-wordpress' ),
						'tag'     => 'h1',
						'color'   => 'ffffff',
						'align'   => 'center',
						'font_size' => '48',
					),
				),
				'text_' . uniqid() => array(
					'type'     => 'module',
					'parent'   => 'column_' . uniqid(),
					'settings' => array(
						'type' => 'rich-text',
						'text' => '<p style="text-align: center; color: #ffffff; font-size: 18px;">' . __( 'Your compelling tagline goes here', 'respira-for-wordpress' ) . '</p>',
					),
				),
				'button_' . uniqid() => array(
					'type'     => 'module',
					'parent'   => 'column_' . uniqid(),
					'settings' => array(
						'type'       => 'button',
						'text'       => __( 'Get Started', 'respira-for-wordpress' ),
						'link'       => '#',
						'align'      => 'center',
						'bg_color'   => 'ffffff',
						'text_color' => '0066cc',
					),
				),
			),
		),
		'three-column-features' => array(
			'title'       => __( 'Three Column Features', 'respira-for-wordpress' ),
			'description' => __( 'Three-column layout with icon and text features', 'respira-for-wordpress' ),
			'category'    => 'content',
			'structure'   => array(
				'row_' . uniqid() => array(
					'type'     => 'row',
					'settings' => array(
						'padding_top' => '60',
						'padding_bottom' => '60',
					),
					'columns'  => array( 'column_1_' . uniqid(), 'column_2_' . uniqid(), 'column_3_' . uniqid() ),
				),
			),
		),
		'call-to-action'        => array(
			'title'       => __( 'Call to Action', 'respira-for-wordpress' ),
			'description' => __( 'Centered CTA section with heading and button', 'respira-for-wordpress' ),
			'category'    => 'cta',
			'note'        => __( 'Complete pattern structure with callout module', 'respira-for-wordpress' ),
		),
		'testimonial'           => array(
			'title'       => __( 'Testimonial', 'respira-for-wordpress' ),
			'description' => __( 'Testimonial section with quote', 'respira-for-wordpress' ),
			'category'    => 'testimonials',
			'note'        => __( 'Use testimonials module for dynamic content', 'respira-for-wordpress' ),
		),
		'pricing-table'         => array(
			'title'       => __( 'Pricing Table', 'respira-for-wordpress' ),
			'description' => __( 'Three-column pricing comparison', 'respira-for-wordpress' ),
			'category'    => 'pricing',
			'note'        => __( 'Use pricing-table modules in three columns', 'respira-for-wordpress' ),
		),
		'image-with-text'       => array(
			'title'       => __( 'Image with Text', 'respira-for-wordpress' ),
			'description' => __( 'Two-column layout with image and text', 'respira-for-wordpress' ),
			'category'    => 'content',
			'note'        => __( 'Photo module in one column, rich text in another', 'respira-for-wordpress' ),
		),
	);
}

/**
 * Get a complete Beaver Builder row structure.
 *
 * Helper function to generate properly structured Beaver Builder rows.
 *
 * @since  1.4.0
 * @param  array $settings Row settings.
 * @param  array $columns  Array of column configurations.
 * @return array Complete row structure.
 */
function respira_beaver_create_row( $settings = array(), $columns = array() ) {
	$row_id = 'row_' . uniqid();
	$data   = array();

	// Default row settings.
	$default_settings = array(
		'width'          => 'fixed',
		'content_width'  => '1100',
		'bg_color'       => '',
		'padding_top'    => '20',
		'padding_bottom' => '20',
		'padding_left'   => '0',
		'padding_right'  => '0',
	);

	$row_settings = array_merge( $default_settings, $settings );

	// Create row.
	$column_ids       = array();
	foreach ( $columns as $column_config ) {
		$column_ids[] = 'column_' . uniqid();
	}

	$data[ $row_id ] = array(
		'type'     => 'row',
		'settings' => $row_settings,
		'columns'  => $column_ids,
	);

	// Create columns.
	foreach ( $columns as $index => $column_config ) {
		$column_id = $column_ids[ $index ];

		$column_data = array(
			'type'     => 'column',
			'parent'   => $row_id,
			'settings' => isset( $column_config['settings'] ) ? $column_config['settings'] : array( 'size' => '100' ),
			'nodes'    => array(),
		);

		// Add modules to column.
		if ( isset( $column_config['modules'] ) && is_array( $column_config['modules'] ) ) {
			foreach ( $column_config['modules'] as $module_config ) {
				$module_id                   = 'module_' . uniqid();
				$column_data['nodes'][]      = $module_id;
				$data[ $module_id ]          = array(
					'type'     => 'module',
					'parent'   => $column_id,
					'settings' => $module_config,
				);
			}
		}

		$data[ $column_id ] = $column_data;
	}

	return $data;
}

/**
 * Get a complete Beaver Builder module.
 *
 * Helper function to generate properly structured Beaver Builder modules.
 *
 * @since  1.4.0
 * @param  string $type     Module type.
 * @param  array  $settings Module settings.
 * @return array Complete module structure.
 */
function respira_beaver_create_module( $type, $settings = array() ) {
	$settings['type'] = $type;

	return array(
		'type'     => 'module',
		'settings' => $settings,
	);
}
