<?php
/**
 * SEO Analyzer
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/analyzers
 */

/**
 * Analyzes SEO factors including meta tags, content structure, and readability.
 *
 * @since 1.0.0
 */
class Respira_SEO_Analyzer {

	/**
	 * Perform comprehensive SEO analysis
	 *
	 * @since 1.0.0
	 * @param int $page_id Page ID to analyze.
	 * @return array SEO analysis results.
	 */
	public function analyze_seo( $page_id ) {
		$post = get_post( $page_id );
		if ( ! $post ) {
			return array(
				'success' => false,
				'message' => 'Page not found',
			);
		}

		$issues         = array();
		$recommendations = array();
		$metrics        = array();

		// Meta tags analysis.
		$meta_analysis  = $this->analyze_meta_tags( $page_id );
		$metrics        = array_merge( $metrics, $meta_analysis['metrics'] );
		$issues         = array_merge( $issues, $meta_analysis['issues'] );
		$recommendations = array_merge( $recommendations, $meta_analysis['recommendations'] );

		// Heading structure analysis.
		$heading_analysis = $this->analyze_headings( $post->post_content );
		$metrics         = array_merge( $metrics, $heading_analysis['metrics'] );
		$issues          = array_merge( $issues, $heading_analysis['issues'] );
		$recommendations = array_merge( $recommendations, $heading_analysis['recommendations'] );

		// Image alt text analysis.
		$image_analysis = $this->analyze_image_alt_text( $post->post_content );
		$metrics       = array_merge( $metrics, $image_analysis['metrics'] );
		$issues        = array_merge( $issues, $image_analysis['issues'] );
		$recommendations = array_merge( $recommendations, $image_analysis['recommendations'] );

		// Internal linking analysis.
		$link_analysis  = $this->analyze_internal_links( $post->post_content );
		$metrics        = array_merge( $metrics, $link_analysis['metrics'] );
		$issues         = array_merge( $issues, $link_analysis['issues'] );
		$recommendations = array_merge( $recommendations, $link_analysis['recommendations'] );

		// Content analysis.
		$content_analysis = $this->analyze_content( $post->post_content, $post->post_title );
		$metrics         = array_merge( $metrics, $content_analysis['metrics'] );
		$issues          = array_merge( $issues, $content_analysis['issues'] );
		$recommendations = array_merge( $recommendations, $content_analysis['recommendations'] );

		// Schema markup analysis.
		$schema_analysis = $this->analyze_schema_markup( $page_id );
		$metrics        = array_merge( $metrics, $schema_analysis['metrics'] );
		$issues         = array_merge( $issues, $schema_analysis['issues'] );
		$recommendations = array_merge( $recommendations, $schema_analysis['recommendations'] );

		// Calculate overall SEO score.
		$score = $this->calculate_seo_score( $issues, $metrics );
		$grade = $this->get_grade_from_score( $score );

		return array(
			'success'         => true,
			'score'           => $score,
			'grade'           => $grade,
			'issues'          => $issues,
			'recommendations' => $recommendations,
			'metrics'         => $metrics,
		);
	}

	/**
	 * Check for common SEO issues
	 *
	 * @since 1.0.0
	 * @param int $page_id Page ID to analyze.
	 * @return array SEO issues.
	 */
	public function check_seo_issues( $page_id ) {
		$post = get_post( $page_id );
		if ( ! $post ) {
			return array(
				'success' => false,
				'message' => 'Page not found',
			);
		}

		$issues = array();

		// Check for critical SEO issues.
		$meta_title = $this->get_meta_title( $page_id );
		if ( empty( $meta_title ) ) {
			$issues[] = array(
				'type'     => 'error',
				'message'  => 'Missing meta title',
				'severity' => 'critical',
				'fix'      => 'Add a meta title using an SEO plugin like Yoast or RankMath',
			);
		}

		$meta_description = $this->get_meta_description( $page_id );
		if ( empty( $meta_description ) ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'Missing meta description',
				'severity' => 'high',
				'fix'      => 'Add a meta description between 120-160 characters',
			);
		}

		// Check content length.
		$content      = strip_tags( $post->post_content );
		$word_count   = str_word_count( $content );
		if ( $word_count < 300 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'Thin content detected',
				'severity' => 'medium',
				'fix'      => sprintf( 'Add more content (current: %d words, recommended: 300+ words)', $word_count ),
			);
		}

		// Check for H1 tag.
		if ( ! preg_match( '/<h1[^>]*>/i', $post->post_content ) ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'No H1 heading found',
				'severity' => 'high',
				'fix'      => 'Add one H1 heading that describes the main topic',
			);
		}

		$score = count( $issues ) === 0 ? 100 : max( 0, 100 - ( count( $issues ) * 15 ) );
		$grade = $this->get_grade_from_score( $score );

		return array(
			'success' => true,
			'score'   => $score,
			'grade'   => $grade,
			'issues'  => $issues,
		);
	}

	/**
	 * Analyze content readability
	 *
	 * @since 1.0.0
	 * @param int $page_id Page ID to analyze.
	 * @return array Readability analysis.
	 */
	public function analyze_readability( $page_id ) {
		$post = get_post( $page_id );
		if ( ! $post ) {
			return array(
				'success' => false,
				'message' => 'Page not found',
			);
		}

		$content = strip_tags( $post->post_content );
		$issues  = array();
		$metrics = array();

		// Calculate readability metrics.
		$word_count      = str_word_count( $content );
		$sentence_count  = preg_match_all( '/[.!?]+/', $content, $matches );
		$avg_words_per_sentence = $sentence_count > 0 ? $word_count / $sentence_count : 0;

		$metrics['word_count']             = $word_count;
		$metrics['sentence_count']         = $sentence_count;
		$metrics['avg_words_per_sentence'] = round( $avg_words_per_sentence, 1 );

		// Calculate Flesch Reading Ease (simplified).
		$flesch_score              = $this->calculate_flesch_score( $content );
		$metrics['flesch_score']   = $flesch_score;
		$metrics['reading_level']  = $this->get_reading_level( $flesch_score );

		// Check for readability issues.
		if ( $avg_words_per_sentence > 25 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'Sentences are too long',
				'severity' => 'medium',
				'fix'      => 'Break long sentences into shorter ones (average: ' . round( $avg_words_per_sentence ) . ' words per sentence)',
			);
		}

		if ( $flesch_score < 60 ) {
			$issues[] = array(
				'type'     => 'info',
				'message'  => 'Content may be difficult to read',
				'severity' => 'low',
				'fix'      => 'Simplify language and use shorter sentences (Flesch score: ' . $flesch_score . ')',
			);
		}

		// Paragraph analysis.
		$paragraphs = preg_split( '/\n\s*\n/', $content );
		$long_paragraphs = 0;
		foreach ( $paragraphs as $paragraph ) {
			if ( str_word_count( $paragraph ) > 150 ) {
				$long_paragraphs++;
			}
		}

		if ( $long_paragraphs > 0 ) {
			$issues[] = array(
				'type'     => 'info',
				'message'  => sprintf( '%d long paragraphs detected', $long_paragraphs ),
				'severity' => 'low',
				'fix'      => 'Break long paragraphs into shorter ones for better readability',
			);
		}

		$score = $this->calculate_readability_score( $issues, $metrics );
		$grade = $this->get_grade_from_score( $score );

		return array(
			'success'         => true,
			'score'           => $score,
			'grade'           => $grade,
			'issues'          => $issues,
			'recommendations' => array(),
			'metrics'         => $metrics,
		);
	}

	/**
	 * Analyze meta tags
	 *
	 * @param int $page_id Page ID.
	 * @return array Meta tag analysis.
	 */
	private function analyze_meta_tags( $page_id ) {
		$issues         = array();
		$recommendations = array();
		$metrics        = array();

		// Get meta title.
		$meta_title = $this->get_meta_title( $page_id );
		$metrics['meta_title_length'] = strlen( $meta_title );

		if ( empty( $meta_title ) ) {
			$issues[] = array(
				'type'     => 'error',
				'message'  => 'Missing meta title',
				'severity' => 'critical',
				'fix'      => 'Add a meta title using an SEO plugin',
			);
		} elseif ( strlen( $meta_title ) < 30 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'Meta title too short',
				'severity' => 'medium',
				'fix'      => 'Expand meta title to 50-60 characters',
			);
		} elseif ( strlen( $meta_title ) > 60 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'Meta title too long',
				'severity' => 'medium',
				'fix'      => 'Shorten meta title to 50-60 characters',
			);
		}

		// Get meta description.
		$meta_description = $this->get_meta_description( $page_id );
		$metrics['meta_description_length'] = strlen( $meta_description );

		if ( empty( $meta_description ) ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'Missing meta description',
				'severity' => 'high',
				'fix'      => 'Add meta description between 120-160 characters',
			);
		} elseif ( strlen( $meta_description ) < 120 ) {
			$issues[] = array(
				'type'     => 'info',
				'message'  => 'Meta description too short',
				'severity' => 'low',
				'fix'      => 'Expand meta description to 120-160 characters',
			);
		} elseif ( strlen( $meta_description ) > 160 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'Meta description too long',
				'severity' => 'medium',
				'fix'      => 'Shorten meta description to 120-160 characters',
			);
		}

		// Check for Open Graph tags.
		$og_title       = get_post_meta( $page_id, '_yoast_wpseo_opengraph-title', true );
		$og_description = get_post_meta( $page_id, '_yoast_wpseo_opengraph-description', true );
		$og_image       = get_post_meta( $page_id, '_yoast_wpseo_opengraph-image', true );

		$metrics['has_og_title']       = ! empty( $og_title );
		$metrics['has_og_description'] = ! empty( $og_description );
		$metrics['has_og_image']       = ! empty( $og_image );

		if ( empty( $og_title ) && empty( $og_description ) && empty( $og_image ) ) {
			$recommendations[] = array(
				'priority' => 'medium',
				'action'   => 'Add Open Graph tags',
				'details'  => 'Add OG tags for better social media sharing',
			);
		}

		return array(
			'issues'          => $issues,
			'recommendations' => $recommendations,
			'metrics'         => $metrics,
		);
	}

	/**
	 * Analyze heading structure
	 *
	 * @param string $content Page content.
	 * @return array Heading analysis.
	 */
	private function analyze_headings( $content ) {
		$issues         = array();
		$recommendations = array();
		$metrics        = array(
			'h1_count' => 0,
			'h2_count' => 0,
			'h3_count' => 0,
			'h4_count' => 0,
			'h5_count' => 0,
			'h6_count' => 0,
		);

		// Count headings.
		for ( $i = 1; $i <= 6; $i++ ) {
			preg_match_all( "/<h{$i}[^>]*>/i", $content, $matches );
			$metrics[ "h{$i}_count" ] = count( $matches[0] );
		}

		// Check H1 usage.
		if ( $metrics['h1_count'] === 0 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'No H1 heading found',
				'severity' => 'high',
				'fix'      => 'Add one H1 heading that describes the main topic',
			);
		} elseif ( $metrics['h1_count'] > 1 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => sprintf( 'Multiple H1 headings found (%d)', $metrics['h1_count'] ),
				'severity' => 'medium',
				'fix'      => 'Use only one H1 heading per page',
			);
		}

		// Check for heading hierarchy.
		if ( $metrics['h2_count'] === 0 && ( $metrics['h3_count'] > 0 || $metrics['h4_count'] > 0 ) ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'Improper heading hierarchy',
				'severity' => 'low',
				'fix'      => 'Use H2 headings before H3/H4 headings',
			);
		}

		return array(
			'issues'          => $issues,
			'recommendations' => $recommendations,
			'metrics'         => $metrics,
		);
	}

	/**
	 * Analyze image alt text
	 *
	 * @param string $content Page content.
	 * @return array Image alt text analysis.
	 */
	private function analyze_image_alt_text( $content ) {
		$issues         = array();
		$recommendations = array();
		$metrics        = array(
			'total_images'       => 0,
			'images_with_alt'    => 0,
			'images_without_alt' => 0,
		);

		// Find all images.
		preg_match_all( '/<img[^>]+>/i', $content, $matches );
		$images = $matches[0];

		$metrics['total_images'] = count( $images );

		foreach ( $images as $img_tag ) {
			if ( preg_match( '/alt=["\']([^"\']*)["\']/', $img_tag, $alt_match ) ) {
				if ( ! empty( $alt_match[1] ) ) {
					$metrics['images_with_alt']++;
				} else {
					$metrics['images_without_alt']++;
				}
			} else {
				$metrics['images_without_alt']++;
			}
		}

		if ( $metrics['images_without_alt'] > 0 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => sprintf( '%d images missing alt text', $metrics['images_without_alt'] ),
				'severity' => 'medium',
				'fix'      => 'Add descriptive alt text to all images',
			);
		}

		$alt_coverage = $metrics['total_images'] > 0 ? ( $metrics['images_with_alt'] / $metrics['total_images'] ) * 100 : 100;
		$metrics['alt_text_coverage'] = round( $alt_coverage, 1 );

		return array(
			'issues'          => $issues,
			'recommendations' => $recommendations,
			'metrics'         => $metrics,
		);
	}

	/**
	 * Analyze internal links
	 *
	 * @param string $content Page content.
	 * @return array Internal link analysis.
	 */
	private function analyze_internal_links( $content ) {
		$issues         = array();
		$recommendations = array();
		$metrics        = array(
			'total_links'    => 0,
			'internal_links' => 0,
			'external_links' => 0,
		);

		// Find all links.
		preg_match_all( '/<a[^>]+href=["\']([^"\']*)["\'][^>]*>/i', $content, $matches );
		$links = $matches[1];

		$metrics['total_links'] = count( $links );

		$site_url = get_site_url();
		foreach ( $links as $link ) {
			if ( strpos( $link, $site_url ) !== false || strpos( $link, '/' ) === 0 ) {
				$metrics['internal_links']++;
			} else {
				$metrics['external_links']++;
			}
		}

		if ( $metrics['internal_links'] < 2 ) {
			$recommendations[] = array(
				'priority' => 'low',
				'action'   => 'Add more internal links',
				'details'  => 'Link to related content on your site to improve SEO and user experience',
			);
		}

		return array(
			'issues'          => $issues,
			'recommendations' => $recommendations,
			'metrics'         => $metrics,
		);
	}

	/**
	 * Analyze content
	 *
	 * @param string $content Page content.
	 * @param string $title Page title.
	 * @return array Content analysis.
	 */
	private function analyze_content( $content, $title ) {
		$issues         = array();
		$recommendations = array();
		$metrics        = array();

		// Remove HTML tags for text analysis.
		$text = strip_tags( $content );

		// Word count.
		$word_count = str_word_count( $text );
		$metrics['word_count'] = $word_count;

		if ( $word_count < 300 ) {
			$issues[] = array(
				'type'     => 'warning',
				'message'  => 'Content too short',
				'severity' => 'medium',
				'fix'      => sprintf( 'Add more content (current: %d words, recommended: 300+ words)', $word_count ),
			);
		}

		// Character count.
		$char_count = strlen( $text );
		$metrics['character_count'] = $char_count;

		// Keyword density (simple check for title words in content).
		$title_words = str_word_count( strtolower( $title ), 1 );
		$content_lower = strtolower( $text );
		$keyword_mentions = 0;
		foreach ( $title_words as $word ) {
			if ( strlen( $word ) > 3 ) { // Only count significant words.
				$keyword_mentions += substr_count( $content_lower, $word );
			}
		}
		$metrics['title_word_mentions'] = $keyword_mentions;

		if ( $word_count > 0 ) {
			$keyword_density = ( $keyword_mentions / $word_count ) * 100;
			$metrics['keyword_density'] = round( $keyword_density, 2 );

			if ( $keyword_density < 1 ) {
				$recommendations[] = array(
					'priority' => 'low',
					'action'   => 'Improve keyword usage',
					'details'  => 'Include title keywords more naturally throughout the content',
				);
			}
		}

		return array(
			'issues'          => $issues,
			'recommendations' => $recommendations,
			'metrics'         => $metrics,
		);
	}

	/**
	 * Analyze schema markup
	 *
	 * @param int $page_id Page ID.
	 * @return array Schema analysis.
	 */
	private function analyze_schema_markup( $page_id ) {
		$issues         = array();
		$recommendations = array();
		$metrics        = array(
			'has_schema' => false,
		);

		// Check for common schema markup meta.
		$schema_types = array(
			'_yoast_wpseo_schema',
			'_genesis_schema',
			'schema_type',
		);

		foreach ( $schema_types as $meta_key ) {
			$schema_data = get_post_meta( $page_id, $meta_key, true );
			if ( ! empty( $schema_data ) ) {
				$metrics['has_schema'] = true;
				break;
			}
		}

		if ( ! $metrics['has_schema'] ) {
			$recommendations[] = array(
				'priority' => 'medium',
				'action'   => 'Add schema markup',
				'details'  => 'Add structured data (JSON-LD) for better search engine understanding',
			);
		}

		return array(
			'issues'          => $issues,
			'recommendations' => $recommendations,
			'metrics'         => $metrics,
		);
	}

	/**
	 * Get meta title
	 *
	 * @param int $page_id Page ID.
	 * @return string Meta title.
	 */
	private function get_meta_title( $page_id ) {
		// Try Yoast.
		$yoast_title = get_post_meta( $page_id, '_yoast_wpseo_title', true );
		if ( ! empty( $yoast_title ) ) {
			return $yoast_title;
		}

		// Try RankMath.
		$rankmath_title = get_post_meta( $page_id, 'rank_math_title', true );
		if ( ! empty( $rankmath_title ) ) {
			return $rankmath_title;
		}

		// Try All in One SEO.
		$aioseo_title = get_post_meta( $page_id, '_aioseo_title', true );
		if ( ! empty( $aioseo_title ) ) {
			return $aioseo_title;
		}

		// Fallback to post title.
		$post = get_post( $page_id );
		return $post ? $post->post_title : '';
	}

	/**
	 * Get meta description
	 *
	 * @param int $page_id Page ID.
	 * @return string Meta description.
	 */
	private function get_meta_description( $page_id ) {
		// Try Yoast.
		$yoast_desc = get_post_meta( $page_id, '_yoast_wpseo_metadesc', true );
		if ( ! empty( $yoast_desc ) ) {
			return $yoast_desc;
		}

		// Try RankMath.
		$rankmath_desc = get_post_meta( $page_id, 'rank_math_description', true );
		if ( ! empty( $rankmath_desc ) ) {
			return $rankmath_desc;
		}

		// Try All in One SEO.
		$aioseo_desc = get_post_meta( $page_id, '_aioseo_description', true );
		if ( ! empty( $aioseo_desc ) ) {
			return $aioseo_desc;
		}

		return '';
	}

	/**
	 * Calculate Flesch Reading Ease score (simplified)
	 *
	 * @param string $text Text to analyze.
	 * @return int Flesch score.
	 */
	private function calculate_flesch_score( $text ) {
		$word_count     = str_word_count( $text );
		$sentence_count = preg_match_all( '/[.!?]+/', $text, $matches );
		$syllable_count = $this->count_syllables( $text );

		if ( $word_count === 0 || $sentence_count === 0 ) {
			return 0;
		}

		$asl = $word_count / $sentence_count;
		$asw = $syllable_count / $word_count;

		$flesch = 206.835 - ( 1.015 * $asl ) - ( 84.6 * $asw );

		return max( 0, min( 100, round( $flesch ) ) );
	}

	/**
	 * Count syllables in text (simplified)
	 *
	 * @param string $text Text to analyze.
	 * @return int Syllable count.
	 */
	private function count_syllables( $text ) {
		$words = str_word_count( strtolower( $text ), 1 );
		$syllables = 0;

		foreach ( $words as $word ) {
			$syllables += $this->count_word_syllables( $word );
		}

		return $syllables;
	}

	/**
	 * Count syllables in a word (simplified)
	 *
	 * @param string $word Word to analyze.
	 * @return int Syllable count.
	 */
	private function count_word_syllables( $word ) {
		$word = strtolower( trim( $word ) );
		if ( strlen( $word ) <= 3 ) {
			return 1;
		}

		$syllables = 0;
		$vowels    = array( 'a', 'e', 'i', 'o', 'u', 'y' );
		$previous_was_vowel = false;

		for ( $i = 0; $i < strlen( $word ); $i++ ) {
			$is_vowel = in_array( $word[ $i ], $vowels, true );

			if ( $is_vowel && ! $previous_was_vowel ) {
				$syllables++;
			}

			$previous_was_vowel = $is_vowel;
		}

		// Adjust for silent 'e'.
		if ( substr( $word, -1 ) === 'e' ) {
			$syllables--;
		}

		return max( 1, $syllables );
	}

	/**
	 * Get reading level from Flesch score
	 *
	 * @param int $score Flesch score.
	 * @return string Reading level.
	 */
	private function get_reading_level( $score ) {
		if ( $score >= 90 ) {
			return 'Very Easy (5th grade)';
		} elseif ( $score >= 80 ) {
			return 'Easy (6th grade)';
		} elseif ( $score >= 70 ) {
			return 'Fairly Easy (7th grade)';
		} elseif ( $score >= 60 ) {
			return 'Standard (8th-9th grade)';
		} elseif ( $score >= 50 ) {
			return 'Fairly Difficult (10th-12th grade)';
		} elseif ( $score >= 30 ) {
			return 'Difficult (College)';
		} else {
			return 'Very Difficult (College graduate)';
		}
	}

	/**
	 * Calculate SEO score
	 *
	 * @param array $issues Issues found.
	 * @param array $metrics Metrics collected.
	 * @return int Score (0-100).
	 */
	private function calculate_seo_score( $issues, $metrics ) {
		$score = 100;

		// Deduct points for issues.
		foreach ( $issues as $issue ) {
			switch ( $issue['severity'] ) {
				case 'critical':
					$score -= 25;
					break;
				case 'high':
					$score -= 15;
					break;
				case 'medium':
					$score -= 10;
					break;
				case 'low':
					$score -= 5;
					break;
			}
		}

		// Bonus points for good metrics.
		if ( isset( $metrics['alt_text_coverage'] ) && $metrics['alt_text_coverage'] >= 90 ) {
			$score += 5;
		}

		if ( isset( $metrics['word_count'] ) && $metrics['word_count'] >= 500 ) {
			$score += 5;
		}

		return max( 0, min( 100, $score ) );
	}

	/**
	 * Calculate readability score
	 *
	 * @param array $issues Issues found.
	 * @param array $metrics Metrics collected.
	 * @return int Score (0-100).
	 */
	private function calculate_readability_score( $issues, $metrics ) {
		$score = 100;

		// Deduct points for issues.
		foreach ( $issues as $issue ) {
			switch ( $issue['severity'] ) {
				case 'high':
					$score -= 20;
					break;
				case 'medium':
					$score -= 15;
					break;
				case 'low':
					$score -= 5;
					break;
			}
		}

		// Bonus for good Flesch score.
		if ( isset( $metrics['flesch_score'] ) && $metrics['flesch_score'] >= 60 ) {
			$score += 10;
		}

		return max( 0, min( 100, $score ) );
	}

	/**
	 * Get letter grade from score
	 *
	 * @param int $score Score (0-100).
	 * @return string Letter grade.
	 */
	private function get_grade_from_score( $score ) {
		if ( $score >= 90 ) {
			return 'A';
		} elseif ( $score >= 80 ) {
			return 'B';
		} elseif ( $score >= 70 ) {
			return 'C';
		} elseif ( $score >= 60 ) {
			return 'D';
		} else {
			return 'F';
		}
	}
}
