<?php
/**
 * WPBakery validator.
 *
 * Validates WPBakery shortcode structure before injection.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/wpbakery-intelligence
 * @since      1.3.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-validator-base.php';

/**
 * WPBakery validator class.
 *
 * @since 1.3.0
 */
class Respira_WPBakery_Validator extends Respira_Builder_Validator_Base {

	/**
	 * Validate a layout structure.
	 *
	 * @since  1.3.0
	 * @param  array $content Content structure to validate.
	 * @return array Validation result with 'valid' boolean and 'errors' array.
	 */
	public function validate_layout( $content ) {
		$this->clear_errors();

		if ( ! is_array( $content ) ) {
			$this->add_error( __( 'Content must be an array.', 'respira-for-wordpress' ) );
			return array(
				'valid'  => false,
				'errors' => $this->get_errors(),
			);
		}

		// Define nesting rules: rows > columns > elements.
		$nesting_rules = array(
			'vc_row'    => array( 'vc_column', 'vc_column_inner' ),
			'vc_column' => array( 'vc_column_text', 'vc_single_image', 'vc_raw_html' ),
			'vc_column_inner' => array( 'vc_column_text', 'vc_single_image', 'vc_raw_html' ),
		);

		// Validate structure nesting.
		$this->validate_nesting( $content, $nesting_rules );

		// Validate each shortcode.
		foreach ( $content as $shortcode ) {
			$this->validate_shortcode( $shortcode );
		}

		return array(
			'valid'  => empty( $this->errors ),
			'errors' => $this->get_errors(),
		);
	}

	/**
	 * Validate a single shortcode.
	 *
	 * @since  1.3.0
	 * @param  array $shortcode Shortcode structure.
	 */
	private function validate_shortcode( $shortcode ) {
		// Check required fields.
		if ( ! isset( $shortcode['type'] ) ) {
			$this->add_error( __( 'Shortcode must have a type.', 'respira-for-wordpress' ) );
			return;
		}

		$shortcode_name = $shortcode['type'];

		// Validate shortcode name format.
		if ( ! preg_match( '/^vc_\w+$/', $shortcode_name ) ) {
			// Allow non-vc_ shortcodes (may be custom).
		}

		// Validate attributes if present.
		if ( isset( $shortcode['attributes'] ) && is_array( $shortcode['attributes'] ) ) {
			$this->validate_shortcode_attributes( $shortcode_name, $shortcode['attributes'] );
		}

		// Validate nested shortcodes recursively.
		if ( ! empty( $shortcode['children'] ) && is_array( $shortcode['children'] ) ) {
			foreach ( $shortcode['children'] as $nested_shortcode ) {
				$this->validate_shortcode( $nested_shortcode );
			}
		}
	}

	/**
	 * Validate shortcode attributes.
	 *
	 * @since  1.3.0
	 * @param  string $shortcode_name Shortcode name.
	 * @param  array  $attrs          Shortcode attributes.
	 */
	private function validate_shortcode_attributes( $shortcode_name, $attrs ) {
		$shortcode = Respira_WPBakery_Shortcode_Registry::get_shortcode( $shortcode_name );
		if ( ! $shortcode || empty( $shortcode['params'] ) ) {
			return; // Unknown shortcode or no params defined.
		}

		// Validate each attribute against shortcode params.
		foreach ( $attrs as $attr_name => $attr_value ) {
			$param = null;
			foreach ( $shortcode['params'] as $p ) {
				if ( isset( $p['param_name'] ) && $p['param_name'] === $attr_name ) {
					$param = $p;
					break;
				}
			}

			if ( ! $param ) {
				// Unknown attribute - allow it (may be custom).
				continue;
			}

			$expected_type = isset( $param['type'] ) ? $param['type'] : 'string';

			if ( ! $this->validate_attribute_format( $attr_value, $expected_type, $attr_name ) ) {
				$this->add_error(
					sprintf(
						/* translators: 1: attribute name, 2: shortcode name */
						__( 'Invalid attribute format for %1$s in shortcode %2$s', 'respira-for-wordpress' ),
						$attr_name,
						$shortcode_name
					)
				);
			}
		}
	}
}

