<?php
/**
 * WPBakery shortcode schema generator.
 *
 * Generates schemas for all registered shortcodes with parameter definitions.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/wpbakery-intelligence
 * @since      1.3.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-schema-base.php';

/**
 * WPBakery shortcode schema class.
 *
 * @since 1.3.0
 */
class Respira_WPBakery_Shortcode_Schema extends Respira_Builder_Schema_Base {

	/**
	 * Constructor.
	 *
	 * @since 1.3.0
	 */
	public function __construct() {
		$version = defined( 'WPB_VC_VERSION' ) ? WPB_VC_VERSION : 'unknown';
		parent::__construct( 'wpbakery', $version );
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since  1.3.0
	 * @param  array $shortcodes_used Optional. Array of shortcode names used on the page.
	 * @return array Builder schema with shortcode information.
	 */
	public function get_builder_schema( $shortcodes_used = array() ) {
		$schema = array(
			'builder'          => 'wpbakery',
			'builder_version'  => $this->builder_version,
			'available_shortcodes' => array(),
			'quick_reference'  => $this->get_quick_reference(),
		);

		// Get schemas for shortcodes used on the page (or all shortcodes if none specified).
		if ( empty( $shortcodes_used ) ) {
			$all_shortcodes = Respira_WPBakery_Shortcode_Registry::get_all_shortcodes();
			$shortcodes_used = array_column( $all_shortcodes, 'name' );
		}

		foreach ( $shortcodes_used as $shortcode_name ) {
			$shortcode = Respira_WPBakery_Shortcode_Registry::get_shortcode( $shortcode_name );
			if ( ! $shortcode ) {
				continue;
			}

			$shortcode_schema = array(
				'title'       => $shortcode['name'],
				'description' => $shortcode['description'],
				'category'    => $shortcode['category'],
				'attributes'  => array(),
			);

			// Extract parameter schemas.
			if ( ! empty( $shortcode['params'] ) ) {
				foreach ( $shortcode['params'] as $param ) {
					$param_name = isset( $param['param_name'] ) ? $param['param_name'] : '';
					if ( empty( $param_name ) ) {
						continue;
					}

					$type = isset( $param['type'] ) ? $this->normalize_type( $param['type'] ) : 'string';
					$format = $this->get_format_hint( $type, $param_name );

					$shortcode_schema['attributes'][ $param_name ] = array(
						'type'    => $type,
						'format'  => $format,
						'default' => isset( $param['value'] ) ? $param['value'] : null,
						'example' => $this->get_example_value( $type, $param_name ),
					);

					// Add label and description.
					if ( isset( $param['heading'] ) ) {
						$shortcode_schema['attributes'][ $param_name ]['label'] = $param['heading'];
					}
					if ( isset( $param['description'] ) ) {
						$shortcode_schema['attributes'][ $param_name ]['description'] = $param['description'];
					}

					// Add options for dropdown/select params.
					if ( isset( $param['value'] ) && is_array( $param['value'] ) ) {
						$shortcode_schema['attributes'][ $param_name ]['options'] = $param['value'];
					}
				}
			}

			$schema['available_shortcodes'][ $shortcode_name ] = $shortcode_schema;
		}

		return $schema;
	}
}

