<?php
/**
 * WPBakery shortcode registry.
 *
 * Dynamic shortcode detection from WPBakery's registry.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/wpbakery-intelligence
 * @since      1.3.0
 */

/**
 * WPBakery shortcode registry class.
 *
 * @since 1.3.0
 */
class Respira_WPBakery_Shortcode_Registry {

	/**
	 * Get all registered shortcodes with metadata.
	 *
	 * @since  1.3.0
	 * @return array Array of shortcode information.
	 */
	public static function get_all_shortcodes() {
		if ( ! class_exists( 'WPBMap' ) && ! function_exists( 'vc_map' ) ) {
			return array();
		}

		// Check cache first.
		$cached = get_transient( 'respira_wpbakery_shortcodes' );
		if ( false !== $cached ) {
			return $cached;
		}

		$shortcode_list = array();

		// Get all shortcodes from WPBakery registry.
		if ( class_exists( 'WPBMap' ) && method_exists( 'WPBMap', 'getAllShortCodes' ) ) {
			$all_shortcodes = WPBMap::getAllShortCodes();

			foreach ( $all_shortcodes as $shortcode_name => $shortcode_config ) {
				$shortcode_info = array(
					'name'        => $shortcode_name,
					'base'        => isset( $shortcode_config['base'] ) ? $shortcode_config['base'] : $shortcode_name,
					'name'        => isset( $shortcode_config['name'] ) ? $shortcode_config['name'] : $shortcode_name,
					'category'    => isset( $shortcode_config['category'] ) ? $shortcode_config['category'] : 'general',
					'description' => isset( $shortcode_config['description'] ) ? $shortcode_config['description'] : '',
					'params'       => isset( $shortcode_config['params'] ) ? $shortcode_config['params'] : array(),
				);

				$shortcode_list[] = $shortcode_info;
			}
		} else {
			// Fallback: scan for vc_* shortcodes in content.
			$shortcode_list = self::scan_shortcodes();
		}

		// Cache for 24 hours.
		set_transient( 'respira_wpbakery_shortcodes', $shortcode_list, DAY_IN_SECONDS );

		return $shortcode_list;
	}

	/**
	 * Scan for WPBakery shortcodes in posts.
	 *
	 * @since  1.3.0
	 * @return array Array of found shortcodes.
	 */
	private static function scan_shortcodes() {
		global $wpdb;

		$shortcodes = array();
		$found = $wpdb->get_col(
			"SELECT DISTINCT post_content FROM {$wpdb->posts} WHERE post_content LIKE '%[vc_%' LIMIT 100"
		);

		$shortcode_names = array();
		foreach ( $found as $content ) {
			preg_match_all( '/\[vc_(\w+)/', $content, $matches );
			if ( ! empty( $matches[1] ) ) {
				$shortcode_names = array_merge( $shortcode_names, $matches[1] );
			}
		}

		$shortcode_names = array_unique( $shortcode_names );

		foreach ( $shortcode_names as $name ) {
			$shortcodes[] = array(
				'name'        => 'vc_' . $name,
				'base'        => 'vc_' . $name,
				'category'    => 'general',
				'description' => '',
				'params'      => array(),
			);
		}

		return $shortcodes;
	}

	/**
	 * Get shortcode by name.
	 *
	 * @since  1.3.0
	 * @param  string $shortcode_name Shortcode name.
	 * @return array|null Shortcode information or null if not found.
	 */
	public static function get_shortcode( $shortcode_name ) {
		$shortcodes = self::get_all_shortcodes();

		foreach ( $shortcodes as $shortcode ) {
			if ( $shortcode['name'] === $shortcode_name || $shortcode['base'] === $shortcode_name ) {
				return $shortcode;
			}
		}

		return null;
	}

	/**
	 * Clear shortcode cache.
	 *
	 * @since 1.3.0
	 */
	public static function clear_cache() {
		delete_transient( 'respira_wpbakery_shortcodes' );
	}
}

