<?php
/**
 * Visual Composer element schema generator.
 *
 * Generates schemas for all registered elements with property definitions.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/visual-composer-intelligence
 * @since      1.4.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-schema-base.php';

/**
 * Visual Composer element schema class.
 *
 * @since 1.4.0
 */
class Respira_Visual_Composer_Element_Schema extends Respira_Builder_Schema_Base {

	/**
	 * Constructor.
	 *
	 * @since 1.4.0
	 */
	public function __construct() {
		$version = defined( 'WPB_VC_VERSION' ) ? WPB_VC_VERSION : 'unknown';
		parent::__construct( 'visual_composer', $version );
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since  1.4.0
	 * @param  array $elements_used Optional. Array of element names used on the page.
	 * @return array Builder schema with element information.
	 */
	public function get_builder_schema( $elements_used = array() ) {
		$schema = array(
			'builder'            => 'visual_composer',
			'builder_version'    => $this->builder_version,
			'available_elements' => array(),
			'quick_reference'    => $this->get_quick_reference(),
			'structure_notes'    => $this->get_structure_notes(),
		);

		// Get schemas for elements used on the page (or all elements if none specified).
		if ( empty( $elements_used ) ) {
			$all_elements  = Respira_Visual_Composer_Element_Registry::get_all_elements();
			$elements_used = array_column( $all_elements, 'type' );
		}

		foreach ( $elements_used as $element_name ) {
			$element = Respira_Visual_Composer_Element_Registry::get_element( $element_name );
			if ( ! $element ) {
				continue;
			}

			$element_schema = array(
				'title'       => $element['title'],
				'description' => $element['description'],
				'type'        => $element['type'],
				'properties'  => array(),
			);

			// Generate property schemas.
			if ( ! empty( $element['properties'] ) ) {
				foreach ( $element['properties'] as $property ) {
					$element_schema['properties'][ $property ] = $this->get_property_schema( $property, $element['type'] );
				}
			}

			$schema['available_elements'][ $element['type'] ] = $element_schema;
		}

		return $schema;
	}

	/**
	 * Get property schema for an element property.
	 *
	 * @since  1.4.0
	 * @param  string $property     Property name.
	 * @param  string $element_type Element type.
	 * @return array Property schema.
	 */
	private function get_property_schema( $property, $element_type ) {
		// Map common properties to their types and formats.
		$property_map = array(
			// Content properties.
			'content'      => array(
				'type'    => 'string',
				'format'  => 'html',
				'example' => '<p>Content here</p>',
			),
			'text'         => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Sample text',
			),
			'title'        => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Title',
			),

			// Link properties.
			'link'         => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'url:https://example.com|title:Link%20Title|target:_blank',
			),
			'url'          => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com',
			),

			// Image properties.
			'image'        => array(
				'type'    => 'string',
				'format'  => 'attachment_id',
				'example' => '123',
			),
			'images'       => array(
				'type'    => 'string',
				'format'  => 'attachment_ids',
				'example' => '123,456,789',
			),
			'img_size'     => array(
				'type'    => 'string',
				'format'  => 'image_size',
				'example' => 'large',
			),

			// Style properties.
			'color'        => array(
				'type'    => 'string',
				'format'  => 'color',
				'example' => 'blue',
			),
			'bgcolor'      => array(
				'type'    => 'string',
				'format'  => 'color',
				'example' => 'blue',
			),
			'custombgcolor' => array(
				'type'    => 'string',
				'format'  => 'hex',
				'example' => '#0066CC',
			),
			'style'        => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'flat',
			),
			'css'          => array(
				'type'    => 'string',
				'format'  => 'css_classes',
				'example' => '.vc_custom_1234567890{margin-top: 20px !important;}',
			),

			// Layout properties.
			'width'        => array(
				'type'    => 'string',
				'format'  => 'percentage',
				'example' => '1/2',
			),
			'offset'       => array(
				'type'    => 'string',
				'format'  => 'offset',
				'example' => 'vc_col-md-offset-1',
			),
			'full_width'   => array(
				'type'    => 'string',
				'format'  => 'boolean',
				'example' => 'true',
			),
			'gap'          => array(
				'type'    => 'string',
				'format'  => 'number',
				'example' => '35',
			),
			'columns_placement' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'middle',
				'options' => array( 'top', 'middle', 'bottom', 'stretch' ),
			),
			'alignment'    => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'center',
				'options' => array( 'left', 'center', 'right' ),
			),
			'align'        => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'center',
				'options' => array( 'left', 'center', 'right', 'inline' ),
			),

			// Size properties.
			'size'         => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'md',
				'options' => array( 'xs', 'sm', 'md', 'lg' ),
			),
			'height'       => array(
				'type'    => 'string',
				'format'  => 'pixels',
				'example' => '50px',
			),

			// Separator properties.
			'border_width' => array(
				'type'    => 'string',
				'format'  => 'pixels',
				'example' => '2',
			),

			// Interactive properties.
			'interval'     => array(
				'type'    => 'string',
				'format'  => 'number',
				'example' => '3',
			),
			'onclick'      => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'link_image',
				'options' => array( '', 'link_image', 'link_no', 'custom_link', 'zoom' ),
			),
			'loop'         => array(
				'type'    => 'string',
				'format'  => 'boolean',
				'example' => 'yes',
			),

			// Icon properties.
			'type'         => array(
				'type'    => 'string',
				'format'  => 'icon_type',
				'example' => 'fontawesome',
			),
			'icon_fontawesome' => array(
				'type'    => 'string',
				'format'  => 'icon_class',
				'example' => 'fa fa-heart',
			),
			'i_icon_fontawesome' => array(
				'type'    => 'string',
				'format'  => 'icon_class',
				'example' => 'fa fa-check',
			),

			// Font properties.
			'font_container' => array(
				'type'    => 'string',
				'format'  => 'font_container',
				'example' => 'tag:h2|font_size:30|text_align:center',
			),
			'use_theme_fonts' => array(
				'type'    => 'string',
				'format'  => 'boolean',
				'example' => 'yes',
			),

			// Progress/Charts properties.
			'values'       => array(
				'type'    => 'string',
				'format'  => 'exploded_values',
				'example' => '90|Skill One,80|Skill Two',
			),
			'value'        => array(
				'type'    => 'string',
				'format'  => 'number',
				'example' => '75',
			),
			'units'        => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => '%',
			),
			'label'        => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Label',
			),

			// Message box properties.
			'message_box_style' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'standard',
				'options' => array( 'standard', 'solid', 'outline', '3d' ),
			),
			'message_box_color' => array(
				'type'    => 'string',
				'format'  => 'color',
				'example' => 'info',
				'options' => array( 'alert-info', 'alert-success', 'alert-warning', 'alert-danger' ),
			),

			// Grid/Layout properties.
			'grid_columns_count' => array(
				'type'    => 'string',
				'format'  => 'number',
				'example' => '3',
			),
			'grid_layout' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'title|link_post',
			),
			'item'        => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'title|link_post',
			),

			// Post properties.
			'count'       => array(
				'type'    => 'string',
				'format'  => 'number',
				'example' => '10',
			),

			// Social properties.
			'flickr_id'   => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => '12345678@N00',
			),
			'display'     => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'latest',
			),
		);

		// Return the schema for this property, or a generic schema if not found.
		if ( isset( $property_map[ $property ] ) ) {
			return $property_map[ $property ];
		}

		// Generic property schema.
		return array(
			'type'    => 'string',
			'format'  => 'text',
			'example' => 'value',
		);
	}

	/**
	 * Get structure notes for Visual Composer.
	 *
	 * @since  1.4.0
	 * @return array Structure notes.
	 */
	private function get_structure_notes() {
		return array(
			'hierarchy'       => 'Row (vc_row) → Column (vc_column) → Elements',
			'shortcode_format' => 'Visual Composer uses WordPress shortcodes with vc_ prefix',
			'nesting'         => 'Supports nested rows via vc_row_inner and vc_column_inner',
			'css_classes'     => 'Custom CSS stored in css parameter',
			'note'            => 'Elements are stored as shortcodes in post_content',
		);
	}
}
