<?php
/**
 * Thrive Architect element schema generator.
 *
 * Generates schemas for all registered elements with property definitions.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/thrive-intelligence
 * @since      1.4.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-schema-base.php';

/**
 * Thrive Architect element schema class.
 *
 * @since 1.4.0
 */
class Respira_Thrive_Element_Schema extends Respira_Builder_Schema_Base {

	/**
	 * Constructor.
	 *
	 * @since 1.4.0
	 */
	public function __construct() {
		$version = defined( 'TVE_VERSION' ) ? TVE_VERSION : 'unknown';
		parent::__construct( 'thrive', $version );
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since  1.4.0
	 * @param  array $elements_used Optional. Array of element names used on the page.
	 * @return array Builder schema with element information.
	 */
	public function get_builder_schema( $elements_used = array() ) {
		$schema = array(
			'builder'          => 'thrive',
			'builder_version'  => $this->builder_version,
			'available_elements' => array(),
			'quick_reference'  => $this->get_quick_reference(),
			'structure_notes'  => $this->get_structure_notes(),
		);

		// Get schemas for elements used on the page (or all elements if none specified).
		if ( empty( $elements_used ) ) {
			$all_elements = Respira_Thrive_Element_Registry::get_all_elements();
			$elements_used = array_column( $all_elements, 'name' );
		}

		foreach ( $elements_used as $element_name ) {
			$element = Respira_Thrive_Element_Registry::get_element( $element_name );
			if ( ! $element ) {
				continue;
			}

			$element_schema = array(
				'title'       => $element['title'],
				'description' => $element['description'],
				'type'        => $element['type'],
				'properties'  => array(),
			);

			// Generate property schemas.
			if ( ! empty( $element['properties'] ) ) {
				foreach ( $element['properties'] as $property ) {
					$element_schema['properties'][ $property ] = $this->get_property_schema( $property, $element['type'] );
				}
			}

			$schema['available_elements'][ $element_name ] = $element_schema;
		}

		return $schema;
	}

	/**
	 * Get property schema for an element property.
	 *
	 * @since  1.4.0
	 * @param  string $property     Property name.
	 * @param  string $element_type Element type.
	 * @return array Property schema.
	 */
	private function get_property_schema( $property, $element_type ) {
		// Map common properties to their types and formats.
		$property_map = array(
			// Text properties.
			'content'    => array(
				'type'    => 'string',
				'format'  => 'html',
				'example' => '<p>Sample content</p>',
			),
			'text'       => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Sample text',
			),
			'typography' => array(
				'type'    => 'object',
				'format'  => 'typography',
				'example' => array(
					'font_family' => 'Arial',
					'font_size'   => '16px',
					'font_weight' => '400',
				),
			),
			'font'       => array(
				'type'    => 'object',
				'format'  => 'font',
				'example' => array(
					'family' => 'Arial',
					'size'   => '16px',
				),
			),
			'color'      => array(
				'type'    => 'string',
				'format'  => 'hex',
				'example' => '#000000',
			),
			'hover_color' => array(
				'type'    => 'string',
				'format'  => 'hex',
				'example' => '#333333',
			),
			'alignment'  => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'left',
				'options' => array( 'left', 'center', 'right', 'justify' ),
			),

			// Image/Media properties.
			'src'        => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com/image.jpg',
			),
			'alt'        => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Image description',
			),
			'width'      => array(
				'type'    => 'string',
				'format'  => 'dimension',
				'example' => '800px',
			),
			'height'     => array(
				'type'    => 'string',
				'format'  => 'dimension',
				'example' => '600px',
			),
			'thumbnail'  => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com/thumbnail.jpg',
			),

			// Link properties.
			'url'        => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com',
			),
			'link'       => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com',
			),

			// Button properties.
			'style'      => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'primary',
				'options' => array( 'primary', 'secondary', 'outline', 'ghost' ),
			),
			'size'       => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'medium',
				'options' => array( 'small', 'medium', 'large' ),
			),

			// Form properties.
			'fields'     => array(
				'type'    => 'array',
				'format'  => 'form_fields',
				'example' => array(
					array(
						'type'  => 'email',
						'label' => 'Email Address',
					),
				),
			),
			'submit_text' => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Submit',
			),
			'success_message' => array(
				'type'    => 'string',
				'format'  => 'html',
				'example' => '<p>Thank you for subscribing!</p>',
			),
			'success_url' => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com/thank-you',
			),
			'integration' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'mailchimp',
				'options' => array( 'mailchimp', 'aweber', 'activecampaign', 'convertkit' ),
			),

			// Layout properties.
			'background' => array(
				'type'    => 'object',
				'format'  => 'background',
				'example' => array(
					'color' => '#FFFFFF',
					'image' => '',
				),
			),
			'padding'    => array(
				'type'    => 'object',
				'format'  => 'spacing',
				'example' => array(
					'top'    => '20px',
					'right'  => '20px',
					'bottom' => '20px',
					'left'   => '20px',
				),
			),
			'margin'     => array(
				'type'    => 'object',
				'format'  => 'spacing',
				'example' => array(
					'top'    => '0px',
					'right'  => '0px',
					'bottom' => '0px',
					'left'   => '0px',
				),
			),
			'border'     => array(
				'type'    => 'object',
				'format'  => 'border',
				'example' => array(
					'width' => '1px',
					'style' => 'solid',
					'color' => '#CCCCCC',
				),
			),

			// Video properties.
			'type'       => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'youtube',
				'options' => array( 'youtube', 'vimeo', 'wistia', 'custom' ),
			),
			'autoplay'   => array(
				'type'    => 'boolean',
				'format'  => 'boolean',
				'example' => false,
			),
			'controls'   => array(
				'type'    => 'boolean',
				'format'  => 'boolean',
				'example' => true,
			),

			// Social properties.
			'networks'   => array(
				'type'    => 'array',
				'format'  => 'social_networks',
				'example' => array( 'facebook', 'twitter', 'linkedin' ),
			),
			'urls'       => array(
				'type'    => 'object',
				'format'  => 'social_urls',
				'example' => array(
					'facebook' => 'https://facebook.com/yourpage',
					'twitter'  => 'https://twitter.com/yourhandle',
				),
			),

			// Interactive elements.
			'tabs'       => array(
				'type'    => 'array',
				'format'  => 'tabs',
				'example' => array(
					array(
						'title'   => 'Tab 1',
						'content' => 'Tab 1 content',
					),
				),
			),
			'orientation' => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'horizontal',
				'options' => array( 'horizontal', 'vertical' ),
			),
			'title'      => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'Toggle Title',
			),

			// Countdown properties.
			'date'       => array(
				'type'    => 'string',
				'format'  => 'date',
				'example' => '2024-12-31 23:59:59',
			),
			'timezone'   => array(
				'type'    => 'string',
				'format'  => 'timezone',
				'example' => 'America/New_York',
			),
			'format'     => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'days_hours_minutes',
				'options' => array( 'days_hours_minutes', 'hours_minutes_seconds', 'days_only' ),
			),
			'action'     => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'hide',
				'options' => array( 'hide', 'show_message', 'redirect' ),
			),

			// Testimonial properties.
			'author'     => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'John Doe',
			),
			'role'       => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => 'CEO, Company',
			),
			'image'      => array(
				'type'    => 'string',
				'format'  => 'url',
				'example' => 'https://example.com/author.jpg',
			),

			// Icon properties.
			'icon'       => array(
				'type'    => 'string',
				'format'  => 'icon',
				'example' => 'fa-check',
			),

			// Pricing table.
			'columns'    => array(
				'type'    => 'array',
				'format'  => 'pricing_columns',
				'example' => array(),
			),
			'features'   => array(
				'type'    => 'array',
				'format'  => 'features',
				'example' => array(),
			),
			'prices'     => array(
				'type'    => 'array',
				'format'  => 'prices',
				'example' => array(),
			),

			// Progress bar / Counter.
			'percentage' => array(
				'type'    => 'number',
				'format'  => 'percentage',
				'example' => 75,
			),
			'animation'  => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'slide',
				'options' => array( 'slide', 'fade', 'none' ),
			),
			'start'      => array(
				'type'    => 'number',
				'format'  => 'number',
				'example' => 0,
			),
			'end'        => array(
				'type'    => 'number',
				'format'  => 'number',
				'example' => 100,
			),
			'duration'   => array(
				'type'    => 'number',
				'format'  => 'milliseconds',
				'example' => 2000,
			),
			'suffix'     => array(
				'type'    => 'string',
				'format'  => 'text',
				'example' => '+',
			),

			// Table properties.
			'rows'       => array(
				'type'    => 'array',
				'format'  => 'table_rows',
				'example' => array(),
			),

			// Menu properties.
			'menu_id'    => array(
				'type'    => 'string',
				'format'  => 'menu_id',
				'example' => 'primary',
			),
			'layout'     => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'horizontal',
				'options' => array( 'horizontal', 'vertical' ),
			),

			// Custom HTML.
			'html'       => array(
				'type'    => 'string',
				'format'  => 'html',
				'example' => '<div>Custom HTML</div>',
			),

			// WordPress content.
			'post_id'    => array(
				'type'    => 'number',
				'format'  => 'post_id',
				'example' => 1,
			),

			// Star rating.
			'rating'     => array(
				'type'    => 'number',
				'format'  => 'number',
				'example' => 4.5,
			),
			'max'        => array(
				'type'    => 'number',
				'format'  => 'number',
				'example' => 5,
			),

			// Heading-specific.
			'tag'        => array(
				'type'    => 'string',
				'format'  => 'enum',
				'example' => 'h2',
				'options' => array( 'h1', 'h2', 'h3', 'h4', 'h5', 'h6' ),
			),
		);

		// Return the schema for this property, or a generic schema if not found.
		if ( isset( $property_map[ $property ] ) ) {
			return $property_map[ $property ];
		}

		// Generic property schema.
		return array(
			'type'    => 'string',
			'format'  => 'text',
			'example' => 'value',
		);
	}

	/**
	 * Get structure notes for Thrive Architect.
	 *
	 * @since  1.4.0
	 * @return array Structure notes.
	 */
	private function get_structure_notes() {
		return array(
			'hierarchy'   => 'Section → Column → Elements',
			'format'      => 'Thrive uses HTML with data attributes for element structure',
			'elements'    => 'Each element has a data-tcb-elem-type attribute',
			'nesting'     => 'Container elements (Section, Column, Content Box) can contain child elements',
			'conversion'  => 'Thrive focuses on conversion elements like opt-in forms, countdown timers, and lead generation',
			'note'        => 'Always preserve the element hierarchy and data attributes when modifying content',
		);
	}
}
