<?php
/**
 * Thrive Architect element registry.
 *
 * Dynamic element detection from Thrive Architect's registry.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/thrive-intelligence
 * @since      1.4.0
 */

/**
 * Thrive Architect element registry class.
 *
 * @since 1.4.0
 */
class Respira_Thrive_Element_Registry {

	/**
	 * Get all registered elements with metadata.
	 *
	 * @since  1.4.0
	 * @return array Array of element information.
	 */
	public static function get_all_elements() {
		// Check cache first.
		$cached = get_transient( 'respira_thrive_elements' );
		if ( false !== $cached ) {
			return $cached;
		}

		$element_list = array();

		// Try to detect elements from Thrive's internal registry.
		if ( function_exists( 'tcb_elements' ) ) {
			$element_list = self::detect_from_api();
		}

		// Fallback: use known element types from Thrive documentation.
		if ( empty( $element_list ) ) {
			$element_list = self::get_known_elements();
		}

		// Cache for 24 hours.
		set_transient( 'respira_thrive_elements', $element_list, DAY_IN_SECONDS );

		return $element_list;
	}

	/**
	 * Detect elements from Thrive API (if available).
	 *
	 * @since  1.4.0
	 * @return array Array of detected elements.
	 */
	private static function detect_from_api() {
		$elements = array();

		try {
			// Thrive uses tcb_elements() to get element list.
			if ( function_exists( 'tcb_elements' ) ) {
				$thrive_elements = tcb_elements();
				if ( is_object( $thrive_elements ) && method_exists( $thrive_elements, 'get_all' ) ) {
					$all_elements = $thrive_elements->get_all();
					foreach ( $all_elements as $tag => $element_instance ) {
						$elements[] = array(
							'name'        => $tag,
							'title'       => method_exists( $element_instance, 'name' ) ? $element_instance->name() : ucfirst( $tag ),
							'type'        => $tag,
							'description' => '',
							'properties'  => array(),
						);
					}
				}
			}
		} catch ( Exception $e ) {
			// Silently fall back to known elements.
		}

		return $elements;
	}

	/**
	 * Get known Thrive Architect elements from documentation.
	 *
	 * @since  1.4.0
	 * @return array Array of known elements.
	 */
	private static function get_known_elements() {
		return array(
			// Content Elements.
			array(
				'name'        => 'text',
				'title'       => 'Text',
				'type'        => 'text',
				'description' => 'Rich text element with formatting options',
				'properties'  => array( 'content', 'font', 'color', 'alignment' ),
			),
			array(
				'name'        => 'paragraph',
				'title'       => 'Paragraph',
				'type'        => 'paragraph',
				'description' => 'Paragraph text block',
				'properties'  => array( 'content', 'typography', 'color' ),
			),
			array(
				'name'        => 'heading',
				'title'       => 'Heading',
				'type'        => 'heading',
				'description' => 'Heading element (H1-H6)',
				'properties'  => array( 'text', 'tag', 'typography', 'color' ),
			),

			// Conversion Elements.
			array(
				'name'        => 'button',
				'title'       => 'Button',
				'type'        => 'button',
				'description' => 'Call-to-action button',
				'properties'  => array( 'text', 'url', 'style', 'size', 'color', 'hover_color' ),
			),
			array(
				'name'        => 'lead_generation',
				'title'       => 'Lead Generation',
				'type'        => 'lead_generation',
				'description' => 'Lead capture form',
				'properties'  => array( 'fields', 'submit_text', 'success_message' ),
			),
			array(
				'name'        => 'optin',
				'title'       => 'Opt-in Form',
				'type'        => 'optin',
				'description' => 'Email opt-in form',
				'properties'  => array( 'fields', 'integration', 'success_url' ),
			),

			// Media Elements.
			array(
				'name'        => 'image',
				'title'       => 'Image',
				'type'        => 'image',
				'description' => 'Image element with styling',
				'properties'  => array( 'src', 'alt', 'width', 'height', 'link' ),
			),
			array(
				'name'        => 'video',
				'title'       => 'Video',
				'type'        => 'video',
				'description' => 'Video embed (YouTube, Vimeo, Wistia)',
				'properties'  => array( 'url', 'type', 'autoplay', 'thumbnail' ),
			),
			array(
				'name'        => 'audio',
				'title'       => 'Audio',
				'type'        => 'audio',
				'description' => 'Audio player',
				'properties'  => array( 'src', 'type', 'controls' ),
			),

			// Social Elements.
			array(
				'name'        => 'social_share',
				'title'       => 'Social Share',
				'type'        => 'social_share',
				'description' => 'Social sharing buttons',
				'properties'  => array( 'networks', 'style', 'size' ),
			),
			array(
				'name'        => 'social_follow',
				'title'       => 'Social Follow',
				'type'        => 'social_follow',
				'description' => 'Social media follow buttons',
				'properties'  => array( 'networks', 'urls', 'style' ),
			),

			// Layout Elements.
			array(
				'name'        => 'section',
				'title'       => 'Section',
				'type'        => 'section',
				'description' => 'Content section container',
				'properties'  => array( 'background', 'padding', 'margin' ),
			),
			array(
				'name'        => 'column',
				'title'       => 'Column',
				'type'        => 'column',
				'description' => 'Column layout element',
				'properties'  => array( 'width', 'background', 'padding' ),
			),
			array(
				'name'        => 'contentbox',
				'title'       => 'Content Box',
				'type'        => 'contentbox',
				'description' => 'Content container with background',
				'properties'  => array( 'background', 'border', 'padding' ),
			),

			// Interactive Elements.
			array(
				'name'        => 'tabs',
				'title'       => 'Tabs',
				'type'        => 'tabs',
				'description' => 'Tabbed content container',
				'properties'  => array( 'tabs', 'orientation', 'style' ),
			),
			array(
				'name'        => 'toggle',
				'title'       => 'Toggle',
				'type'        => 'toggle',
				'description' => 'Collapsible toggle/accordion',
				'properties'  => array( 'title', 'content', 'style' ),
			),
			array(
				'name'        => 'countdown',
				'title'       => 'Countdown Timer',
				'type'        => 'countdown',
				'description' => 'Countdown timer for urgency',
				'properties'  => array( 'date', 'timezone', 'format', 'action' ),
			),

			// Testimonial & Trust.
			array(
				'name'        => 'testimonial',
				'title'       => 'Testimonial',
				'type'        => 'testimonial',
				'description' => 'Customer testimonial block',
				'properties'  => array( 'content', 'author', 'role', 'image' ),
			),
			array(
				'name'        => 'icon',
				'title'       => 'Icon',
				'type'        => 'icon',
				'description' => 'Icon element',
				'properties'  => array( 'icon', 'size', 'color', 'link' ),
			),

			// Advanced Elements.
			array(
				'name'        => 'pricing_table',
				'title'       => 'Pricing Table',
				'type'        => 'pricing_table',
				'description' => 'Pricing comparison table',
				'properties'  => array( 'columns', 'features', 'prices' ),
			),
			array(
				'name'        => 'progressbar',
				'title'       => 'Progress Bar',
				'type'        => 'progressbar',
				'description' => 'Animated progress bar',
				'properties'  => array( 'percentage', 'color', 'animation' ),
			),
			array(
				'name'        => 'fill_counter',
				'title'       => 'Fill Counter',
				'type'        => 'fill_counter',
				'description' => 'Animated number counter',
				'properties'  => array( 'start', 'end', 'duration', 'suffix' ),
			),
			array(
				'name'        => 'table',
				'title'       => 'Table',
				'type'        => 'table',
				'description' => 'Data table',
				'properties'  => array( 'rows', 'columns', 'style' ),
			),

			// Menu & Navigation.
			array(
				'name'        => 'menu',
				'title'       => 'Menu',
				'type'        => 'menu',
				'description' => 'Navigation menu',
				'properties'  => array( 'menu_id', 'style', 'layout' ),
			),

			// Code & Custom.
			array(
				'name'        => 'custom_html',
				'title'       => 'Custom HTML',
				'type'        => 'custom_html',
				'description' => 'Custom HTML/CSS/JS code',
				'properties'  => array( 'html' ),
			),
			array(
				'name'        => 'wordpress',
				'title'       => 'WordPress Content',
				'type'        => 'wordpress',
				'description' => 'WordPress post content',
				'properties'  => array( 'post_id' ),
			),

			// Others.
			array(
				'name'        => 'divider',
				'title'       => 'Divider',
				'type'        => 'divider',
				'description' => 'Visual divider line',
				'properties'  => array( 'style', 'width', 'color' ),
			),
			array(
				'name'        => 'star_rating',
				'title'       => 'Star Rating',
				'type'        => 'star_rating',
				'description' => 'Star rating display',
				'properties'  => array( 'rating', 'max', 'color' ),
			),
		);
	}

	/**
	 * Get element by name.
	 *
	 * @since  1.4.0
	 * @param  string $element_name Element name.
	 * @return array|null Element information or null if not found.
	 */
	public static function get_element( $element_name ) {
		$elements = self::get_all_elements();

		foreach ( $elements as $element ) {
			if ( $element['name'] === $element_name || $element['type'] === $element_name ) {
				return $element;
			}
		}

		return null;
	}

	/**
	 * Clear element cache.
	 *
	 * @since 1.4.0
	 */
	public static function clear_cache() {
		delete_transient( 'respira_thrive_elements' );
	}
}
