<?php
/**
 * Base validator class for page builders.
 *
 * Provides common validation methods that can be extended by builder-specific validators.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/intelligence
 * @since      1.3.0
 */

/**
 * Base validator class.
 *
 * @since 1.3.0
 */
abstract class Respira_Builder_Validator_Base {

	/**
	 * Validation errors.
	 *
	 * @since 1.3.0
	 * @var array
	 */
	protected $errors = array();

	/**
	 * Validate a layout structure.
	 *
	 * @since  1.3.0
	 * @param  array $content Content structure to validate.
	 * @return array Validation result with 'valid' boolean and 'errors' array.
	 */
	abstract public function validate_layout( $content );

	/**
	 * Validate structure nesting.
	 *
	 * @since  1.3.0
	 * @param  array $content Content structure.
	 * @param  array $rules   Nesting rules (parent => [allowed_children]).
	 * @return bool  True if structure is valid.
	 */
	protected function validate_nesting( $content, $rules ) {
		$this->errors = array();

		if ( ! is_array( $content ) ) {
			$this->add_error( __( 'Content must be an array.', 'respira-for-wordpress' ) );
			return false;
		}

		foreach ( $content as $item ) {
			if ( ! isset( $item['type'] ) ) {
				$this->add_error( __( 'Each item must have a type.', 'respira-for-wordpress' ) );
				continue;
			}

			$type = $item['type'];
			$children = isset( $item['children'] ) ? $item['children'] : array();

			// Check if this type can have children.
			if ( ! empty( $children ) && isset( $rules[ $type ] ) ) {
				foreach ( $children as $child ) {
					if ( ! isset( $child['type'] ) ) {
						continue;
					}

					$child_type = $child['type'];
					if ( ! in_array( $child_type, $rules[ $type ], true ) ) {
						$this->add_error(
							sprintf(
								/* translators: 1: child type, 2: parent type */
								__( 'Invalid nesting: %1$s cannot be a child of %2$s.', 'respira-for-wordpress' ),
								$child_type,
								$type
							)
						);
					}
				}

				// Recursively validate children.
				$this->validate_nesting( $children, $rules );
			}
		}

		return empty( $this->errors );
	}

	/**
	 * Validate attribute format.
	 *
	 * @since  1.3.0
	 * @param  mixed  $value Attribute value.
	 * @param  string $type  Expected type (string, number, color, url, etc.).
	 * @param  string $name  Attribute name (for error messages).
	 * @return bool   True if format is valid.
	 */
	protected function validate_attribute_format( $value, $type, $name = '' ) {
		switch ( $type ) {
			case 'string':
				return is_string( $value );

			case 'number':
				return is_numeric( $value );

			case 'boolean':
				return is_bool( $value ) || in_array( $value, array( 'true', 'false', '1', '0', 'on', 'off' ), true );

			case 'color':
				return $this->validate_color( $value );

			case 'url':
				return $this->validate_url( $value );

			case 'email':
				return is_email( $value );

			case 'array':
				return is_array( $value );

			default:
				return true; // Unknown type, assume valid.
		}
	}

	/**
	 * Validate color value.
	 *
	 * @since  1.3.0
	 * @param  string $value Color value.
	 * @return bool  True if valid color.
	 */
	protected function validate_color( $value ) {
		if ( empty( $value ) ) {
			return true; // Empty is valid (optional).
		}

		// Hex color.
		if ( preg_match( '/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/', $value ) ) {
			return true;
		}

		// RGB/RGBA.
		if ( preg_match( '/^rgba?\(/', $value ) ) {
			return true;
		}

		// Named colors (basic check).
		$named_colors = array( 'red', 'blue', 'green', 'black', 'white', 'transparent' );
		if ( in_array( strtolower( $value ), $named_colors, true ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Validate URL.
	 *
	 * @since  1.3.0
	 * @param  string $value URL value.
	 * @return bool  True if valid URL.
	 */
	protected function validate_url( $value ) {
		if ( empty( $value ) ) {
			return true; // Empty is valid (optional).
		}

		return filter_var( $value, FILTER_VALIDATE_URL ) !== false || esc_url_raw( $value ) === $value;
	}

	/**
	 * Add validation error.
	 *
	 * @since 1.3.0
	 * @param string $message Error message.
	 */
	protected function add_error( $message ) {
		$this->errors[] = $message;
	}

	/**
	 * Get validation errors.
	 *
	 * @since  1.3.0
	 * @return array Array of error messages.
	 */
	public function get_errors() {
		return $this->errors;
	}

	/**
	 * Clear validation errors.
	 *
	 * @since 1.3.0
	 */
	public function clear_errors() {
		$this->errors = array();
	}
}

