<?php
/**
 * Base patterns class for page builders.
 *
 * Provides common pattern storage and retrieval methods.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/intelligence
 * @since      1.3.0
 */

/**
 * Base patterns class.
 *
 * @since 1.3.0
 */
abstract class Respira_Builder_Patterns_Base {

	/**
	 * Builder name.
	 *
	 * @since 1.3.0
	 * @var string
	 */
	protected $builder_name;

	/**
	 * Patterns array.
	 *
	 * @since 1.3.0
	 * @var array
	 */
	protected $patterns = array();

	/**
	 * Constructor.
	 *
	 * @since 1.3.0
	 * @param string $builder_name Builder name.
	 */
	public function __construct( $builder_name ) {
		$this->builder_name = $builder_name;
		$this->load_patterns();
	}

	/**
	 * Load patterns.
	 *
	 * @since 1.3.0
	 */
	abstract protected function load_patterns();

	/**
	 * Get all patterns.
	 *
	 * @since  1.3.0
	 * @return array All patterns.
	 */
	public function get_all_patterns() {
		return $this->patterns;
	}

	/**
	 * Get pattern by name.
	 *
	 * @since  1.3.0
	 * @param  string $name Pattern name.
	 * @return array|null Pattern data or null if not found.
	 */
	public function get_pattern( $name ) {
		return isset( $this->patterns[ $name ] ) ? $this->patterns[ $name ] : null;
	}

	/**
	 * Get patterns by category.
	 *
	 * @since  1.3.0
	 * @param  string $category Category name.
	 * @return array Patterns in category.
	 */
	public function get_patterns_by_category( $category ) {
		$filtered = array();

		foreach ( $this->patterns as $name => $pattern ) {
			if ( isset( $pattern['category'] ) && $pattern['category'] === $category ) {
				$filtered[ $name ] = $pattern;
			}
		}

		return $filtered;
	}

	/**
	 * Search patterns.
	 *
	 * @since  1.3.0
	 * @param  string $query Search query.
	 * @return array Matching patterns.
	 */
	public function search_patterns( $query ) {
		$query   = strtolower( $query );
		$results = array();

		foreach ( $this->patterns as $name => $pattern ) {
			$search_text = strtolower( $name . ' ' . ( isset( $pattern['title'] ) ? $pattern['title'] : '' ) . ' ' . ( isset( $pattern['description'] ) ? $pattern['description'] : '' ) );

			if ( false !== strpos( $search_text, $query ) ) {
				$results[ $name ] = $pattern;
			}
		}

		return $results;
	}

	/**
	 * Get pattern categories.
	 *
	 * @since  1.3.0
	 * @return array Unique category names.
	 */
	public function get_categories() {
		$categories = array();

		foreach ( $this->patterns as $pattern ) {
			if ( isset( $pattern['category'] ) ) {
				$categories[] = $pattern['category'];
			}
		}

		return array_unique( $categories );
	}

	/**
	 * Match pattern to content.
	 *
	 * @since  1.3.0
	 * @param  array $content Content structure.
	 * @return array|null Matching pattern or null.
	 */
	public function match_pattern( $content ) {
		// Basic implementation - can be overridden.
		foreach ( $this->patterns as $name => $pattern ) {
			if ( isset( $pattern['structure'] ) && $this->compare_structure( $content, $pattern['structure'] ) ) {
				return $pattern;
			}
		}

		return null;
	}

	/**
	 * Compare content structure to pattern structure.
	 *
	 * @since  1.3.0
	 * @param  array $content Content structure.
	 * @param  array $pattern Pattern structure.
	 * @return bool  True if structures match.
	 */
	protected function compare_structure( $content, $pattern ) {
		if ( ! is_array( $content ) || ! is_array( $pattern ) ) {
			return false;
		}

		// Simple comparison - check if types match.
		$content_types = $this->extract_types( $content );
		$pattern_types = $this->extract_types( $pattern );

		return $content_types === $pattern_types;
	}

	/**
	 * Extract types from structure.
	 *
	 * @since  1.3.0
	 * @param  array $structure Structure array.
	 * @return array Array of types.
	 */
	protected function extract_types( $structure ) {
		$types = array();

		foreach ( $structure as $item ) {
			if ( isset( $item['type'] ) ) {
				$types[] = $item['type'];
			}

			if ( isset( $item['children'] ) && is_array( $item['children'] ) ) {
				$types = array_merge( $types, $this->extract_types( $item['children'] ) );
			}
		}

		return $types;
	}

	/**
	 * Get pattern count.
	 *
	 * @since  1.3.0
	 * @return int Number of patterns.
	 */
	public function get_pattern_count() {
		return count( $this->patterns );
	}
}

