<?php
/**
 * WPBakery Page Builder implementation.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders
 * @since      1.3.0
 */

/**
 * WPBakery builder class (PRO feature).
 *
 * @since 1.3.0
 */
class Respira_Builder_WPBakery extends Respira_Builder_Interface {

	/**
	 * Detect if WPBakery is active.
	 *
	 * @since 1.3.0
	 * @return bool True if WPBakery is active.
	 */
	public function detect() {
		return defined( 'WPB_VC_VERSION' ) || class_exists( 'Vc_Manager' ) || function_exists( 'vc_map' );
	}

	/**
	 * Get builder name.
	 *
	 * @since 1.3.0
	 * @return string Builder name.
	 */
	public function get_name() {
		return 'WPBakery';
	}

	/**
	 * Get builder version.
	 *
	 * @since 1.3.0
	 * @return string Builder version.
	 */
	public function get_version() {
		if ( defined( 'WPB_VC_VERSION' ) ) {
			return WPB_VC_VERSION;
		}
		return 'unknown';
	}

	/**
	 * Check if post uses WPBakery.
	 *
	 * @since 1.3.0
	 * @param int $post_id Post ID.
	 * @return bool True if uses WPBakery.
	 */
	public function is_builder_used( $post_id ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return false;
		}

		// Check for Visual Composer shortcodes.
		if ( preg_match( '/\[vc_/', $post->post_content ) ) {
			return true;
		}

		// Check for WPBakery meta.
		if ( get_post_meta( $post_id, '_wpb_vc_js_status', true ) === 'true' ) {
			return true;
		}

		return false;
	}

	/**
	 * Extract content from post.
	 *
	 * @since 1.3.0
	 * @param int $post_id Post ID.
	 * @return array Extracted shortcodes.
	 */
	public function extract_content( $post_id ) {
		$start = microtime( true );

		$post = get_post( $post_id );
		if ( ! $post ) {
			return array();
		}

		// Parse WPBakery shortcodes.
		$shortcodes = $this->parse_wpbakery_shortcodes( $post->post_content );

		$result = $this->simplify_structure( $shortcodes );

		$this->log_performance( 'extract', $post_id, microtime( true ) - $start );

		return $result;
	}

	/**
	 * Inject content into post.
	 *
	 * @since 1.3.0
	 * @param int   $post_id Post ID.
	 * @param array $content Simplified content.
	 * @return bool|WP_Error True on success.
	 */
	public function inject_content( $post_id, $content ) {
		$start = microtime( true );

		// Validate layout before injection.
		if ( class_exists( 'Respira_WPBakery_Validator' ) ) {
			$validator = new Respira_WPBakery_Validator();
			$validation = $validator->validate_layout( $content );
			if ( ! $validation['valid'] ) {
				return new WP_Error(
					'respira_wpbakery_validation_failed',
					__( 'WPBakery layout validation failed:', 'respira-for-wordpress' ) . ' ' . implode( ' ', $validation['errors'] ),
					$validation['errors']
				);
			}
		}

		// Convert simplified structure back to shortcodes.
		$shortcodes = $this->build_wpbakery_shortcodes( $content );

		// Update post content.
		$result = wp_update_post(
			array(
				'ID'           => $post_id,
				'post_content' => $shortcodes,
			),
			true
		);

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		// Mark as WPBakery-built.
		update_post_meta( $post_id, '_wpb_vc_js_status', 'true' );

		$this->log_performance( 'inject', $post_id, microtime( true ) - $start );

		return true;
	}

	/**
	 * Create a code block.
	 *
	 * @since 1.3.0
	 * @param int    $post_id Post ID.
	 * @param string $html    HTML content.
	 * @param string $css     CSS content.
	 * @param string $js      JavaScript content.
	 * @return bool|WP_Error True on success.
	 */
	public function create_code_block( $post_id, $html, $css = '', $js = '' ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return new WP_Error(
				'respira_post_not_found',
				__( 'Post not found.', 'respira-for-wordpress' )
			);
		}

		// Combine code.
		$combined_code = $html;
		if ( ! empty( $css ) ) {
			$combined_code .= "\n\n<style>\n" . $css . "\n</style>";
		}
		if ( ! empty( $js ) ) {
			$combined_code .= "\n\n<script>\n" . $js . "\n</script>";
		}

		// Create WPBakery raw HTML shortcode.
		$code_shortcode = '[vc_raw_html]' . base64_encode( $combined_code ) . '[/vc_raw_html]';

		// Append to existing content.
		$updated_content = $post->post_content . "\n" . $code_shortcode;

		$result = wp_update_post(
			array(
				'ID'           => $post_id,
				'post_content' => $updated_content,
			),
			true
		);

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return true;
	}

	/**
	 * Get documentation.
	 *
	 * @since 1.3.0
	 * @return array Documentation.
	 */
	public function get_documentation() {
		$documentation = array(
			'name'        => 'WPBakery Page Builder',
			'description' => 'Popular drag-and-drop page builder plugin',
			'overview'    => 'WPBakery uses shortcodes to store layouts. Structure: rows > columns > elements.',
			'structure'   => array(
				'row'     => 'Layout container (vc_row)',
				'column'  => 'Column container (vc_column)',
				'element' => 'Content element (vc_text_block, vc_single_image, etc.)',
			),
			'resources'   => array(
				'https://wpbakery.com/',
				'https://kb.wpbakery.com/docs/inner-api/vc_map/',
			),
		);

		// Add shortcode catalog if available.
		if ( class_exists( 'Respira_WPBakery_Shortcode_Registry' ) ) {
			$all_shortcodes = Respira_WPBakery_Shortcode_Registry::get_all_shortcodes();
			$documentation['shortcodes'] = array(
				'total'     => count( $all_shortcodes ),
				'available' => array_column( $all_shortcodes, 'name' ),
			);
		}

		// Add patterns if available.
		if ( function_exists( 'respira_get_wpbakery_patterns' ) ) {
			$patterns = respira_get_wpbakery_patterns();
			$documentation['patterns'] = array(
				'count'    => count( $patterns ),
				'available' => array_keys( $patterns ),
			);
		}

		return $documentation;
	}

	/**
	 * Get available modules.
	 *
	 * @since 1.3.0
	 * @return array Available shortcodes.
	 */
	public function get_available_modules() {
		// Use dynamic shortcode registry if available.
		if ( class_exists( 'Respira_WPBakery_Shortcode_Registry' ) ) {
			return Respira_WPBakery_Shortcode_Registry::get_all_shortcodes();
		}

		// Fallback to basic list.
		return array(
			array(
				'name'        => 'vc_row',
				'title'       => 'Row',
				'description' => 'Layout container',
			),
			array(
				'name'        => 'vc_column',
				'title'       => 'Column',
				'description' => 'Column container',
			),
			array(
				'name'        => 'vc_column_text',
				'title'       => 'Text Block',
				'description' => 'Text content element',
			),
			array(
				'name'        => 'vc_single_image',
				'title'       => 'Single Image',
				'description' => 'Image element',
			),
			array(
				'name'        => 'vc_raw_html',
				'title'       => 'Raw HTML',
				'description' => 'Custom HTML/CSS/JS',
			),
		);
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since  1.3.0
	 * @param  array $shortcodes_used Optional. Array of shortcode names used on the page.
	 * @return array Builder schema with shortcode information.
	 */
	public function get_builder_schema( $shortcodes_used = array() ) {
		// Use WPBakery Intelligence if available.
		if ( class_exists( 'Respira_WPBakery_Shortcode_Schema' ) ) {
			$schema_generator = new Respira_WPBakery_Shortcode_Schema();
			return $schema_generator->get_builder_schema( $shortcodes_used );
		}

		// Fallback to base implementation.
		return parent::get_builder_schema( $shortcodes_used );
	}

	/**
	 * Parse WPBakery shortcodes.
	 *
	 * @since  1.3.0
	 * @param  string $content Post content with shortcodes.
	 * @return array  Structured array of shortcodes.
	 */
	private function parse_wpbakery_shortcodes( $content ) {
		$shortcodes = array();

		if ( empty( $content ) ) {
			return $shortcodes;
		}

		// Get all WPBakery shortcodes dynamically if available.
		$vc_shortcodes = array();
		if ( class_exists( 'Respira_WPBakery_Shortcode_Registry' ) ) {
			$all_shortcodes = Respira_WPBakery_Shortcode_Registry::get_all_shortcodes();
			$vc_shortcodes = array_column( $all_shortcodes, 'name' );
		}

		// Use generic pattern that matches all vc_* shortcodes.
		$pattern = '/\[vc_(\w+)([^\]]*)\](.*?)\[\/vc_\1\]/s';
		preg_match_all( $pattern, $content, $matches, PREG_SET_ORDER );

		foreach ( $matches as $match ) {
			$shortcode_type = 'vc_' . $match[1];
			$attr_string    = $match[2];
			$shortcode_content = $match[3];

			$shortcode = array(
				'type'    => $shortcode_type,
				'attrs'   => $this->parse_shortcode_attrs( $attr_string ),
				'content' => $shortcode_content,
			);

			// Check for nested shortcodes recursively.
			if ( ! empty( $shortcode['content'] ) && preg_match( '/\[vc_/', $shortcode['content'] ) ) {
				$shortcode['children'] = $this->parse_wpbakery_shortcodes( $shortcode['content'] );
			}

			$shortcodes[] = $shortcode;
		}

		return $shortcodes;
	}

	/**
	 * Parse shortcode attributes.
	 *
	 * @since  1.3.0
	 * @param  string $attr_string Attribute string.
	 * @return array  Attributes array.
	 */
	private function parse_shortcode_attrs( $attr_string ) {
		$attrs = array();

		if ( empty( $attr_string ) ) {
			return $attrs;
		}

		// Pattern: key="value" or key='value' or key=value
		preg_match_all( '/(\w+)(?:\s*=\s*(?:"([^"]*)"|\'([^\']*)\'|([^\s]+)))?/s', $attr_string, $matches, PREG_SET_ORDER );

		foreach ( $matches as $match ) {
			$key = $match[1];
			$value = '';
			if ( ! empty( $match[2] ) ) {
				$value = $match[2]; // Double-quoted.
			} elseif ( ! empty( $match[3] ) ) {
				$value = $match[3]; // Single-quoted.
			} elseif ( ! empty( $match[4] ) ) {
				$value = $match[4]; // Unquoted.
			}

			$value = html_entity_decode( $value, ENT_QUOTES | ENT_HTML5, 'UTF-8' );
			$attrs[ $key ] = $value;
		}

		return $attrs;
	}

	/**
	 * Build WPBakery shortcodes from structured array.
	 *
	 * @since  1.3.0
	 * @param  array $shortcodes Structured shortcodes array.
	 * @return string WPBakery shortcodes.
	 */
	private function build_wpbakery_shortcodes( $shortcodes ) {
		$output = '';

		foreach ( $shortcodes as $shortcode ) {
			$type     = $shortcode['type'] ?? 'vc_column_text';
			$attrs    = $shortcode['attrs'] ?? array();
			$content  = $shortcode['content'] ?? '';
			$children = $shortcode['children'] ?? array();

			// Build attribute string.
			$attr_string = '';
			foreach ( $attrs as $key => $value ) {
				if ( is_array( $value ) || is_object( $value ) ) {
					$value = wp_json_encode( $value );
				}
				$value = (string) $value;
				$value = esc_attr( $value );
				$attr_string .= ' ' . sanitize_key( $key ) . '="' . $value . '"';
			}

			// Build shortcode opening tag.
			$output .= '[' . sanitize_key( $type ) . $attr_string . ']';

			// Add children if present.
			if ( ! empty( $children ) ) {
				$output .= $this->build_wpbakery_shortcodes( $children );
			} else {
				$output .= $content;
			}

			// Close shortcode tag.
			$output .= '[/' . sanitize_key( $type ) . ']' . "\n";
		}

		return $output;
	}

	/**
	 * Simplify WPBakery structure for AI.
	 *
	 * @since  1.3.0
	 * @param  array $shortcodes Parsed shortcodes.
	 * @return array Simplified structure.
	 */
	protected function simplify_structure( $shortcodes ) {
		return array_map(
			function( $shortcode ) {
				return array(
					'type'       => $shortcode['type'],
					'attributes' => $shortcode['attrs'],
					'content'    => $shortcode['content'],
					'children'   => isset( $shortcode['children'] ) ? $this->simplify_structure( $shortcode['children'] ) : array(),
				);
			},
			$shortcodes
		);
	}
}

