<?php
/**
 * Visual Composer Website Builder implementation.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders
 * @since      1.3.0
 */

/**
 * Visual Composer Website Builder class (PRO feature).
 *
 * @since 1.3.0
 */
class Respira_Builder_Visual_Composer extends Respira_Builder_Interface {

	/**
	 * Detect if Visual Composer is active.
	 *
	 * @since 1.3.0
	 * @return bool True if Visual Composer is active.
	 */
	public function detect() {
		return defined( 'VCV_VERSION' ) || class_exists( 'VisualComposer\Framework\Application' );
	}

	/**
	 * Get builder name.
	 *
	 * @since 1.3.0
	 * @return string Builder name.
	 */
	public function get_name() {
		return 'Visual Composer';
	}

	/**
	 * Get builder version.
	 *
	 * @since 1.3.0
	 * @return string Builder version.
	 */
	public function get_version() {
		if ( defined( 'VCV_VERSION' ) ) {
			return VCV_VERSION;
		}
		return 'unknown';
	}

	/**
	 * Check if post uses Visual Composer.
	 *
	 * @since 1.3.0
	 * @param int $post_id Post ID.
	 * @return bool True if uses Visual Composer.
	 */
	public function is_builder_used( $post_id ) {
		$data = get_post_meta( $post_id, 'vcv-pageContent', true );
		return ! empty( $data );
	}

	/**
	 * Extract content from post.
	 *
	 * @since 1.3.0
	 * @param int $post_id Post ID.
	 * @return array Extracted elements.
	 */
	public function extract_content( $post_id ) {
		$start = microtime( true );

		$data = get_post_meta( $post_id, 'vcv-pageContent', true );

		if ( empty( $data ) ) {
			return array();
		}

		// Visual Composer stores data as JSON.
		$elements = json_decode( $data, true );

		if ( ! is_array( $elements ) ) {
			return array();
		}

		$result = $this->simplify_structure( $elements );

		$this->log_performance( 'extract', $post_id, microtime( true ) - $start );

		return $result;
	}

	/**
	 * Inject content into post.
	 *
	 * @since 1.3.0
	 * @param int   $post_id Post ID.
	 * @param array $content Simplified content.
	 * @return bool|WP_Error True on success.
	 */
	public function inject_content( $post_id, $content ) {
		$start = microtime( true );

		// Validate with intelligence if available.
		if ( class_exists( 'Respira_Visual_Composer_Validator' ) ) {
			$validator = new Respira_Visual_Composer_Validator();
			$validation = $validator->validate_layout( $content );

			if ( ! $validation['valid'] ) {
				$this->log_performance( 'inject', $post_id, microtime( true ) - $start );
				return new WP_Error(
					'validation_failed',
					__( 'Content validation failed.', 'respira-for-wordpress' ),
					array( 'errors' => $validation['errors'] )
				);
			}
		}

		// Convert simplified structure back to Visual Composer format.
		$elements = $this->complexify_structure( $content );

		// Update Visual Composer meta.
		update_post_meta( $post_id, 'vcv-pageContent', wp_json_encode( $elements ) );
		update_post_meta( $post_id, 'vcv-pageDesignOptions', get_post_meta( $post_id, 'vcv-pageDesignOptions', true ) ?: '{}' );

		$this->log_performance( 'inject', $post_id, microtime( true ) - $start );

		return true;
	}

	/**
	 * Create a code block.
	 *
	 * @since 1.3.0
	 * @param int    $post_id Post ID.
	 * @param string $html    HTML content.
	 * @param string $css     CSS content.
	 * @param string $js      JavaScript content.
	 * @return bool|WP_Error True on success.
	 */
	public function create_code_block( $post_id, $html, $css = '', $js = '' ) {
		$data = get_post_meta( $post_id, 'vcv-pageContent', true );
		$elements = json_decode( $data, true ) ?: array();

		// Combine code.
		$combined_code = $html;
		if ( ! empty( $css ) ) {
			$combined_code .= "\n\n<style>\n" . $css . "\n</style>";
		}
		if ( ! empty( $js ) ) {
			$combined_code .= "\n\n<script>\n" . $js . "\n</script>";
		}

		// Create raw HTML element.
		$html_element = array(
			'id'         => uniqid( 'respira_' ),
			'tag'        => 'rawHtml',
			'elementType' => 'element',
			'designOptions' => array(),
			'customAttributes' => array(),
			'attributes' => array(
				'rawHtml' => $combined_code,
			),
		);

		$elements[] = $html_element;

		update_post_meta( $post_id, 'vcv-pageContent', wp_json_encode( $elements ) );

		return true;
	}

	/**
	 * Get documentation.
	 *
	 * @since 1.3.0
	 * @return array Documentation.
	 */
	public function get_documentation() {
		$docs = array(
			'name'        => 'Visual Composer Website Builder',
			'description' => 'React-based builder with Element API',
			'overview'    => 'Visual Composer uses elements with settings. Data is stored as JSON in postmeta.',
			'structure'   => array(
				'elements' => 'Content elements with settings',
			),
			'resources'   => array(
				'https://dev.visualcomposer.com/',
			),
		);

		// Add patterns if intelligence is available.
		if ( function_exists( 'respira_get_visual_composer_patterns' ) ) {
			$docs['patterns'] = respira_get_visual_composer_patterns();
		}

		// Add element catalog if intelligence is available.
		if ( class_exists( 'Respira_Visual_Composer_Element_Registry' ) ) {
			$docs['elements'] = Respira_Visual_Composer_Element_Registry::get_all_elements();
		}

		return $docs;
	}

	/**
	 * Get available modules.
	 *
	 * @since 1.3.0
	 * @return array Available elements.
	 */
	public function get_available_modules() {
		// Use dynamic element registry if available.
		if ( class_exists( 'Respira_Visual_Composer_Element_Registry' ) ) {
			return Respira_Visual_Composer_Element_Registry::get_all_elements();
		}

		// Fallback to basic list.
		return array(
			array(
				'name'        => 'textBlock',
				'title'       => 'Text Block',
				'description' => 'Text content element',
			),
			array(
				'name'        => 'image',
				'title'       => 'Image',
				'description' => 'Image element',
			),
			array(
				'name'        => 'rawHtml',
				'title'       => 'Raw HTML',
				'description' => 'Custom HTML/CSS/JS',
			),
		);
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since 1.4.0
	 * @param array $elements_used Optional. Array of element names used on the page.
	 * @return array Builder schema with element information.
	 */
	public function get_builder_schema( $elements_used = array() ) {
		if ( class_exists( 'Respira_Visual_Composer_Element_Schema' ) ) {
			$schema_generator = new Respira_Visual_Composer_Element_Schema();
			return $schema_generator->get_builder_schema( $elements_used );
		}

		// Fallback to basic schema.
		return array(
			'builder'         => 'visual-composer',
			'builder_version' => $this->get_version(),
			'elements'        => $this->get_available_modules(),
		);
	}

	/**
	 * Check if intelligence is available.
	 *
	 * @since 1.4.0
	 * @return bool True if intelligence is available.
	 */
	public function is_intelligence_available() {
		return class_exists( 'Respira_Visual_Composer_Intelligence_Loader' ) &&
			   class_exists( 'Respira_Visual_Composer_Element_Registry' ) &&
			   class_exists( 'Respira_Visual_Composer_Element_Schema' ) &&
			   class_exists( 'Respira_Visual_Composer_Validator' );
	}

	/**
	 * Simplify Visual Composer structure.
	 *
	 * @since  1.3.0
	 * @param  array $elements Visual Composer elements.
	 * @return array Simplified structure.
	 */
	protected function simplify_structure( $elements ) {
		return array_map(
			function( $element ) {
				$tag = $element['tag'] ?? 'unknown';
				$attributes = $element['attributes'] ?? array();

				return array(
					'type'       => $tag,
					'attributes' => $attributes,
					'content'    => isset( $attributes['text'] ) ? $attributes['text'] : ( isset( $attributes['rawHtml'] ) ? $attributes['rawHtml'] : '' ),
					'children'   => isset( $element['elements'] ) ? $this->simplify_structure( $element['elements'] ) : array(),
				);
			},
			$elements
		);
	}

	/**
	 * Convert simplified structure back to Visual Composer format.
	 *
	 * @since  1.3.0
	 * @param  array $simplified Simplified structure.
	 * @return array Visual Composer format.
	 */
	protected function complexify_structure( $simplified ) {
		return array_map(
			function( $item ) {
				$element = array(
					'id'         => uniqid( 'respira_' ),
					'tag'        => $item['type'] ?? 'textBlock',
					'elementType' => 'element',
					'designOptions' => array(),
					'customAttributes' => array(),
					'attributes' => $item['attributes'] ?? array(),
				);

				if ( ! empty( $item['children'] ) ) {
					$element['elements'] = $this->complexify_structure( $item['children'] );
				}

				return $element;
			},
			$simplified
		);
	}
}

