<?php
/**
 * Thrive Architect implementation.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders
 * @since      1.3.0
 */

/**
 * Thrive Architect builder class (PRO feature).
 *
 * @since 1.3.0
 */
class Respira_Builder_Thrive extends Respira_Builder_Interface {

	/**
	 * Detect if Thrive Architect is active.
	 *
	 * @since 1.3.0
	 * @return bool True if Thrive Architect is active.
	 */
	public function detect() {
		return defined( 'TVE_VERSION' ) || class_exists( 'Thrive_Architect' ) || function_exists( 'tcb_get_post_meta' );
	}

	/**
	 * Get builder name.
	 *
	 * @since 1.3.0
	 * @return string Builder name.
	 */
	public function get_name() {
		return 'Thrive Architect';
	}

	/**
	 * Get builder version.
	 *
	 * @since 1.3.0
	 * @return string Builder version.
	 */
	public function get_version() {
		if ( defined( 'TVE_VERSION' ) ) {
			return TVE_VERSION;
		}
		return 'unknown';
	}

	/**
	 * Check if post uses Thrive Architect.
	 *
	 * @since 1.3.0
	 * @param int $post_id Post ID.
	 * @return bool True if uses Thrive Architect.
	 */
	public function is_builder_used( $post_id ) {
		// Check for Thrive meta.
		$tcb_content = get_post_meta( $post_id, 'tcb_editor_enabled', true );
		$tcb_content_data = get_post_meta( $post_id, 'tve_updated_post', true );

		return '1' === $tcb_content || ! empty( $tcb_content_data );
	}

	/**
	 * Extract content from post.
	 *
	 * @since 1.3.0
	 * @param int $post_id Post ID.
	 * @return array Extracted elements.
	 */
	public function extract_content( $post_id ) {
		$start = microtime( true );

		// Thrive stores content in postmeta.
		$tcb_content = get_post_meta( $post_id, 'tve_updated_post', true );

		if ( empty( $tcb_content ) ) {
			return array();
		}

		// Conservative approach - treat as HTML/shortcode combination.
		// Extract text content and basic structure.
		$result = $this->extract_conservative( $tcb_content );

		$this->log_performance( 'extract', $post_id, microtime( true ) - $start );

		return $result;
	}

	/**
	 * Inject content into post.
	 *
	 * @since 1.3.0
	 * @param int   $post_id Post ID.
	 * @param array $content Simplified content.
	 * @return bool|WP_Error True on success.
	 */
	public function inject_content( $post_id, $content ) {
		$start = microtime( true );

		// Validate with intelligence if available.
		if ( class_exists( 'Respira_Thrive_Validator' ) ) {
			$validator = new Respira_Thrive_Validator();
			$validation = $validator->validate_layout( $content );

			if ( ! $validation['valid'] ) {
				$this->log_performance( 'inject', $post_id, microtime( true ) - $start );
				return new WP_Error(
					'validation_failed',
					__( 'Content validation failed.', 'respira-for-wordpress' ),
					array( 'errors' => $validation['errors'] )
				);
			}
		}

		// Conservative approach - only update text content where safe.
		$updated_content = $this->inject_conservative( $post_id, $content );

		if ( is_wp_error( $updated_content ) ) {
			return $updated_content;
		}

		// Update Thrive meta.
		update_post_meta( $post_id, 'tve_updated_post', $updated_content );
		update_post_meta( $post_id, 'tcb_editor_enabled', '1' );

		$this->log_performance( 'inject', $post_id, microtime( true ) - $start );

		return true;
	}

	/**
	 * Create a code block.
	 *
	 * @since 1.3.0
	 * @param int    $post_id Post ID.
	 * @param string $html    HTML content.
	 * @param string $css     CSS content.
	 * @param string $js      JavaScript content.
	 * @return bool|WP_Error True on success.
	 */
	public function create_code_block( $post_id, $html, $css = '', $js = '' ) {
		$tcb_content = get_post_meta( $post_id, 'tve_updated_post', true );

		// Combine code.
		$combined_code = $html;
		if ( ! empty( $css ) ) {
			$combined_code .= "\n\n<style>\n" . $css . "\n</style>";
		}
		if ( ! empty( $js ) ) {
			$combined_code .= "\n\n<script>\n" . $js . "\n</script>";
		}

		// Append to existing content.
		$updated_content = $tcb_content . "\n" . $combined_code;

		update_post_meta( $post_id, 'tve_updated_post', $updated_content );

		return true;
	}

	/**
	 * Get documentation.
	 *
	 * @since 1.3.0
	 * @return array Documentation.
	 */
	public function get_documentation() {
		$docs = array(
			'name'        => 'Thrive Architect',
			'description' => 'Visual front-end builder by Thrive Themes',
			'overview'    => 'Thrive Architect uses visual editor. Limited public API - conservative support.',
			'structure'   => array(
				'elements' => 'Conversion-focused elements',
			),
			'resources'   => array(
				'https://thrivethemes.com/docs-categories/developer-documentation/',
			),
			'note'        => __( 'Limited support due to undocumented API. Safe text-level edits only.', 'respira-for-wordpress' ),
		);

		// Add patterns if intelligence is available.
		if ( function_exists( 'respira_get_thrive_patterns' ) ) {
			$docs['patterns'] = respira_get_thrive_patterns();
		}

		// Add element catalog if intelligence is available.
		if ( class_exists( 'Respira_Thrive_Element_Registry' ) ) {
			$docs['elements'] = Respira_Thrive_Element_Registry::get_all_elements();
		}

		return $docs;
	}

	/**
	 * Get available modules.
	 *
	 * @since 1.3.0
	 * @return array Available elements.
	 */
	public function get_available_modules() {
		// Use dynamic element registry if available.
		if ( class_exists( 'Respira_Thrive_Element_Registry' ) ) {
			return Respira_Thrive_Element_Registry::get_all_elements();
		}

		// Fallback to basic list (known elements).
		return array(
			array(
				'name'        => 'text',
				'title'       => 'Text',
				'description' => 'Text content element',
			),
			array(
				'name'        => 'button',
				'title'       => 'Button',
				'description' => 'Button element',
			),
			array(
				'name'        => 'image',
				'title'       => 'Image',
				'description' => 'Image element',
			),
		);
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since 1.4.0
	 * @param array $elements_used Optional. Array of element names used on the page.
	 * @return array Builder schema with element information.
	 */
	public function get_builder_schema( $elements_used = array() ) {
		if ( class_exists( 'Respira_Thrive_Element_Schema' ) ) {
			$schema_generator = new Respira_Thrive_Element_Schema();
			return $schema_generator->get_builder_schema( $elements_used );
		}

		// Fallback to basic schema.
		return array(
			'builder'         => 'thrive',
			'builder_version' => $this->get_version(),
			'elements'        => $this->get_available_modules(),
		);
	}

	/**
	 * Check if intelligence is available.
	 *
	 * @since 1.4.0
	 * @return bool True if intelligence is available.
	 */
	public function is_intelligence_available() {
		return class_exists( 'Respira_Thrive_Intelligence_Loader' ) &&
			   class_exists( 'Respira_Thrive_Element_Registry' ) &&
			   class_exists( 'Respira_Thrive_Element_Schema' ) &&
			   class_exists( 'Respira_Thrive_Validator' );
	}

	/**
	 * Extract content conservatively.
	 *
	 * @since  1.3.0
	 * @param  string $content Thrive content.
	 * @return array Simplified structure.
	 */
	private function extract_conservative( $content ) {
		// Extract text nodes and basic structure.
		$elements = array();

		// Simple text extraction.
		$text_elements = preg_split( '/<[^>]+>/', $content );
		foreach ( $text_elements as $text ) {
			$text = trim( $text );
			if ( ! empty( $text ) ) {
				$elements[] = array(
					'type'       => 'text',
					'attributes' => array(),
					'content'    => $text,
					'children'   => array(),
				);
			}
		}

		return $elements;
	}

	/**
	 * Inject content conservatively.
	 *
	 * @since  1.3.0
	 * @param  int   $post_id Post ID.
	 * @param  array $content Simplified content.
	 * @return string|WP_Error Updated content or error.
	 */
	private function inject_conservative( $post_id, $content ) {
		$original = get_post_meta( $post_id, 'tve_updated_post', true );

		// Only update text content, preserve structure.
		foreach ( $content as $item ) {
			if ( isset( $item['type'] ) && 'text' === $item['type'] && isset( $item['content'] ) ) {
				// Simple text replacement (conservative).
				// In production, this would be more sophisticated.
			}
		}

		// For now, return original with minimal changes.
		return $original;
	}

	/**
	 * Simplify Thrive structure.
	 *
	 * @since  1.3.0
	 * @param  array $elements Thrive elements.
	 * @return array Simplified structure.
	 */
	protected function simplify_structure( $elements ) {
		// Conservative implementation.
		return $elements;
	}

	/**
	 * Convert simplified structure back to Thrive format.
	 *
	 * @since  1.3.0
	 * @param  array $simplified Simplified structure.
	 * @return array Thrive format.
	 */
	protected function complexify_structure( $simplified ) {
		// Conservative implementation.
		return $simplified;
	}
}

