<?php
/**
 * Oxygen page builder implementation (PRO).
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders
 */

/**
 * Oxygen builder class (PRO feature).
 *
 * @since 1.0.0
 */
class Respira_Builder_Oxygen extends Respira_Builder_Interface {

	/**
	 * Detect if Oxygen is active.
	 *
	 * @since 1.0.0
	 * @return bool True if Oxygen is active.
	 */
	public function detect() {
		return defined( 'CT_VERSION' ) || class_exists( 'CT_Component' );
	}

	/**
	 * Get builder name.
	 *
	 * @since 1.0.0
	 * @return string Builder name.
	 */
	public function get_name() {
		return 'Oxygen';
	}

	/**
	 * Get builder version.
	 *
	 * @since 1.0.0
	 * @return string Builder version.
	 */
	public function get_version() {
		return defined( 'CT_VERSION' ) ? CT_VERSION : 'unknown';
	}

	/**
	 * Check if post uses Oxygen.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if uses Oxygen.
	 */
	public function is_builder_used( $post_id ) {
		$shortcodes = get_post_meta( $post_id, 'ct_builder_shortcodes', true );
		return ! empty( $shortcodes );
	}

	/**
	 * Extract content from post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Extracted components.
	 */
	public function extract_content( $post_id ) {
		// Oxygen stores data in multiple meta keys.
		$shortcodes = get_post_meta( $post_id, 'ct_builder_shortcodes', true );
		$json_data  = get_post_meta( $post_id, 'ct_builder_json', true );

		if ( ! empty( $json_data ) ) {
			$components = json_decode( $json_data, true );
			return $this->simplify_structure( $components );
		}

		// Fallback to shortcodes parsing.
		if ( ! empty( $shortcodes ) ) {
			return $this->parse_oxygen_shortcodes( $shortcodes );
		}

		return array();
	}

	/**
	 * Inject content into post.
	 *
	 * @since 1.0.0
	 * @param int   $post_id Post ID.
	 * @param array $content Simplified content.
	 * @return bool|WP_Error True on success.
	 */
	public function inject_content( $post_id, $content ) {
		$start = microtime( true );

		// Validate with intelligence if available.
		if ( class_exists( 'Respira_Oxygen_Validator' ) ) {
			$validator = new Respira_Oxygen_Validator();
			$validation = $validator->validate_layout( $content );

			if ( ! $validation['valid'] ) {
				$this->log_performance( 'inject', $post_id, microtime( true ) - $start );
				return new WP_Error(
					'validation_failed',
					__( 'Content validation failed.', 'respira-for-wordpress' ),
					array( 'errors' => $validation['errors'] )
				);
			}
		}

		$components = $this->complexify_structure( $content );

		// Store as JSON.
		update_post_meta( $post_id, 'ct_builder_json', wp_json_encode( $components ) );

		// Generate shortcodes for compatibility.
		$shortcodes = $this->build_oxygen_shortcodes( $components );
		update_post_meta( $post_id, 'ct_builder_shortcodes', $shortcodes );

		// Mark as Oxygen-built.
		update_post_meta( $post_id, 'ct_builder_active', 'true' );

		// Clear Oxygen cache if available.
		if ( function_exists( 'oxygen_vsb_cache_page_update' ) ) {
			oxygen_vsb_cache_page_update( $post_id );
		}

		$this->log_performance( 'inject', $post_id, microtime( true ) - $start );

		return true;
	}

	/**
	 * Create a code block.
	 *
	 * @since 1.0.0
	 * @param int    $post_id Post ID.
	 * @param string $html    HTML content.
	 * @param string $css     CSS content.
	 * @param string $js      JavaScript content.
	 * @return bool|WP_Error True on success.
	 */
	public function create_code_block( $post_id, $html, $css = '', $js = '' ) {
		$json_data = get_post_meta( $post_id, 'ct_builder_json', true );
		$components = json_decode( $json_data, true ) ?: array();

		// Create code block component.
		$code_block = array(
			'id'       => 'respira_' . uniqid(),
			'name'     => 'ct_code_block',
			'options'  => array(
				'ct_content' => $html,
			),
			'children' => array(),
		);

		// Add CSS if provided.
		if ( ! empty( $css ) ) {
			$code_block['options']['ct_css'] = $css;
		}

		// Add JS if provided.
		if ( ! empty( $js ) ) {
			$code_block['options']['ct_js'] = $js;
		}

		$components[] = $code_block;

		update_post_meta( $post_id, 'ct_builder_json', wp_json_encode( $components ) );

		return true;
	}

	/**
	 * Get documentation.
	 *
	 * @since 1.0.0
	 * @return array Documentation.
	 */
	public function get_documentation() {
		$docs = array(
			'name'        => 'Oxygen',
			'description' => 'Developer-focused visual website builder',
			'overview'    => 'Oxygen stores components as JSON with options for advanced customization.',
			'structure'   => array(
				'sections'   => 'Structural containers',
				'components' => 'UI elements with extensive customization',
				'classes'    => 'Reusable class system',
			),
			'resources'   => array(
				'https://oxygenbuilder.com/documentation/',
			),
		);

		// Add patterns if intelligence is available.
		if ( function_exists( 'respira_get_oxygen_patterns' ) ) {
			$docs['patterns'] = respira_get_oxygen_patterns();
		}

		// Add component catalog if intelligence is available.
		if ( class_exists( 'Respira_Oxygen_Component_Registry' ) ) {
			$docs['components'] = Respira_Oxygen_Component_Registry::get_all_components();
		}

		return $docs;
	}

	/**
	 * Get available modules.
	 *
	 * @since 1.0.0
	 * @return array Available components.
	 */
	public function get_available_modules() {
		// Use dynamic component registry if available.
		if ( class_exists( 'Respira_Oxygen_Component_Registry' ) ) {
			return Respira_Oxygen_Component_Registry::get_all_components();
		}

		// Fallback to basic list (known components).
		return array(
			array(
				'name'        => 'ct_section',
				'title'       => 'Section',
				'description' => 'Main structural container',
			),
			array(
				'name'        => 'ct_div_block',
				'title'       => 'Div Block',
				'description' => 'Generic container element',
			),
			array(
				'name'        => 'ct_text_block',
				'title'       => 'Text Block',
				'description' => 'Text content element',
			),
			array(
				'name'        => 'ct_headline',
				'title'       => 'Headline',
				'description' => 'Heading element',
			),
			array(
				'name'        => 'ct_image',
				'title'       => 'Image',
				'description' => 'Image element',
			),
			array(
				'name'        => 'ct_link_wrapper',
				'title'       => 'Link Wrapper',
				'description' => 'Link container',
			),
			array(
				'name'        => 'ct_code_block',
				'title'       => 'Code Block',
				'description' => 'Custom HTML/CSS/JS',
			),
		);
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since 1.4.0
	 * @param array $elements_used Optional. Array of element names used on the page.
	 * @return array Builder schema with element information.
	 */
	public function get_builder_schema( $elements_used = array() ) {
		if ( class_exists( 'Respira_Oxygen_Component_Schema' ) ) {
			$schema_generator = new Respira_Oxygen_Component_Schema();
			return $schema_generator->get_builder_schema( $elements_used );
		}

		// Fallback to basic schema.
		return array(
			'builder'         => 'oxygen',
			'builder_version' => $this->get_version(),
			'components'      => $this->get_available_modules(),
		);
	}

	/**
	 * Check if intelligence is available.
	 *
	 * @since 1.4.0
	 * @return bool True if intelligence is available.
	 */
	public function is_intelligence_available() {
		return class_exists( 'Respira_Oxygen_Intelligence_Loader' ) &&
			   class_exists( 'Respira_Oxygen_Component_Registry' ) &&
			   class_exists( 'Respira_Oxygen_Component_Schema' ) &&
			   class_exists( 'Respira_Oxygen_Validator' );
	}

	/**
	 * Parse Oxygen shortcodes.
	 *
	 * @since  1.0.0
	 * @param  string $shortcodes Oxygen shortcodes.
	 * @return array  Parsed structure.
	 */
	private function parse_oxygen_shortcodes( $shortcodes ) {
		// Simplified parser - in production, use Oxygen's own parser.
		$components = array();

		// Basic shortcode pattern matching.
		preg_match_all( '/\[(\w+)([^\]]*)\](.*?)\[\/\1\]/s', $shortcodes, $matches, PREG_SET_ORDER );

		foreach ( $matches as $match ) {
			$components[] = array(
				'name'    => $match[1],
				'attrs'   => $this->parse_shortcode_attrs( $match[2] ),
				'content' => $match[3],
			);
		}

		return $components;
	}

	/**
	 * Build Oxygen shortcodes from components.
	 *
	 * @since  1.0.0
	 * @param  array $components Components array.
	 * @return string Oxygen shortcodes.
	 */
	private function build_oxygen_shortcodes( $components ) {
		$shortcodes = '';

		foreach ( $components as $component ) {
			$name = $component['name'] ?? 'ct_div_block';
			$options = $component['options'] ?? array();

			$attrs = '';
			foreach ( $options as $key => $value ) {
				$attrs .= ' ' . $key . '="' . esc_attr( $value ) . '"';
			}

			$shortcodes .= '[' . $name . $attrs . ']';

			if ( ! empty( $component['children'] ) ) {
				$shortcodes .= $this->build_oxygen_shortcodes( $component['children'] );
			}

			$shortcodes .= '[/' . $name . ']';
		}

		return $shortcodes;
	}

	/**
	 * Parse shortcode attributes.
	 *
	 * @since  1.0.0
	 * @param  string $attr_string Attribute string.
	 * @return array  Attributes array.
	 */
	private function parse_shortcode_attrs( $attr_string ) {
		$attrs = array();
		preg_match_all( '/(\w+)="([^"]*)"/s', $attr_string, $matches, PREG_SET_ORDER );

		foreach ( $matches as $match ) {
			$attrs[ $match[1] ] = $match[2];
		}

		return $attrs;
	}

	/**
	 * Simplify Oxygen structure.
	 *
	 * @since  1.0.0
	 * @param  array $components Oxygen components.
	 * @return array Simplified structure.
	 */
	protected function simplify_structure( $components ) {
		if ( ! is_array( $components ) ) {
			return array();
		}

		return array_map(
			function( $component ) {
				return array(
					'type'     => $component['name'] ?? 'div',
					'options'  => $component['options'] ?? array(),
					'children' => isset( $component['children'] ) ? $this->simplify_structure( $component['children'] ) : array(),
				);
			},
			$components
		);
	}

	/**
	 * Convert simplified structure back to Oxygen format.
	 *
	 * @since  1.0.0
	 * @param  array $simplified Simplified structure.
	 * @return array Oxygen format.
	 */
	protected function complexify_structure( $simplified ) {
		if ( ! is_array( $simplified ) ) {
			return array();
		}

		return array_map(
			function( $item ) {
				$component = array(
					'id'      => 'respira_' . uniqid(),
					'name'    => $item['type'] ?? 'ct_div_block',
					'options' => $item['options'] ?? array(),
				);

				if ( ! empty( $item['children'] ) ) {
					$component['children'] = $this->complexify_structure( $item['children'] );
				} else {
					$component['children'] = array();
				}

				return $component;
			},
			$simplified
		);
	}
}
