<?php
/**
 * Page builder interface and factory.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders
 */

/**
 * Page builder interface.
 *
 * All page builder implementations must implement this interface.
 *
 * @since 1.0.0
 */
abstract class Respira_Builder_Interface {

	/**
	 * Detect if this builder is active on the site.
	 *
	 * @since 1.0.0
	 * @return bool True if builder is active.
	 */
	abstract public function detect();

	/**
	 * Get the builder name.
	 *
	 * @since 1.0.0
	 * @return string Builder name.
	 */
	abstract public function get_name();

	/**
	 * Get the builder version.
	 *
	 * @since 1.0.0
	 * @return string Builder version.
	 */
	abstract public function get_version();

	/**
	 * Extract content from a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Extracted content in simplified format.
	 */
	abstract public function extract_content( $post_id );

	/**
	 * Inject content into a post.
	 *
	 * @since 1.0.0
	 * @param int   $post_id Post ID.
	 * @param array $content Content to inject.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	abstract public function inject_content( $post_id, $content );

	/**
	 * Create a code block in the builder.
	 *
	 * @since 1.0.0
	 * @param int    $post_id Post ID.
	 * @param string $html    HTML content.
	 * @param string $css     Optional. CSS content.
	 * @param string $js      Optional. JavaScript content.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	abstract public function create_code_block( $post_id, $html, $css = '', $js = '' );

	/**
	 * Get builder documentation for AI.
	 *
	 * @since 1.0.0
	 * @return array Documentation array.
	 */
	abstract public function get_documentation();

	/**
	 * Get available modules/widgets/blocks.
	 *
	 * @since 1.0.0
	 * @return array Available modules.
	 */
	abstract public function get_available_modules();

	/**
	 * Get builder schema for AI context.
	 *
	 * @since  1.3.0
	 * @param  array $modules_used Optional. Array of module names used on the page.
	 * @return array Builder schema with module information.
	 */
	public function get_builder_schema( $modules_used = array() ) {
		// Default implementation - can be overridden.
		return array(
			'builder'          => $this->get_name(),
			'builder_version'  => $this->get_version(),
			'available_modules' => $this->get_available_modules(),
			'quick_reference'  => array(
				'colors'  => __( 'Use hex format: #0000FF', 'respira-for-wordpress' ),
				'spacing' => __( 'Format: 10px|20px|10px|20px (top|right|bottom|left)', 'respira-for-wordpress' ),
				'sizing'  => __( 'Use px, em, %, or vw', 'respira-for-wordpress' ),
				'fonts'   => __( 'Font names from Google Fonts or system fonts', 'respira-for-wordpress' ),
			),
		);
	}

	/**
	 * Check if intelligence is available for this builder.
	 *
	 * @since  1.3.0
	 * @return bool True if intelligence package is available.
	 */
	public function is_intelligence_available() {
		$builder_name = strtolower( $this->get_name() );
		$loader_path  = RESPIRA_PLUGIN_DIR . 'includes/' . $builder_name . '-intelligence/class-' . $builder_name . '-intelligence-loader.php';

		return file_exists( $loader_path );
	}

	/**
	 * Detect which builder is used on a specific post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Optional. Post ID to check. If not provided, checks globally.
	 * @return Respira_Builder_Interface|null Builder instance or null.
	 */
	public static function detect_builder( $post_id = null ) {
		$builders = self::get_available_builders();

		foreach ( $builders as $builder_class ) {
			$builder = new $builder_class();

			if ( $builder->detect() ) {
				// If checking specific post, verify it uses this builder.
				if ( $post_id ) {
					if ( $builder->is_builder_used( $post_id ) ) {
						return $builder;
					}
				} else {
					return $builder;
				}
			}
		}

		return null;
	}

	/**
	 * Get a builder instance by name.
	 *
	 * @since 1.0.0
	 * @param string $name Builder name.
	 * @return Respira_Builder_Interface|WP_Error Builder instance or error.
	 */
	public static function get_builder( $name ) {
		$builders = self::get_available_builders();

		foreach ( $builders as $builder_class ) {
			$builder = new $builder_class();

			if ( strtolower( $builder->get_name() ) === strtolower( $name ) ) {
				if ( ! $builder->detect() ) {
					return new WP_Error(
						'respira_builder_not_active',
						sprintf(
							/* translators: %s: builder name */
							__( '%s is not active on this site.', 'respira-for-wordpress' ),
							$builder->get_name()
						)
					);
				}

				return $builder;
			}
		}

		return new WP_Error(
			'respira_builder_not_found',
			sprintf(
				/* translators: %s: builder name */
				__( 'Builder "%s" not supported.', 'respira-for-wordpress' ),
				$name
			)
		);
	}

	/**
	 * Get all available builder classes.
	 *
	 * @since 1.0.0
	 * @return array Builder class names.
	 */
	private static function get_available_builders() {
		return array(
			'Respira_Builder_Gutenberg',
			'Respira_Builder_Divi',
			'Respira_Builder_Elementor',
			'Respira_Builder_WPBakery',
			'Respira_Builder_Oxygen',
			'Respira_Builder_Bricks',
			'Respira_Builder_Beaver',
			'Respira_Builder_Brizy',
			'Respira_Builder_Visual_Composer',
			'Respira_Builder_Thrive',
		);
	}

	/**
	 * Check if this builder is used on a specific post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if builder is used.
	 */
	public function is_builder_used( $post_id ) {
		// Default implementation - can be overridden.
		return false;
	}

	/**
	 * Helper method to simplify block/module structure for AI.
	 *
	 * @since  1.0.0
	 * @param  array $builder_data Raw builder data.
	 * @return array Simplified structure.
	 */
	protected function simplify_structure( $builder_data ) {
		// To be implemented by each builder.
		return $builder_data;
	}

	/**
	 * Helper method to convert simplified structure back to builder format.
	 *
	 * @since  1.0.0
	 * @param  array $simplified_data Simplified data from AI.
	 * @return array Builder-specific format.
	 */
	protected function complexify_structure( $simplified_data ) {
		// To be implemented by each builder.
		return $simplified_data;
	}

	/**
	 * Log performance metrics.
	 *
	 * @since  1.3.0
	 * @param  string $operation Operation name.
	 * @param  int    $post_id   Post ID.
	 * @param  float  $time      Time taken in seconds.
	 * @param  float  $threshold Threshold in seconds (default 2.0).
	 */
	protected function log_performance( $operation, $post_id, $time, $threshold = 2.0 ) {
		if ( $time > $threshold ) {
			error_log(
				sprintf(
					/* translators: 1: builder name, 2: operation, 3: post ID, 4: time in seconds */
					'Respira: Slow %s %s - Post ID %d took %.2fs',
					$this->get_name(),
					$operation,
					$post_id,
					$time
				)
			);
		}
	}
}

// Load all builder implementations.
require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/class-builder-gutenberg.php';
require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/class-builder-divi.php';
require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/class-builder-elementor.php';
require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/class-builder-wpbakery.php';
require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/class-builder-oxygen.php';
require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/class-builder-bricks.php';
require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/class-builder-beaver.php';
require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/class-builder-brizy.php';
require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/class-builder-visual-composer.php';
require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/class-builder-thrive.php';

// Auto-load intelligence packages for all builders.
$intelligence_builders = array(
	'gutenberg'       => 'Gutenberg',
	'divi'            => 'Divi',
	'elementor'       => 'Elementor',
	'wpbakery'        => 'WPBakery',
	'oxygen'          => 'Oxygen',
	'bricks'          => 'Bricks',
	'beaver'          => 'Beaver',
	'brizy'           => 'Brizy',
	'visual-composer' => 'Visual_Composer',
	'thrive'          => 'Thrive',
);

foreach ( $intelligence_builders as $folder => $class_name ) {
	$loader_path = RESPIRA_PLUGIN_DIR . 'includes/' . $folder . '-intelligence/class-' . str_replace( '_', '-', strtolower( $class_name ) ) . '-intelligence-loader.php';
	if ( file_exists( $loader_path ) ) {
		require_once $loader_path;
		$loader_class = 'Respira_' . $class_name . '_Intelligence_Loader';
		if ( class_exists( $loader_class ) && method_exists( $loader_class, 'load' ) ) {
			call_user_func( array( $loader_class, 'load' ) );
		}
	}
}
