<?php
/**
 * Bricks page builder implementation (PRO).
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders
 */

/**
 * Bricks builder class (PRO feature).
 *
 * @since 1.0.0
 */
class Respira_Builder_Bricks extends Respira_Builder_Interface {

	/**
	 * Detect if Bricks is active.
	 *
	 * @since 1.0.0
	 * @return bool True if Bricks is active.
	 */
	public function detect() {
		return defined( 'BRICKS_VERSION' ) || class_exists( 'Bricks\Database' );
	}

	/**
	 * Get builder name.
	 *
	 * @since 1.0.0
	 * @return string Builder name.
	 */
	public function get_name() {
		return 'Bricks';
	}

	/**
	 * Get builder version.
	 *
	 * @since 1.0.0
	 * @return string Builder version.
	 */
	public function get_version() {
		return defined( 'BRICKS_VERSION' ) ? BRICKS_VERSION : 'unknown';
	}

	/**
	 * Check if post uses Bricks.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return bool True if uses Bricks.
	 */
	public function is_builder_used( $post_id ) {
		$content = get_post_meta( $post_id, '_bricks_page_content_2', true );
		return ! empty( $content ) && is_array( $content );
	}

	/**
	 * Extract content from post.
	 *
	 * @since 1.0.0
	 * @param int $post_id Post ID.
	 * @return array Extracted elements.
	 */
	public function extract_content( $post_id ) {
		// Bricks stores content in _bricks_page_content_2 as serialized array.
		$content = get_post_meta( $post_id, '_bricks_page_content_2', true );

		if ( empty( $content ) || ! is_array( $content ) ) {
			return array();
		}

		return $this->simplify_structure( $content );
	}

	/**
	 * Inject content into post.
	 *
	 * @since 1.0.0
	 * @param int   $post_id Post ID.
	 * @param array $content Simplified content.
	 * @return bool|WP_Error True on success.
	 */
	public function inject_content( $post_id, $content ) {
		$start = microtime( true );

		// Validate with intelligence if available.
		if ( class_exists( 'Respira_Bricks_Validator' ) ) {
			$validator = new Respira_Bricks_Validator();
			$validation = $validator->validate_layout( $content );

			if ( ! $validation['valid'] ) {
				$this->log_performance( 'inject', $post_id, microtime( true ) - $start );
				return new WP_Error(
					'validation_failed',
					__( 'Content validation failed.', 'respira-for-wordpress' ),
					array( 'errors' => $validation['errors'] )
				);
			}
		}

		// Convert simplified structure back to Bricks format.
		$elements = $this->complexify_structure( $content );

		// Update Bricks content meta.
		update_post_meta( $post_id, '_bricks_page_content_2', $elements );

		// Mark as Bricks-built.
		update_post_meta( $post_id, '_bricks_editor_mode', 'bricks' );

		// Clear Bricks cache if available.
		if ( function_exists( 'bricks_clear_page_cache' ) ) {
			bricks_clear_page_cache( $post_id );
		}

		$this->log_performance( 'inject', $post_id, microtime( true ) - $start );

		return true;
	}

	/**
	 * Create a code block.
	 *
	 * @since 1.0.0
	 * @param int    $post_id Post ID.
	 * @param string $html    HTML content.
	 * @param string $css     CSS content.
	 * @param string $js      JavaScript content.
	 * @return bool|WP_Error True on success.
	 */
	public function create_code_block( $post_id, $html, $css = '', $js = '' ) {
		$content = get_post_meta( $post_id, '_bricks_page_content_2', true );
		$elements = is_array( $content ) ? $content : array();

		// Create code element.
		$code_element = array(
			'id'       => 'respira_' . uniqid(),
			'name'     => 'code',
			'parent'   => 0,
			'settings' => array(
				'code' => $html,
			),
		);

		// Add CSS if provided.
		if ( ! empty( $css ) ) {
			$code_element['settings']['css'] = $css;
		}

		// Add JS if provided.
		if ( ! empty( $js ) ) {
			$code_element['settings']['executeCode'] = true;
			$code_element['settings']['code'] .= "\n<script>\n" . $js . "\n</script>";
		}

		// Add CSS via style tag if provided.
		if ( ! empty( $css ) ) {
			$code_element['settings']['code'] .= "\n<style>\n" . $css . "\n</style>";
		}

		$elements[] = $code_element;

		update_post_meta( $post_id, '_bricks_page_content_2', $elements );

		return true;
	}

	/**
	 * Get documentation.
	 *
	 * @since 1.0.0
	 * @return array Documentation.
	 */
	public function get_documentation() {
		$docs = array(
			'name'        => 'Bricks',
			'description' => 'Modern, performance-focused visual website builder',
			'overview'    => 'Bricks uses a element-based structure stored as arrays. Focus on speed and clean code output.',
			'structure'   => array(
				'elements'   => 'Individual UI elements',
				'containers' => 'Structural containers',
				'settings'   => 'Element-specific settings',
			),
			'resources'   => array(
				'https://academy.bricksbuilder.io/',
			),
		);

		// Add patterns if intelligence is available.
		if ( function_exists( 'respira_get_bricks_patterns' ) ) {
			$docs['patterns'] = respira_get_bricks_patterns();
		}

		// Add element catalog if intelligence is available.
		if ( class_exists( 'Respira_Bricks_Element_Registry' ) ) {
			$docs['elements'] = Respira_Bricks_Element_Registry::get_all_elements();
		}

		return $docs;
	}

	/**
	 * Get available modules.
	 *
	 * @since 1.0.0
	 * @return array Available elements.
	 */
	public function get_available_modules() {
		// Use dynamic element registry if available.
		if ( class_exists( 'Respira_Bricks_Element_Registry' ) ) {
			return Respira_Bricks_Element_Registry::get_all_elements();
		}

		// Fallback to basic list (known elements).
		return array(
			array(
				'name'        => 'section',
				'title'       => 'Section',
				'description' => 'Main structural container',
			),
			array(
				'name'        => 'container',
				'title'       => 'Container',
				'description' => 'Generic container element',
			),
			array(
				'name'        => 'block',
				'title'       => 'Block',
				'description' => 'Basic block element',
			),
			array(
				'name'        => 'div',
				'title'       => 'Div',
				'description' => 'Division element',
			),
			array(
				'name'        => 'text',
				'title'       => 'Text',
				'description' => 'Text content',
			),
			array(
				'name'        => 'heading',
				'title'       => 'Heading',
				'description' => 'Heading element',
			),
			array(
				'name'        => 'image',
				'title'       => 'Image',
				'description' => 'Image element',
			),
			array(
				'name'        => 'video',
				'title'       => 'Video',
				'description' => 'Video element',
			),
			array(
				'name'        => 'button',
				'title'       => 'Button',
				'description' => 'Button element',
			),
			array(
				'name'        => 'icon',
				'title'       => 'Icon',
				'description' => 'Icon element',
			),
			array(
				'name'        => 'code',
				'title'       => 'Code',
				'description' => 'Custom HTML/CSS/JS',
			),
			array(
				'name'        => 'html',
				'title'       => 'HTML',
				'description' => 'Raw HTML element',
			),
		);
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since 1.4.0
	 * @param array $elements_used Optional. Array of element names used on the page.
	 * @return array Builder schema with element information.
	 */
	public function get_builder_schema( $elements_used = array() ) {
		if ( class_exists( 'Respira_Bricks_Element_Schema' ) ) {
			$schema_generator = new Respira_Bricks_Element_Schema();
			return $schema_generator->get_builder_schema( $elements_used );
		}

		// Fallback to basic schema.
		return array(
			'builder'         => 'bricks',
			'builder_version' => $this->get_version(),
			'elements'        => $this->get_available_modules(),
		);
	}

	/**
	 * Check if intelligence is available.
	 *
	 * @since 1.4.0
	 * @return bool True if intelligence is available.
	 */
	public function is_intelligence_available() {
		return class_exists( 'Respira_Bricks_Intelligence_Loader' ) &&
			   class_exists( 'Respira_Bricks_Element_Registry' ) &&
			   class_exists( 'Respira_Bricks_Element_Schema' ) &&
			   class_exists( 'Respira_Bricks_Validator' );
	}

	/**
	 * Simplify Bricks structure for AI.
	 *
	 * @since  1.0.0
	 * @param  array $elements Bricks elements.
	 * @return array Simplified structure.
	 */
	protected function simplify_structure( $elements ) {
		if ( ! is_array( $elements ) ) {
			return array();
		}

		return array_map(
			function( $element ) {
				$simplified = array(
					'type'     => $element['name'] ?? 'div',
					'id'       => $element['id'] ?? '',
					'parent'   => $element['parent'] ?? 0,
					'settings' => $element['settings'] ?? array(),
				);

				// Include label if present.
				if ( isset( $element['label'] ) ) {
					$simplified['label'] = $element['label'];
				}

				// Include children if this element has them.
				if ( isset( $element['children'] ) && is_array( $element['children'] ) ) {
					$simplified['children'] = $this->simplify_structure( $element['children'] );
				}

				return $simplified;
			},
			$elements
		);
	}

	/**
	 * Convert simplified structure back to Bricks format.
	 *
	 * @since  1.0.0
	 * @param  array $simplified Simplified structure.
	 * @return array Bricks format.
	 */
	protected function complexify_structure( $simplified ) {
		if ( ! is_array( $simplified ) ) {
			return array();
		}

		return array_map(
			function( $item ) {
				$element = array(
					'id'       => $item['id'] ?? 'respira_' . uniqid(),
					'name'     => $item['type'] ?? 'div',
					'parent'   => $item['parent'] ?? 0,
					'settings' => $item['settings'] ?? array(),
				);

				// Include label if present.
				if ( isset( $item['label'] ) ) {
					$element['label'] = $item['label'];
				}

				// Process children if present.
				if ( ! empty( $item['children'] ) ) {
					$element['children'] = $this->complexify_structure( $item['children'] );
				}

				return $element;
			},
			$simplified
		);
	}
}
