<?php
/**
 * Beaver Builder implementation.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders
 * @since      1.3.0
 */

/**
 * Beaver Builder class (PRO feature).
 *
 * @since 1.3.0
 */
class Respira_Builder_Beaver extends Respira_Builder_Interface {

	/**
	 * Detect if Beaver Builder is active.
	 *
	 * @since 1.3.0
	 * @return bool True if Beaver Builder is active.
	 */
	public function detect() {
		return class_exists( 'FLBuilder' ) || defined( 'FL_BUILDER_VERSION' );
	}

	/**
	 * Get builder name.
	 *
	 * @since 1.3.0
	 * @return string Builder name.
	 */
	public function get_name() {
		return 'Beaver Builder';
	}

	/**
	 * Get builder version.
	 *
	 * @since 1.3.0
	 * @return string Builder version.
	 */
	public function get_version() {
		if ( defined( 'FL_BUILDER_VERSION' ) ) {
			return FL_BUILDER_VERSION;
		}
		return 'unknown';
	}

	/**
	 * Check if post uses Beaver Builder.
	 *
	 * @since 1.3.0
	 * @param int $post_id Post ID.
	 * @return bool True if uses Beaver Builder.
	 */
	public function is_builder_used( $post_id ) {
		$data = get_post_meta( $post_id, '_fl_builder_data', true );
		return ! empty( $data );
	}

	/**
	 * Extract content from post.
	 *
	 * @since 1.3.0
	 * @param int $post_id Post ID.
	 * @return array Extracted modules.
	 */
	public function extract_content( $post_id ) {
		$start = microtime( true );

		$data = get_post_meta( $post_id, '_fl_builder_data', true );

		if ( empty( $data ) ) {
			return array();
		}

		$modules = json_decode( $data, true );

		if ( ! is_array( $modules ) ) {
			return array();
		}

		$result = $this->simplify_structure( $modules );

		$this->log_performance( 'extract', $post_id, microtime( true ) - $start );

		return $result;
	}

	/**
	 * Inject content into post.
	 *
	 * @since 1.3.0
	 * @param int   $post_id Post ID.
	 * @param array $content Simplified content.
	 * @return bool|WP_Error True on success.
	 */
	public function inject_content( $post_id, $content ) {
		$start = microtime( true );

		// Validate with intelligence if available.
		if ( class_exists( 'Respira_Beaver_Validator' ) ) {
			$validator = new Respira_Beaver_Validator();
			$validation = $validator->validate_layout( $content );

			if ( ! $validation['valid'] ) {
				$this->log_performance( 'inject', $post_id, microtime( true ) - $start );
				return new WP_Error(
					'validation_failed',
					__( 'Content validation failed.', 'respira-for-wordpress' ),
					array( 'errors' => $validation['errors'] )
				);
			}
		}

		// Convert simplified structure back to Beaver Builder format.
		$modules = $this->complexify_structure( $content );

		// Update postmeta.
		update_post_meta( $post_id, '_fl_builder_data', wp_json_encode( $modules ) );
		update_post_meta( $post_id, '_fl_builder_enabled', '1' );
		update_post_meta( $post_id, '_fl_builder_draft', '' );

		// Clear Beaver Builder cache.
		if ( class_exists( 'FLBuilder' ) && method_exists( 'FLBuilder', 'delete_all_asset_cache' ) ) {
			FLBuilder::delete_all_asset_cache( $post_id );
		}

		$this->log_performance( 'inject', $post_id, microtime( true ) - $start );

		return true;
	}

	/**
	 * Create a code block.
	 *
	 * @since 1.3.0
	 * @param int    $post_id Post ID.
	 * @param string $html    HTML content.
	 * @param string $css     CSS content.
	 * @param string $js      JavaScript content.
	 * @return bool|WP_Error True on success.
	 */
	public function create_code_block( $post_id, $html, $css = '', $js = '' ) {
		$data = get_post_meta( $post_id, '_fl_builder_data', true );
		$modules = json_decode( $data, true ) ?: array();

		// Combine code.
		$combined_code = $html;
		if ( ! empty( $css ) ) {
			$combined_code .= "\n\n<style>\n" . $css . "\n</style>";
		}
		if ( ! empty( $js ) ) {
			$combined_code .= "\n\n<script>\n" . $js . "\n</script>";
		}

		// Create HTML module.
		$html_module = array(
			'type'     => 'html',
			'settings' => array(
				'html' => $combined_code,
			),
		);

		$modules[] = $html_module;

		update_post_meta( $post_id, '_fl_builder_data', wp_json_encode( $modules ) );

		return true;
	}

	/**
	 * Get documentation.
	 *
	 * @since 1.3.0
	 * @return array Documentation.
	 */
	public function get_documentation() {
		$docs = array(
			'name'        => 'Beaver Builder',
			'description' => 'Popular front-end page builder plugin',
			'overview'    => 'Beaver Builder uses rows, columns, and modules. Data is stored as JSON in postmeta.',
			'structure'   => array(
				'row'    => 'Layout container',
				'column' => 'Column container',
				'module' => 'Content element',
			),
			'resources'   => array(
				'https://docs.wpbeaverbuilder.com/',
			),
		);

		// Add patterns if intelligence is available.
		if ( function_exists( 'respira_get_beaver_patterns' ) ) {
			$docs['patterns'] = respira_get_beaver_patterns();
		}

		// Add module catalog if intelligence is available.
		if ( class_exists( 'Respira_Beaver_Module_Registry' ) ) {
			$docs['modules'] = Respira_Beaver_Module_Registry::get_all_modules();
		}

		return $docs;
	}

	/**
	 * Get available modules.
	 *
	 * @since 1.3.0
	 * @return array Available modules.
	 */
	public function get_available_modules() {
		// Use dynamic module registry if available.
		if ( class_exists( 'Respira_Beaver_Module_Registry' ) ) {
			return Respira_Beaver_Module_Registry::get_all_modules();
		}

		// Fallback to basic list.
		return array(
			array(
				'name'        => 'row',
				'title'       => 'Row',
				'description' => 'Layout container',
			),
			array(
				'name'        => 'column',
				'title'       => 'Column',
				'description' => 'Column container',
			),
			array(
				'name'        => 'text',
				'title'       => 'Text',
				'description' => 'Text content module',
			),
			array(
				'name'        => 'html',
				'title'       => 'HTML',
				'description' => 'Custom HTML/CSS/JS',
			),
		);
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since 1.4.0
	 * @param array $elements_used Optional. Array of element names used on the page.
	 * @return array Builder schema with element information.
	 */
	public function get_builder_schema( $elements_used = array() ) {
		if ( class_exists( 'Respira_Beaver_Module_Schema' ) ) {
			$schema_generator = new Respira_Beaver_Module_Schema();
			return $schema_generator->get_builder_schema( $elements_used );
		}

		// Fallback to basic schema.
		return array(
			'builder'         => 'beaver',
			'builder_version' => $this->get_version(),
			'modules'         => $this->get_available_modules(),
		);
	}

	/**
	 * Check if intelligence is available.
	 *
	 * @since 1.4.0
	 * @return bool True if intelligence is available.
	 */
	public function is_intelligence_available() {
		return class_exists( 'Respira_Beaver_Intelligence_Loader' ) &&
			   class_exists( 'Respira_Beaver_Module_Registry' ) &&
			   class_exists( 'Respira_Beaver_Module_Schema' ) &&
			   class_exists( 'Respira_Beaver_Validator' );
	}

	/**
	 * Simplify Beaver Builder structure.
	 *
	 * @since  1.3.0
	 * @param  array $modules Beaver Builder modules.
	 * @return array Simplified structure.
	 */
	protected function simplify_structure( $modules ) {
		return array_map(
			function( $module ) {
				return array(
					'type'       => $module['type'] ?? 'unknown',
					'attributes' => $module['settings'] ?? array(),
					'content'    => $module['content'] ?? '',
					'children'   => isset( $module['modules'] ) ? $this->simplify_structure( $module['modules'] ) : array(),
				);
			},
			$modules
		);
	}

	/**
	 * Convert simplified structure back to Beaver Builder format.
	 *
	 * @since  1.3.0
	 * @param  array $simplified Simplified structure.
	 * @return array Beaver Builder format.
	 */
	protected function complexify_structure( $simplified ) {
		return array_map(
			function( $item ) {
				$module = array(
					'type'     => $item['type'] ?? 'text',
					'settings' => $item['attributes'] ?? array(),
				);

				if ( ! empty( $item['children'] ) ) {
					$module['modules'] = $this->complexify_structure( $item['children'] );
				} elseif ( isset( $item['content'] ) ) {
					$module['content'] = $item['content'];
				}

				return $module;
			},
			$simplified
		);
	}
}

