<?php
/**
 * Oxygen Builder component schema generator.
 *
 * Generates schemas for all registered components with property definitions.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/oxygen-intelligence
 * @since      1.4.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-schema-base.php';

/**
 * Oxygen Builder component schema class.
 *
 * @since 1.4.0
 */
class Respira_Oxygen_Component_Schema extends Respira_Builder_Schema_Base {

	/**
	 * Constructor.
	 *
	 * @since 1.4.0
	 */
	public function __construct() {
		$version = defined( 'CT_VERSION' ) ? CT_VERSION : 'unknown';
		parent::__construct( 'oxygen', $version );
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since  1.4.0
	 * @param  array $components_used Optional. Array of component names used on the page.
	 * @return array Builder schema with component information.
	 */
	public function get_builder_schema( $components_used = array() ) {
		$schema = array(
			'builder'              => 'oxygen',
			'builder_version'      => $this->builder_version,
			'available_components' => array(),
			'quick_reference'      => $this->get_quick_reference(),
			'structure_notes'      => $this->get_structure_notes(),
		);

		// Get schemas for components used on the page (or all components if none specified).
		if ( empty( $components_used ) ) {
			$all_components   = Respira_Oxygen_Component_Registry::get_all_components();
			$components_used = array_column( $all_components, 'type' );
		}

		foreach ( $components_used as $component_name ) {
			$component = Respira_Oxygen_Component_Registry::get_component( $component_name );
			if ( ! $component ) {
				continue;
			}

			$component_schema = array(
				'title'       => $component['title'],
				'description' => $component['description'],
				'type'        => $component['type'],
				'category'    => isset( $component['category'] ) ? $component['category'] : 'general',
				'properties'  => array(),
			);

			// Generate property schemas.
			if ( ! empty( $component['properties'] ) ) {
				foreach ( $component['properties'] as $property ) {
					$component_schema['properties'][ $property ] = $this->get_property_schema( $property, $component['type'] );
				}
			}

			$schema['available_components'][ $component['type'] ] = $component_schema;
		}

		return $schema;
	}

	/**
	 * Get property schema for a component property.
	 *
	 * @since  1.4.0
	 * @param  string $property       Property name.
	 * @param  string $component_type Component type.
	 * @return array Property schema.
	 */
	private function get_property_schema( $property, $component_type ) {
		$property_map = array(
			// Content properties.
			'ct_content'  => array( 'type' => 'string', 'format' => 'html', 'example' => 'Text content' ),
			'content'     => array( 'type' => 'string', 'format' => 'html', 'example' => '<p>Content</p>' ),
			'heading'     => array( 'type' => 'string', 'format' => 'text', 'example' => 'Heading' ),
			'title'       => array( 'type' => 'string', 'format' => 'text', 'example' => 'Title' ),
			'description' => array( 'type' => 'string', 'format' => 'text', 'example' => 'Description' ),
			'code'        => array( 'type' => 'string', 'format' => 'code', 'example' => '<div>Code</div>' ),
			'shortcode'   => array( 'type' => 'string', 'format' => 'text', 'example' => '[shortcode]' ),

			// Tag properties.
			'tag'         => array( 'type' => 'string', 'format' => 'enum', 'example' => 'h2', 'options' => array( 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'p', 'div', 'span' ) ),

			// Link/URL properties.
			'url'         => array( 'type' => 'string', 'format' => 'url', 'example' => 'https://example.com' ),
			'target'      => array( 'type' => 'string', 'format' => 'enum', 'example' => '_blank', 'options' => array( '_self', '_blank' ) ),
			'button_url'  => array( 'type' => 'string', 'format' => 'url', 'example' => 'https://example.com' ),

			// Image properties.
			'src'         => array( 'type' => 'string', 'format' => 'url', 'example' => 'https://example.com/image.jpg' ),
			'alt'         => array( 'type' => 'string', 'format' => 'text', 'example' => 'Image alt text' ),
			'images'      => array( 'type' => 'array', 'format' => 'image_ids', 'example' => array() ),
			'size'        => array( 'type' => 'string', 'format' => 'image_size', 'example' => 'large' ),
			'object-fit'  => array( 'type' => 'string', 'format' => 'enum', 'example' => 'cover', 'options' => array( 'cover', 'contain', 'fill', 'none', 'scale-down' ) ),
			'photo'       => array( 'type' => 'string', 'format' => 'url', 'example' => 'https://example.com/photo.jpg' ),

			// Video/Audio properties.
			'embed_code'  => array( 'type' => 'string', 'format' => 'html', 'example' => '<iframe src="..."></iframe>' ),
			'autoplay'    => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => false ),
			'controls'    => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => true ),
			'loop'        => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => false ),

			// Icon properties.
			'icon-id'     => array( 'type' => 'string', 'format' => 'icon_id', 'example' => 'FontAwesomeicon-heart' ),
			'icon-size'   => array( 'type' => 'string', 'format' => 'size', 'example' => '32px' ),
			'icon-color'  => array( 'type' => 'string', 'format' => 'color', 'example' => '#000000' ),
			'icon'        => array( 'type' => 'string', 'format' => 'icon_id', 'example' => 'fa-heart' ),

			// Dimensions.
			'width'       => array( 'type' => 'string', 'format' => 'size', 'example' => '100%' ),
			'height'      => array( 'type' => 'string', 'format' => 'size', 'example' => 'auto' ),
			'padding'     => array( 'type' => 'string', 'format' => 'spacing', 'example' => '20px' ),
			'margin'      => array( 'type' => 'string', 'format' => 'spacing', 'example' => '10px' ),
			'spacing'     => array( 'type' => 'string', 'format' => 'size', 'example' => '15px' ),

			// Colors.
			'color'       => array( 'type' => 'string', 'format' => 'color', 'example' => '#000000' ),
			'background-color' => array( 'type' => 'string', 'format' => 'color', 'example' => '#ffffff' ),

			// Typography.
			'font-size'   => array( 'type' => 'string', 'format' => 'size', 'example' => '16px' ),
			'font-family' => array( 'type' => 'string', 'format' => 'font_family', 'example' => 'Arial' ),
			'line-height' => array( 'type' => 'string', 'format' => 'size', 'example' => '1.5' ),

			// Layout.
			'column-count' => array( 'type' => 'number', 'format' => 'number', 'example' => 2 ),
			'column-gap'  => array( 'type' => 'string', 'format' => 'size', 'example' => '20px' ),
			'column-width' => array( 'type' => 'string', 'format' => 'size', 'example' => 'auto' ),
			'columns'     => array( 'type' => 'number', 'format' => 'number', 'example' => 3 ),

			// Slider properties.
			'slides'      => array( 'type' => 'array', 'format' => 'slides_array', 'example' => array() ),
			'speed'       => array( 'type' => 'number', 'format' => 'milliseconds', 'example' => 1000 ),
			'arrows'      => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => true ),
			'dots'        => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => true ),

			// Tabs/Accordion properties.
			'tabs'        => array( 'type' => 'array', 'format' => 'tabs_array', 'example' => array() ),
			'items'       => array( 'type' => 'array', 'format' => 'items_array', 'example' => array() ),
			'active_tab'  => array( 'type' => 'number', 'format' => 'number', 'example' => 0 ),
			'active_item' => array( 'type' => 'number', 'format' => 'number', 'example' => 0 ),
			'orientation' => array( 'type' => 'string', 'format' => 'enum', 'example' => 'horizontal', 'options' => array( 'horizontal', 'vertical' ) ),
			'multiple_open' => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => false ),
			'open'        => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => false ),

			// Modal properties.
			'trigger'     => array( 'type' => 'string', 'format' => 'enum', 'example' => 'click' ),
			'overlay'     => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => true ),

			// Query/Repeater properties.
			'query_type'  => array( 'type' => 'string', 'format' => 'enum', 'example' => 'custom' ),
			'post_type'   => array( 'type' => 'string', 'format' => 'post_type', 'example' => 'post' ),
			'posts_per_page' => array( 'type' => 'number', 'format' => 'number', 'example' => 10 ),
			'order_by'    => array( 'type' => 'string', 'format' => 'enum', 'example' => 'date' ),
			'order'       => array( 'type' => 'string', 'format' => 'enum', 'example' => 'DESC', 'options' => array( 'ASC', 'DESC' ) ),

			// Menu properties.
			'menu'        => array( 'type' => 'string', 'format' => 'menu_id', 'example' => 'primary' ),
			'mobile_breakpoint' => array( 'type' => 'number', 'format' => 'pixels', 'example' => 768 ),

			// Form properties.
			'fields'      => array( 'type' => 'array', 'format' => 'form_fields', 'example' => array() ),
			'submit_button_text' => array( 'type' => 'string', 'format' => 'text', 'example' => 'Submit' ),
			'email_to'    => array( 'type' => 'string', 'format' => 'email', 'example' => 'admin@example.com' ),
			'type'        => array( 'type' => 'string', 'format' => 'enum', 'example' => 'text' ),
			'name'        => array( 'type' => 'string', 'format' => 'text', 'example' => 'field_name' ),
			'placeholder' => array( 'type' => 'string', 'format' => 'text', 'example' => 'Enter text' ),
			'required'    => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => false ),
			'rows'        => array( 'type' => 'number', 'format' => 'number', 'example' => 5 ),

			// Advanced properties.
			'percent'     => array( 'type' => 'number', 'format' => 'number', 'example' => 75 ),
			'start'       => array( 'type' => 'number', 'format' => 'number', 'example' => 0 ),
			'end'         => array( 'type' => 'number', 'format' => 'number', 'example' => 100 ),
			'duration'    => array( 'type' => 'number', 'format' => 'milliseconds', 'example' => 2000 ),
			'prefix'      => array( 'type' => 'string', 'format' => 'text', 'example' => '$' ),
			'suffix'      => array( 'type' => 'string', 'format' => 'text', 'example' => '+' ),
			'price'       => array( 'type' => 'string', 'format' => 'text', 'example' => '$99' ),
			'features'    => array( 'type' => 'array', 'format' => 'features_array', 'example' => array() ),
			'button_text' => array( 'type' => 'string', 'format' => 'text', 'example' => 'Buy Now' ),
			'author'      => array( 'type' => 'string', 'format' => 'text', 'example' => 'John Doe' ),
			'rating'      => array( 'type' => 'number', 'format' => 'number', 'example' => 5 ),
			'address'     => array( 'type' => 'string', 'format' => 'text', 'example' => '123 Main St, City' ),
			'zoom'        => array( 'type' => 'number', 'format' => 'number', 'example' => 12 ),
			'marker'      => array( 'type' => 'boolean', 'format' => 'boolean', 'example' => true ),
			'language'    => array( 'type' => 'string', 'format' => 'text', 'example' => 'javascript' ),
		);

		// Return the schema for this property, or a generic schema if not found.
		if ( isset( $property_map[ $property ] ) ) {
			return $property_map[ $property ];
		}

		// Generic property schema.
		return array(
			'type'    => 'string',
			'format'  => 'text',
			'example' => 'value',
		);
	}

	/**
	 * Get structure notes for Oxygen Builder.
	 *
	 * @since  1.4.0
	 * @return array Structure notes.
	 */
	private function get_structure_notes() {
		return array(
			'hierarchy'    => 'Section → Div/Columns → Components',
			'data_format'  => 'Oxygen stores data as JSON tree in ct_builder_shortcodes postmeta',
			'meta_key'     => 'ct_builder_shortcodes',
			'components'   => 'Each component has id, name (type), options (settings), and children',
			'tree_structure' => 'Data is stored as tree with parent-child relationships',
			'note'         => 'Component IDs must be unique within the page',
		);
	}
}
