<?php
/**
 * Oxygen Builder component registry.
 *
 * Dynamic component detection from Oxygen Builder's registry.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/oxygen-intelligence
 * @since      1.4.0
 */

/**
 * Oxygen Builder component registry class.
 *
 * @since 1.4.0
 */
class Respira_Oxygen_Component_Registry {

	/**
	 * Get all registered components with metadata.
	 *
	 * @since  1.4.0
	 * @return array Array of component information.
	 */
	public static function get_all_components() {
		// Check cache first.
		$cached = get_transient( 'respira_oxygen_components' );
		if ( false !== $cached ) {
			return $cached;
		}

		$component_list = array();

		// Try to detect components from Oxygen Builder's registry.
		if ( function_exists( 'oxygen_vsb_get_components' ) ) {
			$component_list = self::detect_from_api();
		}

		// Fallback: use known component types from Oxygen Builder documentation.
		if ( empty( $component_list ) ) {
			$component_list = self::get_known_components();
		}

		// Cache for 24 hours.
		set_transient( 'respira_oxygen_components', $component_list, DAY_IN_SECONDS );

		return $component_list;
	}

	/**
	 * Detect components from Oxygen Builder API (if available).
	 *
	 * @since  1.4.0
	 * @return array Array of detected components.
	 */
	private static function detect_from_api() {
		$components = array();

		try {
			if ( function_exists( 'oxygen_vsb_get_components' ) ) {
				$oxygen_components = oxygen_vsb_get_components();

				foreach ( $oxygen_components as $component_name => $component_data ) {
					$components[] = array(
						'name'        => $component_name,
						'title'       => isset( $component_data['name'] ) ? $component_data['name'] : $component_name,
						'type'        => $component_name,
						'description' => '',
						'properties'  => array(),
					);
				}
			}
		} catch ( Exception $e ) {
			// Silently fall back to known components.
		}

		return $components;
	}

	/**
	 * Get known Oxygen Builder components from documentation.
	 *
	 * @since  1.4.0
	 * @return array Array of known components.
	 */
	private static function get_known_components() {
		return array(
			// Structure components.
			array(
				'name'        => 'Section',
				'title'       => 'Section',
				'type'        => 'ct_section',
				'description' => 'Section container',
				'category'    => 'structure',
				'properties'  => array( 'width', 'height', 'background-color', 'padding' ),
			),
			array(
				'name'        => 'Div',
				'title'       => 'Div',
				'type'        => 'ct_div_block',
				'description' => 'Div block container',
				'category'    => 'structure',
				'properties'  => array( 'width', 'height', 'background-color', 'padding', 'margin' ),
			),
			array(
				'name'        => 'Columns',
				'title'       => 'Columns',
				'type'        => 'ct_new_columns',
				'description' => 'Column layout container',
				'category'    => 'structure',
				'properties'  => array( 'column-count', 'column-gap', 'column-width' ),
			),
			array(
				'name'        => 'Column',
				'title'       => 'Column',
				'type'        => 'ct_column',
				'description' => 'Individual column',
				'category'    => 'structure',
				'properties'  => array( 'width', 'padding', 'background-color' ),
			),

			// Basic components.
			array(
				'name'        => 'Heading',
				'title'       => 'Heading',
				'type'        => 'ct_headline',
				'description' => 'Heading element (H1-H6)',
				'category'    => 'basic',
				'properties'  => array( 'ct_content', 'tag', 'font-size', 'color', 'font-family' ),
			),
			array(
				'name'        => 'Text',
				'title'       => 'Text',
				'type'        => 'ct_text_block',
				'description' => 'Rich text block',
				'category'    => 'basic',
				'properties'  => array( 'ct_content', 'font-size', 'color', 'line-height' ),
			),
			array(
				'name'        => 'Link',
				'title'       => 'Link',
				'type'        => 'ct_link',
				'description' => 'Link element',
				'category'    => 'basic',
				'properties'  => array( 'ct_content', 'url', 'target', 'color' ),
			),
			array(
				'name'        => 'Button',
				'title'       => 'Button',
				'type'        => 'ct_link_button',
				'description' => 'Button element',
				'category'    => 'basic',
				'properties'  => array( 'ct_content', 'url', 'target', 'background-color', 'color', 'padding' ),
			),
			array(
				'name'        => 'Icon',
				'title'       => 'Icon',
				'type'        => 'ct_fancy_icon',
				'description' => 'Icon element',
				'category'    => 'basic',
				'properties'  => array( 'icon-id', 'icon-size', 'icon-color', 'url' ),
			),

			// Media components.
			array(
				'name'        => 'Image',
				'title'       => 'Image',
				'type'        => 'ct_image',
				'description' => 'Image element',
				'category'    => 'media',
				'properties'  => array( 'src', 'alt', 'width', 'height', 'object-fit' ),
			),
			array(
				'name'        => 'Video',
				'title'       => 'Video',
				'type'        => 'oxy_video',
				'description' => 'Video element',
				'category'    => 'media',
				'properties'  => array( 'url', 'embed_code', 'autoplay', 'controls' ),
			),
			array(
				'name'        => 'Audio',
				'title'       => 'Audio',
				'type'        => 'oxy_audio',
				'description' => 'Audio player',
				'category'    => 'media',
				'properties'  => array( 'url', 'autoplay', 'loop' ),
			),
			array(
				'name'        => 'Gallery',
				'title'       => 'Gallery',
				'type'        => 'oxy_gallery',
				'description' => 'Image gallery',
				'category'    => 'media',
				'properties'  => array( 'images', 'columns', 'spacing' ),
			),

			// Interactive components.
			array(
				'name'        => 'Slider',
				'title'       => 'Slider',
				'type'        => 'oxy_easy_posts_slider',
				'description' => 'Content slider',
				'category'    => 'interactive',
				'properties'  => array( 'slides', 'autoplay', 'speed', 'arrows', 'dots' ),
			),
			array(
				'name'        => 'Tabs',
				'title'       => 'Tabs',
				'type'        => 'oxy_tabs',
				'description' => 'Tabbed content',
				'category'    => 'interactive',
				'properties'  => array( 'tabs', 'active_tab', 'orientation' ),
			),
			array(
				'name'        => 'Accordion',
				'title'       => 'Accordion',
				'type'        => 'oxy_accordion',
				'description' => 'Accordion content',
				'category'    => 'interactive',
				'properties'  => array( 'items', 'active_item', 'multiple_open' ),
			),
			array(
				'name'        => 'Toggle',
				'title'       => 'Toggle',
				'type'        => 'oxy_toggle',
				'description' => 'Toggle content',
				'category'    => 'interactive',
				'properties'  => array( 'heading', 'content', 'open' ),
			),
			array(
				'name'        => 'Modal',
				'title'       => 'Modal',
				'type'        => 'oxy_modal',
				'description' => 'Modal popup',
				'category'    => 'interactive',
				'properties'  => array( 'trigger', 'content', 'width', 'overlay' ),
			),

			// WordPress components.
			array(
				'name'        => 'Repeater',
				'title'       => 'Repeater',
				'type'        => 'oxy_repeater',
				'description' => 'Query loop / repeater',
				'category'    => 'wordpress',
				'properties'  => array( 'query_type', 'post_type', 'posts_per_page', 'order_by', 'order' ),
			),
			array(
				'name'        => 'Post Title',
				'title'       => 'Post Title',
				'type'        => 'oxy_post_title',
				'description' => 'Dynamic post title',
				'category'    => 'wordpress',
				'properties'  => array( 'tag', 'font-size', 'color' ),
			),
			array(
				'name'        => 'Post Content',
				'title'       => 'Post Content',
				'type'        => 'oxy_post_content',
				'description' => 'Dynamic post content',
				'category'    => 'wordpress',
				'properties'  => array( 'font-size', 'color', 'line-height' ),
			),
			array(
				'name'        => 'Featured Image',
				'title'       => 'Featured Image',
				'type'        => 'oxy_featured_image',
				'description' => 'Dynamic featured image',
				'category'    => 'wordpress',
				'properties'  => array( 'size', 'width', 'height' ),
			),
			array(
				'name'        => 'Menu',
				'title'       => 'Menu',
				'type'        => 'oxy_pro_menu',
				'description' => 'WordPress menu',
				'category'    => 'wordpress',
				'properties'  => array( 'menu', 'orientation', 'mobile_breakpoint' ),
			),

			// Form components.
			array(
				'name'        => 'Form',
				'title'       => 'Form',
				'type'        => 'oxy_contact_form',
				'description' => 'Contact form',
				'category'    => 'forms',
				'properties'  => array( 'fields', 'submit_button_text', 'email_to' ),
			),
			array(
				'name'        => 'Input',
				'title'       => 'Input',
				'type'        => 'ct_input',
				'description' => 'Form input field',
				'category'    => 'forms',
				'properties'  => array( 'type', 'name', 'placeholder', 'required' ),
			),
			array(
				'name'        => 'Textarea',
				'title'       => 'Textarea',
				'type'        => 'ct_textarea',
				'description' => 'Form textarea',
				'category'    => 'forms',
				'properties'  => array( 'name', 'placeholder', 'rows', 'required' ),
			),

			// Advanced components.
			array(
				'name'        => 'Progress Bar',
				'title'       => 'Progress Bar',
				'type'        => 'oxy_progress_bar',
				'description' => 'Progress bar',
				'category'    => 'advanced',
				'properties'  => array( 'percent', 'height', 'color', 'background-color' ),
			),
			array(
				'name'        => 'Counter',
				'title'       => 'Counter',
				'type'        => 'oxy_counter',
				'description' => 'Animated counter',
				'category'    => 'advanced',
				'properties'  => array( 'start', 'end', 'duration', 'prefix', 'suffix' ),
			),
			array(
				'name'        => 'Pricing Box',
				'title'       => 'Pricing Box',
				'type'        => 'oxy_pricing_box',
				'description' => 'Pricing table',
				'category'    => 'advanced',
				'properties'  => array( 'title', 'price', 'features', 'button_text', 'button_url' ),
			),
			array(
				'name'        => 'Testimonial',
				'title'       => 'Testimonial',
				'type'        => 'oxy_testimonial',
				'description' => 'Testimonial component',
				'category'    => 'advanced',
				'properties'  => array( 'content', 'author', 'photo', 'rating' ),
			),
			array(
				'name'        => 'Icon Box',
				'title'       => 'Icon Box',
				'type'        => 'oxy_icon_box',
				'description' => 'Icon with title and description',
				'category'    => 'advanced',
				'properties'  => array( 'icon', 'title', 'description', 'url' ),
			),
			array(
				'name'        => 'Google Map',
				'title'       => 'Google Map',
				'type'        => 'oxy_map',
				'description' => 'Google Maps embed',
				'category'    => 'advanced',
				'properties'  => array( 'address', 'zoom', 'height', 'marker' ),
			),
			array(
				'name'        => 'Code Block',
				'title'       => 'Code Block',
				'type'        => 'ct_code_block',
				'description' => 'Custom code (HTML/CSS/JS)',
				'category'    => 'advanced',
				'properties'  => array( 'code', 'language' ),
			),
			array(
				'name'        => 'Shortcode',
				'title'       => 'Shortcode',
				'type'        => 'ct_shortcode',
				'description' => 'WordPress shortcode',
				'category'    => 'advanced',
				'properties'  => array( 'shortcode' ),
			),
		);
	}

	/**
	 * Get component by name or type.
	 *
	 * @since  1.4.0
	 * @param  string $component_name Component name or type.
	 * @return array|null Component information or null if not found.
	 */
	public static function get_component( $component_name ) {
		$components = self::get_all_components();

		foreach ( $components as $component ) {
			if ( $component['name'] === $component_name || $component['type'] === $component_name ) {
				return $component;
			}
		}

		return null;
	}

	/**
	 * Clear component cache.
	 *
	 * @since 1.4.0
	 */
	public static function clear_cache() {
		delete_transient( 'respira_oxygen_components' );
	}
}
