<?php
/**
 * Elementor widget schema generator.
 *
 * Generates schemas for all registered widgets with control definitions.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/elementor-intelligence
 * @since      1.3.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-schema-base.php';

/**
 * Elementor widget schema class.
 *
 * @since 1.3.0
 */
class Respira_Elementor_Widget_Schema extends Respira_Builder_Schema_Base {

	/**
	 * Constructor.
	 *
	 * @since 1.3.0
	 */
	public function __construct() {
		$version = defined( 'ELEMENTOR_VERSION' ) ? ELEMENTOR_VERSION : 'unknown';
		parent::__construct( 'elementor', $version );
	}

	/**
	 * Get builder schema for AI context.
	 *
	 * @since  1.3.0
	 * @param  array $widgets_used Optional. Array of widget names used on the page.
	 * @return array Builder schema with widget information.
	 */
	public function get_builder_schema( $widgets_used = array() ) {
		$schema = array(
			'builder'          => 'elementor',
			'builder_version'  => $this->builder_version,
			'available_widgets' => array(),
			'quick_reference'  => $this->get_quick_reference(),
		);

		// Get schemas for widgets used on the page (or all widgets if none specified).
		if ( empty( $widgets_used ) ) {
			$all_widgets = Respira_Elementor_Widget_Registry::get_all_widgets();
			$widgets_used = array_column( $all_widgets, 'name' );
		}

		foreach ( $widgets_used as $widget_name ) {
			$widget = Respira_Elementor_Widget_Registry::get_widget( $widget_name );
			if ( ! $widget ) {
				continue;
			}

			$widget_schema = array(
				'title'      => $widget['title'],
				'categories' => $widget['categories'],
				'attributes' => array(),
			);

			// Extract control schemas.
			if ( ! empty( $widget['controls'] ) ) {
				foreach ( $widget['controls'] as $control_name => $control_def ) {
					$type = isset( $control_def['type'] ) ? $this->normalize_type( $control_def['type'] ) : 'string';
					$format = $this->get_format_hint( $type, $control_name );

					$widget_schema['attributes'][ $control_name ] = array(
						'type'    => $type,
						'format'  => $format,
						'default' => isset( $control_def['default'] ) ? $control_def['default'] : null,
						'example' => $this->get_example_value( $type, $control_name ),
					);

					// Add label and description.
					if ( isset( $control_def['label'] ) ) {
						$widget_schema['attributes'][ $control_name ]['label'] = $control_def['label'];
					}
					if ( isset( $control_def['description'] ) ) {
						$widget_schema['attributes'][ $control_name ]['description'] = $control_def['description'];
					}

					// Add options for select controls.
					if ( isset( $control_def['options'] ) ) {
						$widget_schema['attributes'][ $control_name ]['options'] = $control_def['options'];
					}
				}
			}

			$schema['available_widgets'][ $widget_name ] = $widget_schema;
		}

		return $schema;
	}
}

