<?php
/**
 * Elementor validator.
 *
 * Validates Elementor structure and widget settings before injection.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/page-builders/elementor-intelligence
 * @since      1.3.0
 */

require_once RESPIRA_PLUGIN_DIR . 'includes/page-builders/intelligence/class-builder-validator-base.php';

/**
 * Elementor validator class.
 *
 * @since 1.3.0
 */
class Respira_Elementor_Validator extends Respira_Builder_Validator_Base {

	/**
	 * Validate a layout structure.
	 *
	 * @since  1.3.0
	 * @param  array $content Content structure to validate.
	 * @return array Validation result with 'valid' boolean and 'errors' array.
	 */
	public function validate_layout( $content ) {
		$this->clear_errors();

		if ( ! is_array( $content ) ) {
			$this->add_error( __( 'Content must be an array.', 'respira-for-wordpress' ) );
			return array(
				'valid'  => false,
				'errors' => $this->get_errors(),
			);
		}

		// Define nesting rules: sections > columns > widgets.
		$nesting_rules = array(
			'section' => array( 'column' ),
			'column'  => array( 'widget' ),
			'widget'  => array(), // Widgets don't have children.
		);

		// Validate structure nesting.
		$this->validate_nesting( $content, $nesting_rules );

		// Validate each element.
		foreach ( $content as $element ) {
			$this->validate_element( $element );
		}

		return array(
			'valid'  => empty( $this->errors ),
			'errors' => $this->get_errors(),
		);
	}

	/**
	 * Validate a single element.
	 *
	 * @since  1.3.0
	 * @param  array $element Element structure.
	 */
	private function validate_element( $element ) {
		// Check required fields.
		if ( ! isset( $element['type'] ) ) {
			$this->add_error( __( 'Element must have a type.', 'respira-for-wordpress' ) );
			return;
		}

		$element_type = $element['type'];

		// Validate element ID if present.
		if ( isset( $element['id'] ) && empty( $element['id'] ) ) {
			$this->add_error( __( 'Element ID cannot be empty.', 'respira-for-wordpress' ) );
		}

		// Validate widget type if element is a widget.
		if ( 'widget' === $element_type ) {
			if ( ! isset( $element['widget'] ) ) {
				$this->add_error( __( 'Widget element must have a widget type.', 'respira-for-wordpress' ) );
			} else {
				$widget_name = $element['widget'];
				$widget = Respira_Elementor_Widget_Registry::get_widget( $widget_name );
				if ( ! $widget ) {
					// Unknown widget - allow it (may be custom).
				}
			}
		}

		// Validate settings if present.
		if ( isset( $element['settings'] ) && is_array( $element['settings'] ) ) {
			$this->validate_widget_settings( $element, $element['settings'] );
		}

		// Validate nested elements recursively.
		if ( ! empty( $element['elements'] ) && is_array( $element['elements'] ) ) {
			foreach ( $element['elements'] as $nested_element ) {
				$this->validate_element( $nested_element );
			}
		}
	}

	/**
	 * Validate widget settings.
	 *
	 * @since  1.3.0
	 * @param  array $element Element structure.
	 * @param  array $settings Widget settings.
	 */
	private function validate_widget_settings( $element, $settings ) {
		if ( 'widget' !== $element['type'] || ! isset( $element['widget'] ) ) {
			return;
		}

		$widget_name = $element['widget'];
		$widget = Respira_Elementor_Widget_Registry::get_widget( $widget_name );
		if ( ! $widget || empty( $widget['controls'] ) ) {
			return; // Unknown widget or no controls defined.
		}

		// Validate each setting against widget controls.
		foreach ( $settings as $setting_name => $setting_value ) {
			if ( ! isset( $widget['controls'][ $setting_name ] ) ) {
				// Unknown setting - allow it (may be custom).
				continue;
			}

			$control_def = $widget['controls'][ $setting_name ];
			$expected_type = isset( $control_def['type'] ) ? $control_def['type'] : 'string';

			if ( ! $this->validate_attribute_format( $setting_value, $expected_type, $setting_name ) ) {
				$this->add_error(
					sprintf(
						/* translators: 1: setting name, 2: widget name */
						__( 'Invalid setting format for %1$s in widget %2$s', 'respira-for-wordpress' ),
						$setting_name,
						$widget_name
					)
				);
			}
		}
	}
}

