<?php
/**
 * Divi Intelligence - Module Schema System.
 *
 * Provides complete attribute schemas for all Divi modules.
 * Part of Divi Intelligence addon for Respira for WordPress.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/divi-intelligence
 */

/**
 * Divi module schema class.
 *
 * @since 1.0.0
 */
class Respira_Divi_Module_Schema {

	/**
	 * Cache key for schemas.
	 *
	 * @since 1.0.0
	 * @var string
	 */
	const CACHE_KEY = 'respira_divi_schemas';

	/**
	 * Cache expiration (24 hours).
	 *
	 * @since 1.0.0
	 * @var int
	 */
	const CACHE_EXPIRATION = 86400;

	/**
	 * Get schema for a specific module.
	 *
	 * @since 1.0.0
	 * @param string $module_name Module name (e.g., 'et_pb_text').
	 * @return array|false Module schema or false if not found.
	 */
	public static function get_module_schema( $module_name ) {
		$all_schemas = self::get_all_schemas();
		return isset( $all_schemas[ $module_name ] ) ? $all_schemas[ $module_name ] : false;
	}

	/**
	 * Get schemas for multiple modules.
	 *
	 * @since 1.0.0
	 * @param array $module_names Array of module names.
	 * @return array Array of module schemas.
	 */
	public static function get_module_schemas( $module_names ) {
		$all_schemas = self::get_all_schemas();
		$schemas     = array();

		foreach ( $module_names as $module_name ) {
			if ( isset( $all_schemas[ $module_name ] ) ) {
				$schemas[ $module_name ] = $all_schemas[ $module_name ];
			}
		}

		return $schemas;
	}

	/**
	 * Get all module schemas.
	 *
	 * @since 1.0.0
	 * @return array Array of all module schemas.
	 */
	public static function get_all_schemas() {
		// Try cache first.
		$cached = get_transient( self::CACHE_KEY );
		if ( false !== $cached ) {
			return $cached;
		}

		$schemas = self::build_schemas();

		// Cache for 24 hours.
		set_transient( self::CACHE_KEY, $schemas, self::CACHE_EXPIRATION );

		return $schemas;
	}

	/**
	 * Build all module schemas.
	 *
	 * @since 1.0.0
	 * @return array Array of module schemas.
	 */
	private static function build_schemas() {
		// Load schema definitions from data file.
		$schema_file = RESPIRA_PLUGIN_DIR . 'includes/divi-intelligence/divi-modules.php';
		if ( file_exists( $schema_file ) ) {
			require_once $schema_file;
			if ( function_exists( 'respira_get_divi_module_schemas' ) ) {
				return respira_get_divi_module_schemas();
			}
		}

		// Fallback to basic schemas.
		return self::get_basic_schemas();
	}

	/**
	 * Get basic schemas for Tier 1 modules.
	 *
	 * @since 1.0.0
	 * @return array Array of basic module schemas.
	 */
	private static function get_basic_schemas() {
		return array(
			'et_pb_section'  => self::get_section_schema(),
			'et_pb_row'      => self::get_row_schema(),
			'et_pb_text'     => self::get_text_schema(),
			'et_pb_image'    => self::get_image_schema(),
			'et_pb_button'   => self::get_button_schema(),
			'et_pb_cta'      => self::get_cta_schema(),
			'et_pb_blurb'    => self::get_blurb_schema(),
			'et_pb_video'    => self::get_video_schema(),
			'et_pb_html_content' => self::get_html_content_schema(),
			'et_pb_code'     => self::get_code_schema(),
		);
	}

	/**
	 * Get common attributes shared by all modules.
	 *
	 * @since 1.0.0
	 * @return array Common attributes.
	 */
	public static function get_common_attributes() {
		return array(
			'admin_label'     => array(
				'type'        => 'string',
				'format'      => 'text',
				'description' => 'Label shown in builder for identification',
				'required'    => false,
				'example'     => 'Hero Section',
			),
			'module_id'       => array(
				'type'        => 'string',
				'format'      => 'text',
				'description' => 'Unique CSS ID for the module',
				'required'    => false,
				'example'     => 'hero-section',
			),
			'module_class'    => array(
				'type'        => 'string',
				'format'      => 'text',
				'description' => 'Custom CSS class for the module',
				'required'    => false,
				'example'     => 'custom-class',
			),
			'disabled_on'     => array(
				'type'        => 'string',
				'format'      => 'comma-separated',
				'description' => 'Hide module on specific devices (on, off, tablet, phone)',
				'required'    => false,
				'example'     => 'off|off|off',
			),
			'custom_css'      => array(
				'type'        => 'string',
				'format'      => 'css',
				'description' => 'Custom CSS for this module',
				'required'    => false,
				'example'     => '.module-class { color: red; }',
			),
		);
	}

	/**
	 * Get section module schema.
	 *
	 * @since 1.0.0
	 * @return array Section schema.
	 */
	private static function get_section_schema() {
		$common = self::get_common_attributes();
		return array(
			'name'        => 'et_pb_section',
			'title'       => 'Section',
			'description' => 'Top-level container for rows and modules',
			'category'    => 'layout',
			'attributes'  => array_merge(
				$common,
				array(
					'background_color'     => array(
						'type'        => 'color',
						'format'      => 'hex',
						'description' => 'Background color',
						'required'    => false,
						'example'     => '#FFFFFF',
					),
					'background_image'    => array(
						'type'        => 'string',
						'format'      => 'url',
						'description' => 'Background image URL',
						'required'    => false,
						'example'     => 'https://example.com/image.jpg',
					),
					'fullwidth'           => array(
						'type'        => 'string',
						'format'      => 'on|off',
						'description' => 'Make section fullwidth',
						'required'    => false,
						'example'     => 'off',
					),
					'specialty'           => array(
						'type'        => 'string',
						'format'      => 'on|off',
						'description' => 'Enable specialty layout',
						'required'    => false,
						'example'     => 'off',
					),
					'padding'             => array(
						'type'        => 'string',
						'format'      => 'px|em|%',
						'description' => 'Padding (top|right|bottom|left)',
						'required'    => false,
						'example'     => '50px|50px|50px|50px',
					),
				)
			),
		);
	}

	/**
	 * Get row module schema.
	 *
	 * @since 1.0.0
	 * @return array Row schema.
	 */
	private static function get_row_schema() {
		$common = self::get_common_attributes();
		return array(
			'name'        => 'et_pb_row',
			'title'       => 'Row',
			'description' => 'Column layout container',
			'category'    => 'layout',
			'attributes'  => array_merge(
				$common,
				array(
					'column_structure'    => array(
						'type'        => 'string',
						'format'      => 'text',
						'description' => 'Column structure (e.g., 4_4, 3_3_3)',
						'required'    => false,
						'example'     => '4_4',
					),
					'background_color'    => array(
						'type'        => 'color',
						'format'      => 'hex',
						'description' => 'Background color',
						'required'    => false,
						'example'     => '#FFFFFF',
					),
					'padding'              => array(
						'type'        => 'string',
						'format'      => 'px|em|%',
						'description' => 'Padding (top|right|bottom|left)',
						'required'    => false,
						'example'     => '20px|20px|20px|20px',
					),
				)
			),
		);
	}

	/**
	 * Get text module schema.
	 *
	 * @since 1.0.0
	 * @return array Text schema.
	 */
	private static function get_text_schema() {
		$common = self::get_common_attributes();
		return array(
			'name'        => 'et_pb_text',
			'title'       => 'Text',
			'description' => 'Text content module for paragraphs and headings',
			'category'    => 'content',
			'attributes'  => array_merge(
				$common,
				array(
					'content'              => array(
						'type'        => 'html',
						'format'      => 'html',
						'description' => 'Text content (HTML allowed)',
						'required'    => true,
						'example'     => '<h1>Heading</h1><p>Paragraph text</p>',
					),
					'text_color'          => array(
						'type'        => 'color',
						'format'      => 'hex',
						'description' => 'Text color',
						'required'    => false,
						'example'     => '#000000',
					),
					'background_color'    => array(
						'type'        => 'color',
						'format'      => 'hex',
						'description' => 'Background color',
						'required'    => false,
						'example'     => '#FFFFFF',
					),
					'font_size'           => array(
						'type'        => 'string',
						'format'      => 'px|em|rem',
						'description' => 'Font size',
						'required'    => false,
						'example'     => '16px',
					),
					'text_orientation'   => array(
						'type'        => 'string',
						'format'      => 'left|center|right|justify',
						'description' => 'Text alignment',
						'required'    => false,
						'example'     => 'left',
					),
				)
			),
		);
	}

	/**
	 * Get image module schema.
	 *
	 * @since 1.0.0
	 * @return array Image schema.
	 */
	private static function get_image_schema() {
		$common = self::get_common_attributes();
		return array(
			'name'        => 'et_pb_image',
			'title'       => 'Image',
			'description' => 'Image module',
			'category'    => 'media',
			'attributes'  => array_merge(
				$common,
				array(
					'src'                  => array(
						'type'        => 'string',
						'format'      => 'url',
						'description' => 'Image URL',
						'required'    => true,
						'example'     => 'https://example.com/image.jpg',
					),
					'alt'                  => array(
						'type'        => 'string',
						'format'      => 'text',
						'description' => 'Image alt text',
						'required'    => false,
						'example'     => 'Description of image',
					),
					'title_text'          => array(
						'type'        => 'string',
						'format'      => 'text',
						'description' => 'Image title',
						'required'    => false,
						'example'     => 'Image Title',
					),
					'url'                 => array(
						'type'        => 'string',
						'format'      => 'url',
						'description' => 'Link URL (if image should be clickable)',
						'required'    => false,
						'example'     => 'https://example.com',
					),
					'align'               => array(
						'type'        => 'string',
						'format'      => 'left|center|right',
						'description' => 'Image alignment',
						'required'    => false,
						'example'     => 'center',
					),
				)
			),
		);
	}

	/**
	 * Get button module schema.
	 *
	 * @since 1.0.0
	 * @return array Button schema.
	 */
	private static function get_button_schema() {
		$common = self::get_common_attributes();
		return array(
			'name'        => 'et_pb_button',
			'title'       => 'Button',
			'description' => 'Button module',
			'category'    => 'content',
			'attributes'  => array_merge(
				$common,
				array(
					'button_text'         => array(
						'type'        => 'string',
						'format'      => 'text',
						'description' => 'Button text',
						'required'    => true,
						'example'     => 'Click Here',
					),
					'button_url'          => array(
						'type'        => 'string',
						'format'      => 'url',
						'description' => 'Button link URL',
						'required'    => false,
						'example'     => 'https://example.com',
					),
					'button_alignment'    => array(
						'type'        => 'string',
						'format'      => 'left|center|right',
						'description' => 'Button alignment',
						'required'    => false,
						'example'     => 'center',
					),
					'button_bg_color'    => array(
						'type'        => 'color',
						'format'      => 'hex',
						'description' => 'Button background color',
						'required'    => false,
						'example'     => '#0000FF',
					),
					'button_text_color'   => array(
						'type'        => 'color',
						'format'      => 'hex',
						'description' => 'Button text color',
						'required'    => false,
						'example'     => '#FFFFFF',
					),
				)
			),
		);
	}

	/**
	 * Get CTA module schema.
	 *
	 * @since 1.0.0
	 * @return array CTA schema.
	 */
	private static function get_cta_schema() {
		$common = self::get_common_attributes();
		return array(
			'name'        => 'et_pb_cta',
			'title'       => 'Call to Action',
			'description' => 'Call to action module with heading, text, and button',
			'category'    => 'content',
			'attributes'  => array_merge(
				$common,
				array(
					'title'                => array(
						'type'        => 'string',
						'format'      => 'text',
						'description' => 'CTA heading',
						'required'    => false,
						'example'     => 'Get Started Today',
					),
					'content'              => array(
						'type'        => 'html',
						'format'      => 'html',
						'description' => 'CTA content text',
						'required'    => false,
						'example'     => 'Join thousands of satisfied customers',
					),
					'button_text'         => array(
						'type'        => 'string',
						'format'      => 'text',
						'description' => 'Button text',
						'required'    => false,
						'example'     => 'Sign Up',
					),
					'button_url'          => array(
						'type'        => 'string',
						'format'      => 'url',
						'description' => 'Button link URL',
						'required'    => false,
						'example'     => 'https://example.com/signup',
					),
				)
			),
		);
	}

	/**
	 * Get blurb module schema.
	 *
	 * @since 1.0.0
	 * @return array Blurb schema.
	 */
	private static function get_blurb_schema() {
		$common = self::get_common_attributes();
		return array(
			'name'        => 'et_pb_blurb',
			'title'       => 'Blurb',
			'description' => 'Icon with text module',
			'category'    => 'content',
			'attributes'  => array_merge(
				$common,
				array(
					'title'                => array(
						'type'        => 'string',
						'format'      => 'text',
						'description' => 'Blurb title',
						'required'    => false,
						'example'     => 'Feature Title',
					),
					'content'              => array(
						'type'        => 'html',
						'format'      => 'html',
						'description' => 'Blurb content',
						'required'    => false,
						'example'     => 'Feature description',
					),
					'image'                => array(
						'type'        => 'string',
						'format'      => 'url',
						'description' => 'Icon/image URL',
						'required'    => false,
						'example'     => 'https://example.com/icon.png',
					),
					'url'                  => array(
						'type'        => 'string',
						'format'      => 'url',
						'description' => 'Link URL',
						'required'    => false,
						'example'     => 'https://example.com',
					),
				)
			),
		);
	}

	/**
	 * Get video module schema.
	 *
	 * @since 1.0.0
	 * @return array Video schema.
	 */
	private static function get_video_schema() {
		$common = self::get_common_attributes();
		return array(
			'name'        => 'et_pb_video',
			'title'       => 'Video',
			'description' => 'Video module',
			'category'    => 'media',
			'attributes'  => array_merge(
				$common,
				array(
					'src'                  => array(
						'type'        => 'string',
						'format'      => 'url',
						'description' => 'Video URL (YouTube, Vimeo, or direct video file)',
						'required'    => true,
						'example'     => 'https://www.youtube.com/watch?v=example',
					),
					'image_src'           => array(
						'type'        => 'string',
						'format'      => 'url',
						'description' => 'Video thumbnail image URL',
						'required'    => false,
						'example'     => 'https://example.com/thumbnail.jpg',
					),
				)
			),
		);
	}

	/**
	 * Get HTML content module schema.
	 *
	 * @since 1.0.0
	 * @return array HTML content schema.
	 */
	private static function get_html_content_schema() {
		$common = self::get_common_attributes();
		return array(
			'name'        => 'et_pb_html_content',
			'title'       => 'HTML Content',
			'description' => 'Custom HTML content module (supports scripts)',
			'category'    => 'content',
			'attributes'  => array_merge(
				$common,
				array(
					'content'              => array(
						'type'        => 'html',
						'format'      => 'html',
						'description' => 'HTML content (can include scripts)',
						'required'    => true,
						'example'     => '<div>Custom HTML</div><script>console.log("test");</script>',
					),
				)
			),
		);
	}

	/**
	 * Get code module schema.
	 *
	 * @since 1.0.0
	 * @return array Code schema.
	 */
	private static function get_code_schema() {
		$common = self::get_common_attributes();
		return array(
			'name'        => 'et_pb_code',
			'title'       => 'Code',
			'description' => 'Custom HTML/CSS/JS module',
			'category'    => 'content',
			'attributes'  => array_merge(
				$common,
				array(
					'content'              => array(
						'type'        => 'string',
						'format'      => 'base64',
						'description' => 'Code content (base64 encoded)',
						'required'    => true,
						'example'     => 'base64 encoded content',
					),
				)
			),
		);
	}

	/**
	 * Clear schema cache.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public static function clear_cache() {
		delete_transient( self::CACHE_KEY );
	}
}

