<?php
/**
 * Divi Intelligence - Global Modules Handler.
 *
 * Handles detection and management of Divi global modules.
 * Part of Divi Intelligence addon for Respira for WordPress.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes/divi-intelligence
 */

/**
 * Divi global modules class.
 *
 * @since 1.0.0
 */
class Respira_Divi_Global_Modules {

	/**
	 * Detect if a module is a global module.
	 *
	 * @since 1.0.0
	 * @param array $module Module array.
	 * @return int|false Global module ID or false.
	 */
	public static function is_global_module( $module ) {
		$attributes = $module['attributes'] ?? $module['attrs'] ?? array();
		$global_id  = $attributes['global_module'] ?? '';

		if ( ! empty( $global_id ) && is_numeric( $global_id ) ) {
			return (int) $global_id;
		}

		return false;
	}

	/**
	 * Get global module content.
	 *
	 * @since 1.0.0
	 * @param int $global_module_id Global module post ID.
	 * @return array|false Module content or false if not found.
	 */
	public static function get_global_module_content( $global_module_id ) {
		$post = get_post( $global_module_id );

		if ( ! $post ) {
			return false;
		}

		// Check if it's a Divi layout post type.
		$post_type = get_post_type( $global_module_id );
		if ( 'et_pb_layout' !== $post_type && 'page' !== $post_type && 'post' !== $post_type ) {
			return false;
		}

		// Get the module content from the global module post.
		$content = $post->post_content;

		if ( empty( $content ) ) {
			return false;
		}

		// Parse the content to extract the module structure.
		// Use the builder interface to avoid circular dependency.
		$builder = Respira_Builder_Interface::get_builder( 'Divi' );
		if ( is_wp_error( $builder ) ) {
			return false;
		}

		$modules = $builder->extract_content( $global_module_id );

		return $modules;
	}

	/**
	 * Extract global modules from content.
	 *
	 * @since 1.0.0
	 * @param array $modules Array of modules to check.
	 * @return array Array of global module references found.
	 */
	public static function extract_global_modules( $modules ) {
		$global_modules = array();

		foreach ( $modules as $module ) {
			$global_id = self::is_global_module( $module );
			if ( $global_id ) {
				$global_modules[] = array(
					'global_id' => $global_id,
					'module'    => $module,
				);
			}

			// Check children recursively.
			if ( ! empty( $module['children'] ) ) {
				$child_globals = self::extract_global_modules( $module['children'] );
				$global_modules = array_merge( $global_modules, $child_globals );
			}
		}

		return $global_modules;
	}

	/**
	 * Resolve global module references in content.
	 *
	 * Replaces global_module="123" references with actual module content.
	 *
	 * @since 1.0.0
	 * @param array $modules Array of modules.
	 * @return array Modules with global references resolved.
	 */
	public static function resolve_global_modules( $modules ) {
		$resolved = array();

		foreach ( $modules as $module ) {
			$global_id = self::is_global_module( $module );

			if ( $global_id ) {
				// Get the global module content.
				$global_content = self::get_global_module_content( $global_id );
				if ( $global_content ) {
					// Replace this module with the global module content.
					// Merge attributes (local attributes override global).
					$global_module = $global_content[0] ?? array();
					if ( ! empty( $global_module ) ) {
						$local_attrs = $module['attributes'] ?? $module['attrs'] ?? array();
						$global_attrs = $global_module['attributes'] ?? $global_module['attrs'] ?? array();
						
						// Merge attributes (local overrides global).
						$merged_attrs = array_merge( $global_attrs, $local_attrs );
						
						// Remove global_module attribute.
						unset( $merged_attrs['global_module'] );

						$module['attributes'] = $merged_attrs;
						$module['type']       = $global_module['type'] ?? $module['type'];
						$module['content']    = $global_module['content'] ?? $module['content'];
						$module['children']   = $global_module['children'] ?? $module['children'] ?? array();
					}
				}
			}

			// Resolve children recursively.
			if ( ! empty( $module['children'] ) ) {
				$module['children'] = self::resolve_global_modules( $module['children'] );
			}

			$resolved[] = $module;
		}

		return $resolved;
	}
}

