<?php
/**
 * Usage Statistics Tracker
 *
 * Tracks plugin usage and sends stats to Supabase for admin dashboard.
 * 
 * IMPORTANT: This only tracks general usage metrics (page/post counts, content length).
 * NO personal data, code content, or sensitive information is collected or stored.
 * All tracking is anonymous and aggregated to help improve the product.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes
 */

/**
 * Usage Statistics Tracker class.
 * 
 * Privacy: Tracks only aggregate statistics. No code content, personal data,
 * or sensitive information is collected or transmitted.
 *
 * @since 1.8.3
 */
class Respira_Usage_Tracker {

	/**
	 * Usage API endpoint.
	 *
	 * @var string
	 */
	const USAGE_API_ENDPOINT = 'https://respira.press/api/usage/track';

	/**
	 * Track usage stat.
	 *
	 * Sends usage statistics to Supabase for tracking across all sites.
	 * 
	 * PRIVACY: Only tracks aggregate metrics:
	 * - Action type (update_page, update_post, etc.)
	 * - Resource type (page/post)
	 * - Resource ID (internal WordPress ID)
	 * - Content length estimate (line count, NOT actual content)
	 * 
	 * NO CODE CONTENT, personal data, or sensitive information is collected.
	 *
	 * @since 1.8.3
	 * @param string $action_type      Action type: 'update_page', 'update_post', 'inject_builder_content'.
	 * @param string $resource_type    Resource type: 'page' or 'post'.
	 * @param int    $resource_id      Resource ID.
	 * @param mixed  $content          Content being updated (for calculating lines of code ONLY - content itself is NOT sent).
	 * @return void
	 */
	public static function track( $action_type, $resource_type, $resource_id, $content = null ) {
		// Only track if license is active (PRO features).
		if ( ! Respira_License::is_pro_active() ) {
			return;
		}

		$license_key = get_option( 'respira_license_key' );
		if ( empty( $license_key ) ) {
			return;
		}

		$site_url = get_site_url();

		// Calculate content length (lines of code/content).
		$content_length = 0;
		if ( ! empty( $content ) ) {
			if ( is_string( $content ) ) {
				// Count lines in content.
				$content_length = substr_count( $content, "\n" ) + 1;
				// Also approximate based on content length (average 80 chars per line).
				$content_length = max( $content_length, ceil( strlen( $content ) / 80 ) );
			} elseif ( is_array( $content ) ) {
				// For builder content, estimate from array structure.
				$content_str = wp_json_encode( $content );
				$content_length = ceil( strlen( $content_str ) / 80 );
			}
		}

		// Prepare request data.
		$request_data = array(
			'license_key'   => $license_key,
			'site_url'      => $site_url,
			'action_type'   => $action_type,
			'resource_type' => $resource_type,
			'resource_id'   => $resource_id,
			'content_length' => $content_length,
		);

		/**
		 * PRIVACY: Transmission to Our Servers
		 * 
		 * This is an asynchronous, non-blocking request.
		 * 
		 * TRANSMITTED DATA (reminder):
		 * - License key, site URL, action type, resource type, resource ID
		 * - Approximate line count (NOT actual content)
		 * 
		 * NOT TRANSMITTED:
		 * - Actual page/post content
		 * - Code, scripts, or any sensitive data
		 * - Personal information or user data
		 * 
		 * The request is fire-and-forget to avoid impacting performance.
		 * If the request fails, it's silently ignored (tracking is non-critical).
		 */
		wp_remote_post(
			self::USAGE_API_ENDPOINT,
			array(
				'timeout'  => 5, // Short timeout to avoid blocking
				'blocking' => false, // Non-blocking - fire and forget
				'headers'  => array(
					'Content-Type' => 'application/json',
				),
				'body'     => wp_json_encode( $request_data ), // Only aggregate metrics, NO content
			)
		);

		// Don't wait for response or handle errors - tracking is non-critical.
		// Your content remains safely on your server.
	}
}

