<?php
/**
 * Sentry Error Tracking Integration
 *
 * @package Respira_For_WordPress
 * @since 1.7.7
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handles Sentry error tracking and monitoring.
 */
class Respira_Sentry {
    /**
     * Singleton instance.
     *
     * @var Respira_Sentry|null
     */
    private static $instance = null;

    /**
     * Get singleton instance.
     *
     * @return Respira_Sentry
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Private constructor to enforce singleton pattern.
     */
    private function __construct() {
        $this->init_sentry();
        $this->init_user_feedback();
    }

    /**
     * Initialize Sentry SDK.
     */
    private function init_sentry() {
        // Only initialize if Sentry is available and DSN is configured
        if (!class_exists('\Sentry\init')) {
            return;
        }

        // Hardcoded Sentry DSN for plugin error tracking
        // All plugin installations report to this Sentry project
        $dsn = 'https://bf3ba2ee593deb0360e31b1066a306fd@o4510381871333376.ingest.de.sentry.io/4510386267553872';

        if (empty($dsn)) {
            return;
        }

        \Sentry\init([
            'dsn' => $dsn,
            'environment' => $this->get_environment(),
            'release' => $this->get_release_version(),
            'traces_sample_rate' => 0.2, // 20% performance monitoring
            'profiles_sample_rate' => 0.2,
            'send_default_pii' => false, // GDPR compliance
            'before_send' => [$this, 'filter_error'],
        ]);

        // Add WordPress context
        \Sentry\configureScope(function (\Sentry\State\Scope $scope): void {
            $scope->setContext('wordpress', [
                'version' => get_bloginfo('version'),
                'php_version' => PHP_VERSION,
                'plugin_version' => RESPIRA_VERSION,
                'active_theme' => wp_get_theme()->get('Name'),
                'is_multisite' => is_multisite(),
            ]);
        });
    }

    /**
     * Get current environment.
     *
     * @return string
     */
    private function get_environment() {
        if (defined('WP_ENVIRONMENT_TYPE')) {
            return WP_ENVIRONMENT_TYPE;
        }
        return defined('WP_DEBUG') && WP_DEBUG ? 'development' : 'production';
    }

    /**
     * Get release version string.
     *
     * @return string|null
     */
    private function get_release_version() {
        return defined('RESPIRA_VERSION') ? 'respira-wordpress@' . RESPIRA_VERSION : null;
    }

    /**
     * Filter errors before sending to Sentry.
     *
     * @param \Sentry\Event $event The Sentry event.
     * @return \Sentry\Event|null The filtered event or null to skip.
     */
    public function filter_error(\Sentry\Event $event): ?\Sentry\Event {
        // Don't send errors from local development
        if ($this->get_environment() === 'local') {
            return null;
        }

        // Filter out common WordPress notices that aren't actionable
        $message = $event->getMessage();
        $ignored_patterns = [
            'Undefined array key',
            'Trying to access array offset',
        ];

        foreach ($ignored_patterns as $pattern) {
            if (strpos($message, $pattern) !== false) {
                return null;
            }
        }

        return $event;
    }

    /**
     * Capture exception with additional context.
     *
     * @param \Throwable $exception The exception to capture.
     * @param array $context Additional context data.
     */
    public static function capture_exception($exception, $context = []) {
        if (!class_exists('\Sentry\captureException')) {
            error_log('Respira Error: ' . $exception->getMessage());
            return;
        }

        \Sentry\withScope(function (\Sentry\State\Scope $scope) use ($exception, $context): void {
            if (!empty($context)) {
                $scope->setContext('additional', $context);
            }
            \Sentry\captureException($exception);
        });
    }

    /**
     * Capture message with level.
     *
     * @param string $message The message to capture.
     * @param string $level The severity level (debug, info, warning, error, fatal).
     * @param array $context Additional context data.
     */
    public static function capture_message($message, $level = 'info', $context = []) {
        if (!class_exists('\Sentry\captureMessage')) {
            error_log("Respira {$level}: {$message}");
            return;
        }

        \Sentry\withScope(function (\Sentry\State\Scope $scope) use ($message, $level, $context): void {
            if (!empty($context)) {
                $scope->setContext('additional', $context);
            }
            $severity = self::map_severity($level);
            \Sentry\captureMessage($message, $severity);
        });
    }

    /**
     * Map severity level string to Sentry severity constant.
     *
     * @param string $level The severity level string.
     * @return \Sentry\Severity
     */
    private static function map_severity($level) {
        $map = [
            'debug' => \Sentry\Severity::debug(),
            'info' => \Sentry\Severity::info(),
            'warning' => \Sentry\Severity::warning(),
            'error' => \Sentry\Severity::error(),
            'fatal' => \Sentry\Severity::fatal(),
        ];
        return $map[$level] ?? \Sentry\Severity::info();
    }

    /**
     * Initialize Sentry User Feedback widget in WordPress admin.
     */
    private function init_user_feedback() {
        // Hardcoded Sentry DSN for plugin error tracking
        $dsn = 'https://bf3ba2ee593deb0360e31b1066a306fd@o4510381871333376.ingest.de.sentry.io/4510386267553872';

        if (empty($dsn) || !is_admin()) {
            return;
        }

        // Hook into admin_enqueue_scripts to add Sentry feedback widget
        add_action('admin_enqueue_scripts', [$this, 'enqueue_user_feedback_script']);
    }

    /**
     * Enqueue Sentry User Feedback script in WordPress admin.
     */
    public function enqueue_user_feedback_script() {
        // Only load on Respira admin pages
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'respira') === false) {
            return;
        }

        // Hardcoded Sentry DSN for plugin error tracking
        $dsn = 'https://bf3ba2ee593deb0360e31b1066a306fd@o4510381871333376.ingest.de.sentry.io/4510386267553872';
        if (empty($dsn)) {
            return;
        }

        // Extract the public key from DSN for the loader script URL
        // DSN format: https://PUBLIC_KEY@o4510381871333376.ingest.de.sentry.io/PROJECT_ID
        preg_match('/https:\/\/([a-f0-9]+)@.*\.sentry\.io\/([0-9]+)/', $dsn, $matches);

        if (empty($matches) || count($matches) < 3) {
            return;
        }

        $public_key = $matches[1];
        $project_id = $matches[2];

        // Get the correct Sentry CDN domain from DSN
        preg_match('/https:\/\/[a-f0-9]+@(.*?)\/[0-9]+/', $dsn, $domain_matches);
        $sentry_domain = !empty($domain_matches[1]) ? $domain_matches[1] : 'o4510381871333376.ingest.de.sentry.io';

        // Determine the correct CDN subdomain (js-de for Germany region, js for others)
        $cdn_subdomain = (strpos($sentry_domain, '.de.sentry') !== false) ? 'js-de' : 'js';

        // Enqueue Sentry Loader Script
        wp_enqueue_script(
            'sentry-loader',
            "https://{$cdn_subdomain}.sentry-cdn.com/{$public_key}.min.js",
            array(),
            null,
            true
        );

        // Get current user info
        $current_user = wp_get_current_user();
        $user_email = $current_user->user_email;
        $user_name = $current_user->display_name;

        // Add inline script to initialize Sentry with User Feedback
        $inline_script = "
        window.sentryOnLoad = function() {
            Sentry.init({
                dsn: '" . esc_js($dsn) . "',
                environment: '" . esc_js($this->get_environment()) . "',
                release: '" . esc_js($this->get_release_version()) . "',
            });

            // Set user context
            Sentry.setUser({
                email: '" . esc_js($user_email) . "',
                username: '" . esc_js($user_name) . "',
            });

            // Load and initialize User Feedback integration
            Sentry.lazyLoadIntegration('feedbackIntegration')
                .then(function(feedbackIntegration) {
                    Sentry.addIntegration(feedbackIntegration({
                        colorScheme: 'system',
                        showBranding: false,
                        // User email and name will be pre-filled since we set them via setUser()
                    }));
                })
                .catch(function() {
                    // Silently fail if feedback integration can't be loaded
                    console.log('Sentry User Feedback not available');
                });
        };
        ";

        wp_add_inline_script('sentry-loader', $inline_script, 'before');
    }
}
