<?php
/**
 * Page/post duplication functionality.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes
 */

/**
 * Page/post duplication functionality.
 *
 * Handles the duplication of pages and posts, including all meta data
 * and page builder content.
 *
 * @since 1.0.0
 */
class Respira_Duplicator {

	/**
	 * Duplicate a post or page.
	 *
	 * @since 1.0.0
	 * @param int    $post_id The ID of the post to duplicate.
	 * @param string $suffix  Optional. Suffix to add to the duplicated post title.
	 * @return int|WP_Error The ID of the duplicated post or WP_Error on failure.
	 */
	public static function duplicate_post( $post_id, $suffix = null ) {
		$post = get_post( $post_id );

		if ( ! $post ) {
			return new WP_Error(
				'respira_post_not_found',
				__( 'Original post not found.', 'respira-for-wordpress' )
			);
		}

		// Generate suffix if not provided.
		if ( null === $suffix ) {
			$suffix = '_duplicate_' . time();
		}

		// Prepare duplicate post data.
		$duplicate_post = array(
			'post_title'     => $post->post_title . $suffix,
			'post_content'   => $post->post_content,
			'post_excerpt'   => $post->post_excerpt,
			'post_status'    => 'draft', // Always create as draft for safety.
			'post_type'      => $post->post_type,
			'post_author'    => get_current_user_id() ?? $post->post_author,
			'post_parent'    => $post->post_parent,
			'menu_order'     => $post->menu_order,
			'comment_status' => $post->comment_status,
			'ping_status'    => $post->ping_status,
		);

		// Insert the duplicate post.
		$duplicate_id = wp_insert_post( $duplicate_post, true );

		if ( is_wp_error( $duplicate_id ) ) {
			return $duplicate_id;
		}

		// Duplicate all post meta.
		self::duplicate_post_meta( $post_id, $duplicate_id );

		// Duplicate taxonomies (categories, tags, etc.).
		self::duplicate_taxonomies( $post_id, $duplicate_id );

		// Mark as Respira duplicate.
		update_post_meta( $duplicate_id, '_respira_duplicate', true );
		update_post_meta( $duplicate_id, '_respira_original_id', $post_id );
		update_post_meta( $duplicate_id, '_respira_duplicated_at', current_time( 'mysql' ) );

		/**
		 * Fires after a post has been duplicated.
		 *
		 * @since 1.0.0
		 * @param int $duplicate_id The ID of the duplicated post.
		 * @param int $post_id      The ID of the original post.
		 */
		do_action( 'respira_after_duplicate_post', $duplicate_id, $post_id );

		return $duplicate_id;
	}

	/**
	 * Duplicate post meta.
	 *
	 * @since 1.0.0
	 * @param int $from_post_id Source post ID.
	 * @param int $to_post_id   Destination post ID.
	 */
	private static function duplicate_post_meta( $from_post_id, $to_post_id ) {
		$post_meta = get_post_meta( $from_post_id );

		foreach ( $post_meta as $meta_key => $meta_values ) {
			// Skip certain meta keys.
			$skip_keys = array(
				'_edit_lock',
				'_edit_last',
				'_wp_old_slug',
				'_wp_old_date',
			);

			if ( in_array( $meta_key, $skip_keys, true ) ) {
				continue;
			}

			foreach ( $meta_values as $meta_value ) {
				// Maybe unserialize.
				$meta_value = maybe_unserialize( $meta_value );

				// Add meta to duplicate post.
				add_post_meta( $to_post_id, $meta_key, $meta_value );
			}
		}
	}

	/**
	 * Duplicate taxonomies.
	 *
	 * @since 1.0.0
	 * @param int $from_post_id Source post ID.
	 * @param int $to_post_id   Destination post ID.
	 */
	private static function duplicate_taxonomies( $from_post_id, $to_post_id ) {
		$post_type = get_post_type( $from_post_id );
		$taxonomies = get_object_taxonomies( $post_type );

		foreach ( $taxonomies as $taxonomy ) {
			$terms = wp_get_object_terms(
				$from_post_id,
				$taxonomy,
				array( 'fields' => 'ids' )
			);

			if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
				wp_set_object_terms( $to_post_id, $terms, $taxonomy );
			}
		}
	}

	/**
	 * Check if a post is a Respira duplicate.
	 *
	 * @since 1.0.0
	 * @param int $post_id The post ID to check.
	 * @return bool True if is a Respira duplicate, false otherwise.
	 */
	public static function is_duplicate( $post_id ) {
		return (bool) get_post_meta( $post_id, '_respira_duplicate', true );
	}

	/**
	 * Get the original post ID from a duplicate.
	 *
	 * @since 1.0.0
	 * @param int $duplicate_id The duplicate post ID.
	 * @return int|null The original post ID or null if not a duplicate.
	 */
	public static function get_original_id( $duplicate_id ) {
		$original_id = get_post_meta( $duplicate_id, '_respira_original_id', true );

		return $original_id ? (int) $original_id : null;
	}

	/**
	 * Get all duplicates of a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id The original post ID.
	 * @return array Array of duplicate post objects.
	 */
	public static function get_duplicates( $post_id ) {
		$args = array(
			'post_type'      => get_post_type( $post_id ),
			'post_status'    => 'any',
			'posts_per_page' => -1,
			'meta_query'     => array(
				array(
					'key'   => '_respira_original_id',
					'value' => $post_id,
					'type'  => 'NUMERIC',
				),
			),
		);

		$query = new WP_Query( $args );

		return $query->posts;
	}

	/**
	 * Delete all duplicates of a post.
	 *
	 * @since 1.0.0
	 * @param int $post_id The original post ID.
	 * @return int Number of duplicates deleted.
	 */
	public static function delete_all_duplicates( $post_id ) {
		$duplicates = self::get_duplicates( $post_id );
		$count      = 0;

		foreach ( $duplicates as $duplicate ) {
			if ( wp_delete_post( $duplicate->ID, true ) ) {
				$count++;
			}
		}

		return $count;
	}
}
