<?php
/**
 * Authentication and authorization functionality.
 *
 * @package    Respira_For_WordPress
 * @subpackage Respira_For_WordPress/includes
 */

/**
 * Authentication and authorization functionality.
 *
 * Handles API key generation, validation, and rate limiting.
 *
 * @since 1.0.0
 */
class Respira_Auth {

	/**
	 * Generate a new API key.
	 *
	 * @since 1.0.0
	 * @param int    $user_id  The user ID who owns this key.
	 * @param string $name     Optional. A friendly name for the key.
	 * @param array  $permissions Optional. Array of permissions for this key.
	 * @return string|WP_Error The generated API key or WP_Error on failure.
	 */
	public static function generate_api_key( $user_id, $name = null, $permissions = array() ) {
		global $wpdb;

		// Verify user has required capability.
		$user = get_user_by( 'id', $user_id );
		if ( ! $user || ! user_can( $user, 'manage_options' ) ) {
			return new WP_Error(
				'respira_insufficient_permissions',
				__( 'User does not have permission to generate API keys.', 'respira-for-wordpress' )
			);
		}

		// Check if license is active (required for API key generation).
		$license_key = get_option( 'respira_license_key' );
		$license_status = get_option( 'respira_license_status' );

		if ( empty( $license_key ) || ( 'active' !== $license_status && 'trial' !== $license_status ) ) {
			return new WP_Error(
				'respira_license_required',
				__( 'An active license is required to generate API keys. Please activate your license first.', 'respira-for-wordpress' ),
				array( 'status' => 403 )
			);
		}

		// Validate license matches current domain
		if ( ! Respira_License::is_pro_active() ) {
			return new WP_Error(
				'respira_license_invalid',
				__( 'License validation failed. The license may not be valid for this domain. Please check your license in the License page.', 'respira-for-wordpress' ),
				array( 'status' => 403 )
			);
		}

		// Generate UUID v4 format key.
		$api_key = 'respira_' . self::generate_uuid();

		// Hash the key for storage.
		$hashed_key = wp_hash_password( $api_key );

		// Default permissions.
		if ( empty( $permissions ) ) {
			$permissions = array(
				'read_pages',
				'write_pages',
				'read_posts',
				'write_posts',
				'read_context',
				'upload_media',
			);
		}

		// Insert into database.
		$table_name = $wpdb->prefix . 'respira_api_keys';
		
		// Check if table exists
		$table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) );
		if ( ! $table_exists ) {
			// Table doesn't exist - try to create it
			require_once ABSPATH . 'wp-admin/includes/upgrade.php';
			$charset_collate = $wpdb->get_charset_collate();
			$sql = "CREATE TABLE IF NOT EXISTS $table_name (
				id bigint(20) NOT NULL AUTO_INCREMENT,
				api_key varchar(255) NOT NULL,
				user_id bigint(20) NOT NULL,
				name varchar(255) DEFAULT NULL,
				permissions text DEFAULT NULL,
				license_key varchar(255) DEFAULT NULL,
				last_used datetime DEFAULT NULL,
				created_at datetime NOT NULL,
				expires_at datetime DEFAULT NULL,
				is_active tinyint(1) DEFAULT 1,
				PRIMARY KEY  (id),
				UNIQUE KEY api_key (api_key),
				KEY user_id (user_id),
				KEY license_key (license_key)
			) $charset_collate;";
			dbDelta( $sql );
		}
		
		$result = $wpdb->insert(
			$table_name,
			array(
				'api_key'     => $hashed_key,
				'user_id'     => $user_id,
				'name'        => $name,
				'permissions' => wp_json_encode( $permissions ),
				'license_key' => $license_key,
				'created_at'  => current_time( 'mysql' ),
				'is_active'   => 1,
			),
			array( '%s', '%d', '%s', '%s', '%s', '%s', '%d' )
		);

		if ( false === $result ) {
			$error_message = __( 'Failed to create API key in database.', 'respira-for-wordpress' );
			if ( ! empty( $wpdb->last_error ) ) {
				$error_message .= ' ' . sprintf(
					/* translators: %s: database error message */
					__( 'Database error: %s', 'respira-for-wordpress' ),
					$wpdb->last_error
				);
			}
			return new WP_Error(
				'respira_db_error',
				$error_message
			);
		}

		// Log the key generation.
		self::log_action( $wpdb->insert_id, $user_id, 'generate_api_key', 'api_key', $wpdb->insert_id );

		// Return the plain text key (only time it will be visible).
		return $api_key;
	}

	/**
	 * Validate an API key.
	 *
	 * @since 1.0.0
	 * @param string $api_key The API key to validate.
	 * @return array|WP_Error Array with key info on success, WP_Error on failure.
	 */
	public static function validate_api_key( $api_key ) {
		global $wpdb;

		if ( empty( $api_key ) || ! is_string( $api_key ) ) {
			return new WP_Error(
				'respira_invalid_key_format',
				__( 'Invalid API key format.', 'respira-for-wordpress' ),
				array( 'status' => 401 )
			);
		}

		// Remove 'respira_' prefix if present for validation.
		if ( 0 !== strpos( $api_key, 'respira_' ) ) {
			return new WP_Error(
				'respira_invalid_key_prefix',
				__( 'API key must start with "respira_" prefix.', 'respira-for-wordpress' ),
				array( 'status' => 401 )
			);
		}

		// Get all active API keys.
		$table_name = $wpdb->prefix . 'respira_api_keys';
		$keys       = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT * FROM $table_name WHERE is_active = %d",
				1
			)
		);

		if ( empty( $keys ) ) {
			return new WP_Error(
				'respira_no_active_keys',
				__( 'No active API keys found.', 'respira-for-wordpress' ),
				array( 'status' => 401 )
			);
		}

		// Check each key using wp_check_password.
		foreach ( $keys as $key_data ) {
			if ( wp_check_password( $api_key, $key_data->api_key ) ) {
				// Check if key has expired.
				if ( ! empty( $key_data->expires_at ) ) {
					$expires = strtotime( $key_data->expires_at );
					if ( $expires < time() ) {
						return new WP_Error(
							'respira_key_expired',
							__( 'API key has expired.', 'respira-for-wordpress' ),
							array( 'status' => 401 )
						);
					}
				}

				// Check if license is still active (if key is linked to a license).
				if ( ! empty( $key_data->license_key ) ) {
					$current_license_key = get_option( 'respira_license_key' );
					$license_status = get_option( 'respira_license_status' );

					// If license key changed or license is inactive, disable this API key.
					if ( $key_data->license_key !== $current_license_key || 'active' !== $license_status ) {
						// Disable the API key.
						$wpdb->update(
							$table_name,
							array( 'is_active' => 0 ),
							array( 'id' => $key_data->id ),
							array( '%d' ),
							array( '%d' )
						);

						return new WP_Error(
							'respira_license_inactive',
							__( 'API key is disabled because the license is inactive or has been changed. Please reactivate your license.', 'respira-for-wordpress' ),
							array( 'status' => 403 )
						);
					}

					// Re-validate license periodically (every 7 days).
					$last_check = get_option( 'respira_license_last_check', 0 );
					if ( ( time() - $last_check ) > ( 7 * DAY_IN_SECONDS ) ) {
						Respira_License::validate_license( $key_data->license_key );
						$license_status = get_option( 'respira_license_status' );
						if ( 'active' !== $license_status ) {
							// Disable the API key.
							$wpdb->update(
								$table_name,
								array( 'is_active' => 0 ),
								array( 'id' => $key_data->id ),
								array( '%d' ),
								array( '%d' )
							);

							return new WP_Error(
								'respira_license_expired',
								__( 'API key is disabled because the license has expired. Please renew your license.', 'respira-for-wordpress' ),
								array( 'status' => 403 )
							);
						}
					}
				}

				// Check rate limiting.
				$rate_limit_check = self::check_rate_limit( $key_data->id );
				if ( is_wp_error( $rate_limit_check ) ) {
					return $rate_limit_check;
				}

				// Update last used timestamp.
				$wpdb->update(
					$table_name,
					array( 'last_used' => current_time( 'mysql' ) ),
					array( 'id' => $key_data->id ),
					array( '%s' ),
					array( '%d' )
				);

				// Return key data.
				return array(
					'id'          => $key_data->id,
					'user_id'     => $key_data->user_id,
					'name'        => $key_data->name,
					'permissions' => json_decode( $key_data->permissions, true ),
					'license_key' => $key_data->license_key ?? null,
				);
			}
		}

		return new WP_Error(
			'respira_invalid_key',
			__( 'Invalid API key.', 'respira-for-wordpress' ),
			array( 'status' => 401 )
		);
	}

	/**
	 * Revoke an API key.
	 *
	 * @since 1.0.0
	 * @param int $key_id The ID of the key to revoke.
	 * @return bool|WP_Error True on success, WP_Error on failure.
	 */
	public static function revoke_api_key( $key_id ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'respira_api_keys';
		$result     = $wpdb->update(
			$table_name,
			array( 'is_active' => 0 ),
			array( 'id' => $key_id ),
			array( '%d' ),
			array( '%d' )
		);

		if ( false === $result ) {
			return new WP_Error(
				'respira_revoke_failed',
				__( 'Failed to revoke API key.', 'respira-for-wordpress' )
			);
		}

		// Log the revocation.
		self::log_action( $key_id, get_current_user_id(), 'revoke_api_key', 'api_key', $key_id );

		return true;
	}

	/**
	 * Check if API key has permission for an action.
	 *
	 * @since 1.0.0
	 * @param array  $key_data   The key data from validate_api_key().
	 * @param string $permission The permission to check.
	 * @return bool True if has permission, false otherwise.
	 */
	public static function has_permission( $key_data, $permission ) {
		if ( ! isset( $key_data['permissions'] ) || ! is_array( $key_data['permissions'] ) ) {
			return false;
		}

		return in_array( $permission, $key_data['permissions'], true );
	}

	/**
	 * Check rate limiting for an API key.
	 *
	 * @since 1.0.0
	 * @param int $key_id The API key ID.
	 * @return bool|WP_Error True if within limits, WP_Error if exceeded.
	 */
	private static function check_rate_limit( $key_id ) {
		global $wpdb;

		$rate_limit = get_option( 'respira_rate_limit', 100 );
		$table_name = $wpdb->prefix . 'respira_audit_log';

		// Count requests in the last hour.
		$count = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM $table_name
				WHERE api_key_id = %d
				AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)",
				$key_id
			)
		);

		if ( $count >= $rate_limit ) {
			return new WP_Error(
				'respira_rate_limit_exceeded',
				sprintf(
					/* translators: %d: rate limit */
					__( 'Rate limit exceeded. Maximum %d requests per hour.', 'respira-for-wordpress' ),
					$rate_limit
				),
				array( 'status' => 429 )
			);
		}

		return true;
	}

	/**
	 * Log an API action to the audit log.
	 *
	 * @since 1.0.0
	 * @param int    $api_key_id    The API key ID.
	 * @param int    $user_id       The user ID.
	 * @param string $action        The action performed.
	 * @param string $resource_type The type of resource.
	 * @param int    $resource_id   The resource ID.
	 * @param array  $request_data  Optional. Request data.
	 * @param int    $response_code Optional. HTTP response code.
	 */
	public static function log_action( $api_key_id, $user_id, $action, $resource_type = null, $resource_id = null, $request_data = array(), $response_code = 200 ) {
		global $wpdb;

		// Check if audit logging is enabled.
		if ( ! get_option( 'respira_audit_logging', 1 ) ) {
			return;
		}

		$table_name = $wpdb->prefix . 'respira_audit_log';

		$wpdb->insert(
			$table_name,
			array(
				'api_key_id'    => $api_key_id,
				'user_id'       => $user_id,
				'action'        => $action,
				'resource_type' => $resource_type,
				'resource_id'   => $resource_id,
				'ip_address'    => self::get_client_ip(),
				'user_agent'    => isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '',
				'request_data'  => ! empty( $request_data ) ? wp_json_encode( $request_data ) : null,
				'response_code' => $response_code,
				'created_at'    => current_time( 'mysql' ),
			),
			array( '%d', '%d', '%s', '%s', '%d', '%s', '%s', '%s', '%d', '%s' )
		);
	}

	/**
	 * Generate a UUID v4.
	 *
	 * @since  1.0.0
	 * @return string UUID v4 string.
	 */
	private static function generate_uuid() {
		$data    = random_bytes( 16 );
		$data[6] = chr( ord( $data[6] ) & 0x0f | 0x40 );
		$data[8] = chr( ord( $data[8] ) & 0x3f | 0x80 );
		return vsprintf( '%s%s-%s-%s-%s-%s%s%s', str_split( bin2hex( $data ), 4 ) );
	}

	/**
	 * Get client IP address.
	 *
	 * @since  1.0.0
	 * @return string IP address.
	 */
	private static function get_client_ip() {
		$ip_keys = array(
			'HTTP_CLIENT_IP',
			'HTTP_X_FORWARDED_FOR',
			'HTTP_X_FORWARDED',
			'HTTP_X_CLUSTER_CLIENT_IP',
			'HTTP_FORWARDED_FOR',
			'HTTP_FORWARDED',
			'REMOTE_ADDR',
		);

		foreach ( $ip_keys as $key ) {
			if ( isset( $_SERVER[ $key ] ) && filter_var( wp_unslash( $_SERVER[ $key ] ), FILTER_VALIDATE_IP ) ) {
				return sanitize_text_field( wp_unslash( $_SERVER[ $key ] ) );
			}
		}

		return '0.0.0.0';
	}

	/**
	 * Get all API keys for a user.
	 *
	 * @since 1.0.0
	 * @param int $user_id The user ID.
	 * @return array Array of API key data (without the actual keys).
	 */
	public static function get_user_keys( $user_id ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'respira_api_keys';
		
		// Check if table exists
		$table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) );
		if ( ! $table_exists ) {
			// Table doesn't exist - return empty array
			return array();
		}
		
		$keys = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT id, name, permissions, last_used, created_at, expires_at, is_active
				FROM $table_name
				WHERE user_id = %d
				ORDER BY created_at DESC",
				$user_id
			)
		);

		// Return empty array if query failed
		if ( false === $keys ) {
			return array();
		}

		return $keys;
	}
}
